<?php
declare(strict_types = 1);

namespace Formal\ORM;

use Formal\ORM\{
    Definition\Aggregates,
    Definition\Types,
    Adapter\Transaction\Failure,
};
use Formal\AccessLayer\Connection;
use Innmind\Filesystem\Adapter as Storage;
use Innmind\Immutable\Either;

final class Manager
{
    private Adapter $adapter;
    private Aggregates $aggregates;
    private Repository\Active $repositories;
    private bool $inTransaction;
    private Repository\Context $context;

    private function __construct(Adapter $adapter, Aggregates $aggregates)
    {
        $this->adapter = $adapter;
        $this->aggregates = $aggregates;
        $this->repositories = Repository\Active::new();
        $this->inTransaction = false;
        $this->context = new Repository\Context;
    }

    public static function of(
        Adapter $adapter,
        ?Aggregates $aggregates = null,
    ): self {
        return new self($adapter, $aggregates ?? Aggregates::of(Types::default()));
    }

    public static function sql(
        Connection $connection,
        ?Aggregates $aggregates = null,
    ): self {
        return self::of(Adapter\SQL::of($connection), $aggregates);
    }

    public static function filesystem(
        Storage $storage,
        ?Aggregates $aggregates = null,
    ): self {
        return self::of(Adapter\Filesystem::of($storage), $aggregates);
    }

    /**
     * @template T of object
     *
     * @param class-string<T> $class
     *
     * @return Repository<T>
     */
    public function repository(string $class): Repository
    {
        return $this
            ->repositories
            ->get($class)
            ->match(
                static fn($repository) => $repository,
                function() use ($class) {
                    $definition = $this->aggregates->get($class);

                    $repository = Repository::of(
                        $this->repositories,
                        $this->adapter->repository($definition),
                        $definition,
                        fn() => $this->inTransaction,
                        $this->context,
                    );
                    $this->repositories->register($class, $repository);

                    return $repository;
                },
            );
    }

    /**
     * @template E
     * @template R
     *
     * @param callable(): Either<E, R> $transaction
     *
     * @return Either<E|Failure, R>
     */
    public function transactional(callable $transaction): Either
    {
        if ($this->inTransaction) {
            throw new \LogicException('Nested transactions not allowed');
        }

        $this->inTransaction = true;
        $transactionAdapter = $this->adapter->transaction();

        try {
            // We force unwrapping the Either monad to prevent leaving this
            // method with a deferred Either meaning the system would have an
            // opened transaction hanging around
            return $transactionAdapter
                ->start()
                ->either()
                ->leftMap(Failure::of(...))
                ->flatMap(static fn() => $transaction())
                ->memoize()
                ->flatMap(
                    static fn($value) => $transactionAdapter
                        ->commit($value)
                        ->either()
                        ->leftMap(Failure::of(...)),
                )
                ->leftMap(
                    static fn($value) => $transactionAdapter
                        ->rollback($value)
                        ->match(
                            static fn($value) => $value,
                            Failure::of(...),
                        ),
                );
        } catch (\Throwable $e) {
            throw $transactionAdapter->rollback($e)->unwrap();
        } finally {
            $this->inTransaction = false;
        }
    }
}
