#!/usr/bin/env python
# Copyright (c) 2008 David Aguilar
"""This module provides an inotify plugin for Linux and other systems
which provide the pyinotify module.

"""

import os
import time
from PyQt4.QtCore import QCoreApplication
from PyQt4.QtCore import QThread
from PyQt4.QtCore import QEvent
from PyQt4.QtCore import SIGNAL
from pyinotify import ProcessEvent
from pyinotify import WatchManager, Notifier, EventsCodes

INOTIFY_EVENT_TYPE = QEvent.User + 0

class FileSysEvent(ProcessEvent):
    """FileSysEvents are generated by the GitNotifier thread
    and represent the inotify events.
    """
    def __init__(self, parent):
        """FileSysEvent keeps a reference to the Qt parent and remembers
        the last event time to avoid event floods.
        """
        ProcessEvent.__init__(self)
        self.parent = parent
        self.last_event_time = time.time()
    def process_default(self, event):
        """Notifies the Qt parent when actions occur."""
        # Prevent notificaiton floods
        if time.time() - self.last_event_time > 1.0:
            self.parent.notify()
        self.last_event_time = time.time()

class GitNotifier(QThread):
    """This polls inotify for changes and generates FileSysEvents
    in response to updates.
    """
    def __init__(self, receiver, git):
        QThread.__init__(self)
        self.receiver = receiver #: The Qt receiver of the event
        self.git = git # A: git command object
        self.path = git.get_work_tree() #: Path to monitor
        self.abort = False #: Whether to abort (using during destruction)
        self.dirs_seen = {} #: Directories we're watching
        #: Events we capture
        self.mask = (EventsCodes.IN_CREATE | EventsCodes.IN_DELETE |
                     EventsCodes.IN_MODIFY | EventsCodes.IN_MOVED_TO)

    def notify(self):
        """Posts a Qt event in response to inotify updates.
        """
        if not self.abort:
            event_type = QEvent.Type(INOTIFY_EVENT_TYPE)
            event = QEvent(event_type)
            QCoreApplication.postEvent(self.receiver, event)

    def watch_directory(self, directory):
        """Sets up a directory for monitoring by inotify.
        """
        directory = os.path.realpath(directory)
        if directory not in self.dirs_seen:
            self.wm.add_watch(directory, self.mask)
            self.dirs_seen[directory] = True

    def run(self):
        """This creates the inotify WatchManager and generates
        FileSysEvents."""
        # Only capture those events that git cares about
        self.wm = WatchManager()
        notifier = Notifier(self.wm, FileSysEvent(self))
        dirs_seen = {}
        added_flag = False
        # self.abort signals app termination.  The timeout is a tradeoff
        # between fast notification response and waiting too long to exit.
        while not self.abort:
            if not added_flag:
                self.watch_directory(self.path)
                # Register files/directories known to git
                for file in self.git.ls_files().splitlines():
                    directory = os.path.dirname(file)
                    self.watch_directory(directory)
                added_flag = True
            notifier.process_events()
            if notifier.check_events(timeout=250):
                notifier.read_events()
        notifier.stop()
