/*
 * winston.js: Top-level include defining Winston.
 *
 * (C) 2010 Charlie Robbins
 * MIT LICENCE
 *
 */

var winston = exports;

//
// Expose version using `pkginfo`
//
require('pkginfo')(module, 'version');

//
// Include transports defined by default by winston
//
winston.transports = require('./winston/transports');

//
// Expose utility methods 
//
var common             = require('./winston/common');
winston.hash           = common.hash;
winston.clone          = common.clone;
winston.longestElement = common.longestElement;
winston.exception      = require('./winston/exception');
winston.config         = require('./winston/config');
winston.addColors      = winston.config.addColors; 

//
// Expose core Logging-related prototypes.
//
winston.Container      = require('./winston/container').Container;
winston.Logger         = require('./winston/logger').Logger;
winston.Transport      = require('./winston/transports/transport').Transport;

//
// We create and expose a default `Container` to `winston.loggers` so that the 
// programmer may manage multiple `winston.Logger` instances without any additional overhead.
//
// ### some-file1.js
//
//     var logger = require('winston').loggers.get('something');
//
// ### some-file2.js
//
//     var logger = require('winston').loggers.get('something');
//
winston.loggers = new winston.Container();

//
// We create and expose a 'defaultLogger' so that the programmer may do the
// following without the need to create an instance of winston.Logger directly:
//
//     var winston = require('winston');
//     winston.log('info', 'some message');
//     winston.error('some error'); 
//
var defaultLogger = new winston.Logger({ 
  transports: [new winston.transports.Console()] 
});

//
// Pass through the target methods onto `winston.
//
var methods = [
  'log', 
  'add', 
  'remove', 
  'profile', 
  'extend', 
  'cli', 
  'handleExceptions', 
  'unhandleExceptions'
];
common.setLevels(winston, null, defaultLogger.levels);
methods.forEach(function (method) {
  winston[method] = function () {
    return defaultLogger[method].apply(defaultLogger, arguments);
  };
});

//
// ### function cli ()
// Configures the default winston logger to have the
// settings for command-line interfaces: no timestamp,
// colors enabled, padded output, and additional levels.
//
winston.cli = function () {
  winston.padLevels = true;
  common.setLevels(winston, defaultLogger.levels, winston.config.cli.levels);
  defaultLogger.setLevels(winston.config.cli.levels);
  winston.config.addColors(winston.config.cli.colors);
  
  if (defaultLogger.transports.console) {
    defaultLogger.transports.console.colorize = true;
    defaultLogger.transports.console.timestamp = false;
  }
  
  return winston;
};

//
// ### function setLevels (target)
// #### @target {Object} Target levels to use
// Sets the `target` levels specified on the default winston logger.
//
winston.setLevels = function (target) {
  common.setLevels(winston, defaultLogger.levels, target);
  defaultLogger.setLevels(target);
};

//
// Define getters / setters for appropriate properties of the 
// default logger which need to be exposed by winston.
//
['emitErrs', 'exitOnError', 'padLevels', 'level', 'levelLength', 'stripColors'].forEach(function (prop) {
  Object.defineProperty(winston, prop, {
    get: function () {
      return defaultLogger[prop];
    },
    set: function (val) {
      defaultLogger[prop] = val;
    }
  });
});

//
// @default {Object} 
// The default transports and exceptionHandlers for 
// the default winston logger.
//
Object.defineProperty(winston, 'default', {
  get: function () {
    return {
      transports: defaultLogger.transports,
      exceptionHandlers: defaultLogger.exceptionHandlers
    };
  }
});