import { ProcedureType } from '@trpc/server';
import { TRPCResponse } from '@trpc/server/rpc';
import { LinkRuntimeOptions, PromiseAndCancel } from '../links/core';

// https://github.com/trpc/trpc/pull/669
function arrayToDict(array: unknown[]) {
  const dict: Record<number, unknown> = {};
  for (let index = 0; index < array.length; index++) {
    const element = array[index];
    dict[index] = element;
  }
  return dict;
}

export function httpRequest<TResponseShape = TRPCResponse>(
  props: {
    runtime: LinkRuntimeOptions;
    type: ProcedureType;
    path: string;
    url: string;
  } & ({ inputs: unknown[] } | { input: unknown }),
): PromiseAndCancel<TResponseShape> {
  const { type, runtime: rt, path } = props;
  const ac = rt.AbortController ? new rt.AbortController() : null;
  const method = {
    query: 'GET',
    mutation: 'POST',
    subscription: 'PATCH',
  };
  const input = 'input' in props ? props.input : arrayToDict(props.inputs);
  function getUrl() {
    let url = props.url + '/' + path;
    const queryParts: string[] = [];
    if ('inputs' in props) {
      queryParts.push('batch=1');
    }
    if (type === 'query' && input !== undefined) {
      queryParts.push(
        `input=${encodeURIComponent(
          JSON.stringify(rt.transformer.serialize(input)),
        )}`,
      );
    }
    if (queryParts.length) {
      url += '?' + queryParts.join('&');
    }
    return url;
  }
  function getBody() {
    if (type === 'query') {
      return undefined;
    }
    const rawInput = rt.transformer.serialize(input);
    return rawInput !== undefined ? JSON.stringify(rawInput) : undefined;
  }

  const promise = new Promise<TResponseShape>((resolve, reject) => {
    const url = getUrl();

    rt.fetch(url, {
      method: method[type],
      signal: ac?.signal,
      body: getBody(),
      headers: {
        'content-type': 'application/json',
        ...rt.headers(),
      },
    })
      .then((res) => {
        return res.json();
      })
      .then((json) => {
        resolve(json);
      })
      .catch(reject);
  });
  const cancel = () => {
    ac?.abort();
  };
  return { promise, cancel };
}
