/*************************GO-LICENSE-START*********************************
 * Copyright 2014 ThoughtWorks, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *************************GO-LICENSE-END***********************************/


package com.thoughtworks.go.domain;

import org.apache.log4j.Logger;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLFilterImpl;

/**
 * This filter deletes elements without child nodes (text or elements) with
 * the given tagname.
 * It is used by the AntBuilder to get rid of empty tags
 * generated by Ant's XmlLogger, like empty messages and tasks.
 *
 * @author <a href="mailto:joriskuipers@xs4all.nl">Joris Kuipers</a>
 */
public class EmptyElementFilter extends XMLFilterImpl {

    private static final Logger LOG = Logger.getLogger(EmptyElementFilter.class);

    private final String tagName;

    private boolean checkingTag = false;
    // atts for storing the Element info
    private String messageUri;
    private String messageQName;
    private Attributes messageAtts;

    /**
     * Constructor for EmptyElementFilter.
     */
    public EmptyElementFilter(String tagName) {
        super();
        this.tagName = tagName;
    }

    /**
     * Constructor for EmptyElementFilter.
     * @param arg0
     */
    public EmptyElementFilter(String tagName, XMLReader arg0) {
        super(arg0);
        this.tagName = tagName;
    }

    /**
     * @see org.xml.sax.ContentHandler#characters(char[], int, int)
     */
    public void characters(char[] ch, int start, int length) throws SAXException {
        if (checkingTag) {
            cancelCheck();
        }
        super.characters(ch, start, length);
    }

    /**
     * @see org.xml.sax.ContentHandler#endElement(String, String, String)
     */
    public void endElement(String uri, String localName, String qName) throws SAXException {
        if (checkingTag) {
            // localName must be tagName now, and we're dealing with an empty tag
            checkingTag = false;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Skipping empty <" + localName + ">-tag");
            }
        } else {
            super.endElement(uri, localName, qName);
        }
    }

    /**
     * @see org.xml.sax.ContentHandler#startElement(String, String, String, org.xml.sax.Attributes)
     */
    public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException {
        if (checkingTag) {
            cancelCheck();
        }
        if (localName.equals(tagName)) {
            // hold the delegation until we're sure this is not an empty tag
            checkingTag = true;
            messageUri = uri;
            messageQName = qName;
            messageAtts = atts;
        } else {
            super.startElement(uri, localName, qName, atts);
        }
    }

    private void cancelCheck() throws SAXException {
        // delegate the describeChange that we've held back and stop checking
        super.startElement(messageUri, tagName, messageQName, messageAtts);
        checkingTag = false;
    }

}