/*************************GO-LICENSE-START*********************************
 * Copyright 2014 ThoughtWorks, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *************************GO-LICENSE-END***********************************/

package com.thoughtworks.go.util.pool;

import java.io.IOException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import org.apache.commons.codec.binary.Hex;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;

public class DigestObjectPoolsTest {

    private DigestObjectPools pools;

    @Before public void setUp() throws Exception {
        pools = new DigestObjectPools();
        pools.clearThreadLocals();

    }

    @After public void tearDown() throws Exception {
        pools.clearThreadLocals();
    }

    @Test
    public void shouldCallPerformActionWithADigest() throws IOException {
        DigestObjectPools.DigestOperation operation = mock(DigestObjectPools.DigestOperation.class);
        pools.computeDigest(DigestObjectPools.SHA_256, operation);
        verify(operation).perform(any(MessageDigest.class));
    }

    @Test
    public void shouldThrowIllegalArgumentExceptionWhenTheAlgorithmIsUnknown() throws IOException {
        DigestObjectPools.DigestOperation operation = mock(DigestObjectPools.DigestOperation.class);
        try {
            pools.computeDigest("upside_down_fred_rubble_bubble_cake", operation);
            fail("Expected to get an exception as the algorithm is Flintstones proprietary!");
        } catch (IllegalArgumentException expected) {
            assertThat(expected.getMessage(), is("Algorithm not supported"));
        }
    }

    @Test
    public void shouldResetDigestForFutureUsage() {
        DigestObjectPools.DigestOperation operation = new DigestObjectPools.DigestOperation() {
            public String perform(MessageDigest digest) throws IOException {
                digest.update(org.apache.commons.codec.binary.StringUtils.getBytesUtf8("foo"));
                return Hex.encodeHexString(digest.digest());
            }
        };
        String shaFirst = pools.computeDigest(DigestObjectPools.SHA_256, operation);
        String shaSecond = pools.computeDigest(DigestObjectPools.SHA_256, operation);
        assertThat(shaFirst, is(shaSecond));
    }

    @Test
    public void shouldCreateDigestOnlyIfItIsNotAlreadyInitializedOnThisThreadsThreadLocal() throws NoSuchAlgorithmException {
        DigestObjectPools.CreateDigest creator = mock(DigestObjectPools.CreateDigest.class);
        when(creator.create(DigestObjectPools.SHA_256)).thenReturn(MessageDigest.getInstance(DigestObjectPools.SHA_256));
        DigestObjectPools pools = new DigestObjectPools(creator);
        try {
            DigestObjectPools.DigestOperation operation = mock(DigestObjectPools.DigestOperation.class);

            pools.computeDigest(DigestObjectPools.SHA_256, operation);
            pools.computeDigest(DigestObjectPools.SHA_256, operation);

            verify(creator).create(DigestObjectPools.SHA_256);
            verifyNoMoreInteractions(creator);
        } finally {
            pools.clearThreadLocals();
        }
    }

    @Test
    public void shouldCreateDigestOnlyIfItIsNotAlreadyInitializedOnThreads() throws NoSuchAlgorithmException, InterruptedException {
        DigestObjectPools.CreateDigest creator = mock(DigestObjectPools.CreateDigest.class);
        when(creator.create(DigestObjectPools.SHA_256)).thenReturn(MessageDigest.getInstance(DigestObjectPools.SHA_256));
        final DigestObjectPools pools = new DigestObjectPools(creator);
        try {
            final DigestObjectPools.DigestOperation operation = mock(DigestObjectPools.DigestOperation.class);

            pools.computeDigest(DigestObjectPools.SHA_256, operation);
            pools.computeDigest(DigestObjectPools.SHA_256, operation);

            Thread thread = new Thread(new Runnable() {
                public void run() {
                    pools.computeDigest(DigestObjectPools.SHA_256, operation);
                    pools.computeDigest(DigestObjectPools.SHA_256, operation);
                }
            });
            thread.start();
            thread.join();
            verify(creator, times(2)).create(DigestObjectPools.SHA_256);
            verifyNoMoreInteractions(creator);
        } finally {
            pools.clearThreadLocals();
        }
    }
}
