/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.qpid.jms.provider.amqp.message;

import static org.apache.qpid.jms.provider.amqp.message.AmqpMessageSupport.encodeMessage;

import java.nio.charset.StandardCharsets;

import org.apache.qpid.jms.JmsDestination;
import org.apache.qpid.jms.JmsTopic;
import org.apache.qpid.jms.meta.JmsConnectionId;
import org.apache.qpid.jms.meta.JmsConnectionInfo;
import org.apache.qpid.jms.meta.JmsConsumerId;
import org.apache.qpid.jms.meta.JmsConsumerInfo;
import org.apache.qpid.jms.provider.amqp.AmqpConnection;
import org.apache.qpid.jms.provider.amqp.AmqpConsumer;
import org.apache.qpid.jms.test.QpidJmsTestCase;
import org.apache.qpid.proton.message.Message;
import org.junit.Before;
import org.mockito.Mockito;

public class AmqpJmsMessageTypesTestCase extends QpidJmsTestCase {

    private JmsDestination consumerDestination;

    @Before
    @Override
    public void setUp() throws Exception {
        super.setUp();
        consumerDestination = new JmsTopic("TestTopic");
    };

    //---------- Test Support Methods ----------------------------------------//

    protected AmqpJmsMessageFacade createNewMessageFacade() {
        return new AmqpJmsMessageFacade(createMockAmqpConnection());
    }

    protected AmqpJmsMessageFacade createReceivedMessageFacade(AmqpConsumer amqpConsumer, Message message) {
        return new AmqpJmsMessageFacade(amqpConsumer, message);
    }

    protected AmqpJmsTextMessageFacade createNewTextMessageFacade() {
        return new AmqpJmsTextMessageFacade(createMockAmqpConnection());
    }

    protected AmqpJmsTextMessageFacade createReceivedTextMessageFacade(AmqpConsumer amqpConsumer, Message message) {
        return new AmqpJmsTextMessageFacade(amqpConsumer, message, StandardCharsets.UTF_8);
    }

    protected AmqpJmsBytesMessageFacade createNewBytesMessageFacade() {
        return new AmqpJmsBytesMessageFacade(createMockAmqpConnection());
    }

    protected AmqpJmsBytesMessageFacade createReceivedBytesMessageFacade(AmqpConsumer amqpConsumer, Message message) {
        return new AmqpJmsBytesMessageFacade(amqpConsumer, message);
    }

    protected AmqpJmsMapMessageFacade createNewMapMessageFacade() {
        return new AmqpJmsMapMessageFacade(createMockAmqpConnection());
    }

    protected AmqpJmsMapMessageFacade createReceivedMapMessageFacade(AmqpConsumer amqpConsumer, Message message) {
        return new AmqpJmsMapMessageFacade(amqpConsumer, message);
    }

    protected AmqpJmsStreamMessageFacade createNewStreamMessageFacade() {
        return new AmqpJmsStreamMessageFacade(createMockAmqpConnection());
    }

    protected AmqpJmsStreamMessageFacade createReceivedStreamMessageFacade(AmqpConsumer amqpConsumer, Message message) {
        return new AmqpJmsStreamMessageFacade(amqpConsumer, message);
    }

    protected AmqpJmsObjectMessageFacade createNewObjectMessageFacade(boolean amqpTyped) {
        return new AmqpJmsObjectMessageFacade(createMockAmqpConnection(), amqpTyped);
    }

    protected AmqpJmsObjectMessageFacade createReceivedObjectMessageFacade(AmqpConsumer amqpConsumer, Message message) {
        return new AmqpJmsObjectMessageFacade(amqpConsumer, message, encodeMessage(message));
    }

    protected AmqpConsumer createMockAmqpConsumer() {
        JmsConsumerId consumerId = new JmsConsumerId("ID:MOCK:1:1:1");
        AmqpConnection connection = createMockAmqpConnection();
        AmqpConsumer consumer = Mockito.mock(AmqpConsumer.class);
        Mockito.when(consumer.getConnection()).thenReturn(connection);
        Mockito.when(consumer.getDestination()).thenReturn(consumerDestination);
        Mockito.when(consumer.getResourceInfo()).thenReturn(new JmsConsumerInfo(consumerId));
        return consumer;
    }

    protected AmqpConnection createMockAmqpConnection() {
        JmsConnectionId connectionId = new JmsConnectionId("ID:MOCK:1");
        AmqpConnection connection = Mockito.mock(AmqpConnection.class);
        Mockito.when(connection.getResourceInfo()).thenReturn(new JmsConnectionInfo(connectionId));

        return connection;
    }
}
