// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System;
using Xunit;

namespace System.Linq.Expressions.Tests
{
    public static class ArrayBoundsOneOffTests
    {
        [Fact]
        public static void CompileWithCastTest()
        {
            Expression<Func<object[]>> expr = () => (object[])new BaseClass[1];
            expr.Compile();
        }

        [Fact]
        public static void ToStringTest()
        {
            Expression<Func<int, object>> x = c => new double[c, c];
            Assert.Equal("c => new System.Double[,](c, c)", x.ToString());

            object y = x.Compile()(2);
            Assert.Equal("System.Double[,]", y.ToString());
        }

        [Fact]
        public static void ArrayBoundsVectorNegativeThrowsOverflowException()
        {
            Expression<Func<int, int[]>> e = a => new int[a];
            Func<int, int[]> f = e.Compile();

            Assert.Throws<OverflowException>(() => f(-1));
        }

        [Fact]
        public static void ArrayBoundsMultiDimensionalNegativeThrowsOverflowException()
        {
            Expression<Func<int, int, int[,]>> e = (a, b) => new int[a, b];
            Func<int, int, int[,]> f = e.Compile();

            Assert.Throws<OverflowException>(() => f(-1, 1));
            Assert.Throws<OverflowException>(() => f(1, -1));
        }
    }
}
