/*
 * Copyright 2017 Crown Copyright
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import Cookies from 'cookies-js';

export const EMAIL_CHANGE = 'login/EMAIL_CHANGE';
export const PASSWORD_CHANGE = 'login/PASSWORD_CHANGE';
export const TOKEN_CHANGE = 'login/TOKEN_CHANGE';
export const TOKEN_DELETE = 'login/TOKEN_DELETE';
export const SHOW_LOADER = 'login/SHOW_LOADER';
export const CHANGE_LOGGED_IN_USER = 'login/CHANGE_LOGGED_IN_USER';
export const SET_REDIRECT_URL = 'login/SET_REDIRECT_URL';
export const SET_CLIENT_ID = 'login/SET_CLIENT_ID';
export const SET_SESSION_ID = 'login/SET_SESSION_ID';

const initialState = {
  token: '',
  showLoader: false,
  loggedInUserEmail: undefined,
};

export default (state = initialState, action) => {
  switch (action.type) {
    case EMAIL_CHANGE:
      return {
        ...state,
        email: action.email,
      };

    case TOKEN_CHANGE:
      return {
        ...state,
        token: action.token,
      };
    case TOKEN_DELETE:
      return {
        ...state,
        token: '',
      };
    case SHOW_LOADER:
      return {
        ...state,
        showLoader: action.showLoader,
      };

    case CHANGE_LOGGED_IN_USER:
      return {
        ...state,
        loggedInUserEmail: action.userEmail,
      };

    case SET_REDIRECT_URL:
      return {
        ...state,
        redirectUrl: action.redirectUrl,
      };

    case SET_CLIENT_ID:
      return {
        ...state,
        clientId: action.clientId,
      };

    case SET_SESSION_ID:
      return {
        ...state,
        sessionId: action.sessionId,
      };

    default:
      return state;
  }
};

export function changeToken(token) {
  return {
    type: TOKEN_CHANGE,
    token: token,
  };
}

export function deleteToken() {
  return {
    type: TOKEN_DELETE,
  };
}

export const logout = () => {
  localStorage.removeItem('token');
  localStorage.removeItem('userEmail');
  return dispatch => {
    dispatch(deleteToken());
    dispatch(storeLoggedInUser(undefined));
  };
};

export function showLoader(showLoader) {
  return {
    type: SHOW_LOADER,
    showLoader,
  };
}

export function storeLoggedInUser(userEmail) {
  return {
    type: CHANGE_LOGGED_IN_USER,
    userEmail,
  };
}

/** This is the URL to redirect to after logging in, e.g. Stroom's location. */
export const changeRedirectUrl = redirectUrl => {
  return {
    type: SET_REDIRECT_URL,
    redirectUrl,
  };
};

/** This is the client ID of the relying party that originally requested authentication, e.g. Stroom. */
export const changeClientIdUrl = clientId => {
  return {
    type: SET_CLIENT_ID,
    clientId,
  };
};

export const changeSessionId = sessionId => {
  return {
    type: SET_SESSION_ID,
    sessionId,
  };
};

export const login = (credentials, {setStatus, setSubmitting}) => {
  return (dispatch, getState) => {
    const {email, password} = credentials;

    // We want to show a preloader while we're making the request. We turn it off when we receive a response or catch an error.
    dispatch(showLoader(true));
    const state = getState();
    const authenticationServiceUrl = state.config.authenticationServiceUrl;
    const loginServiceUrl = `${authenticationServiceUrl}/authenticate`;
    const clientId = state.login.clientId;
    const stroomUiUrl = state.config.stroomUiUrl;

    // Call the authentication service to get a token.
    // If successful we re-direct to Stroom, otherwise we display a message.
    // It's essential we return the promise, otherwise any errors we get won't be handled.
    return (
      fetch(loginServiceUrl, {
        // This option means we send the cookies along with the request, which means
        // the auth service gets the sessionId.
        credentials: "include",
        headers: {
          Accept: 'application/json',
          'Content-Type': 'application/json',
        },
        method: 'post',
        mode: 'cors',
        body: JSON.stringify({
          email,
          password,
          requestingClientId: clientId,
        }),
      })
        .then(response => {
          // We'll check for a 422, which we know might indicate there's no session
          if (response.status === 422) {
            setStatus(
              'There is no session on the authentication service! This might be caused ' +
                'by incorrectly configured service URLs.',
            );
            window.location.href = stroomUiUrl 
          } else {
            response.json().then(loginResponse => {
              if (loginResponse.loginSuccessful) {
                // Otherwise we'll extract what we expect to be the successful login redirect URL
                Cookies.set('username', email);
                window.location.href = loginResponse.redirectUrl;
              } else {
                setStatus(loginResponse.message);
                setSubmitting(false);
              }
              return;
            });
          }
        })
    );
  };
};

export function getStatusAndText(response) {
  return response.text().then(text => {
    return {status: response.status, text};
  });
}
