/*
 * Copyright 2017 Crown Copyright
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import jwtDecode from 'jwt-decode'
import uuidv4 from 'uuid/v4'
import sjcl from 'sjcl'
import { push } from 'react-router-redux'

import { canManageUsers } from './authorisation'

export const TOKEN_ID_CHANGE = 'authentication/TOKEN_ID_CHANGE'

const initialState = {
  idToken: ''
}

export const authenticationReducer = (state = initialState, action) => {
  switch (action.type) {
    case TOKEN_ID_CHANGE:
      return Object.assign({}, state, {
        idToken: action.idToken
      })

    default:
      return state
  }
}

function changeIdToken (idToken) {
  return {
    type: TOKEN_ID_CHANGE,
    idToken
  }
}

export const sendAuthenticationRequest = (referrer, uiUrl, appClientId, authenticationServiceUrl) => {
  return (dispatch, getState) => {
    const redirectUrl = `${uiUrl}/s/handleAuthenticationResponse`
    const state = ''

    // Create nonce and store, and create nonce hash
    const nonce = uuidv4()
    const nonceHashBytes = sjcl.hash.sha256.hash(nonce)
    const nonceHash = sjcl.codec.hex.fromBits(nonceHashBytes)
    localStorage.setItem('nonce', nonce)

    // We need to remember where the user was going
    localStorage.setItem('preAuthenticationRequestReferrer', referrer)

    // Compose the new URL
    const authenticationRequestParams = `?scope=openid&response_type=code&client_id=${appClientId}&redirect_url=${redirectUrl}&state=${state}&nonce=${nonceHash}`
    const authenticationRequestUrl = `${authenticationServiceUrl}/authenticate/${authenticationRequestParams}`

    // We hand off to the authenticationService.
    window.location.href = authenticationRequestUrl
  }
}

export const handleAuthenticationResponse = (accessCode, advertisedUrl, authorisationServiceUrl) => {
  return (dispatch) => {
    const idTokenRequestUrl = `${advertisedUrl}/api/authentication/v1/noauth/exchange`

    fetch(idTokenRequestUrl, {
      headers: {
        'Accept': 'application/json',
        'Content-Type': 'application/json'
      },
      method: 'post',
      credentials: 'include', // Send cookies, i.e. the sessionId
      mode: 'cors',
      body: JSON.stringify({accessCode})
    })
    .then(response => response.text())
    .then(idToken => {
      const decodedToken = jwtDecode(idToken)
      const nonce = localStorage.getItem('nonce')
      const nonceHashBytes = sjcl.hash.sha256.hash(nonce)
      const nonceHash = sjcl.codec.hex.fromBits(nonceHashBytes)
      const returnedNonce = decodedToken.nonce
      const referrer = localStorage.getItem('preAuthenticationRequestReferrer')

      if (nonceHash === returnedNonce) {
        localStorage.removeItem('nonce')
        localStorage.removeItem('preAuthenticationRequestReferrer')
        dispatch(changeIdToken(idToken))
        dispatch(canManageUsers(idToken, authorisationServiceUrl))
      } else {
        console.error('Nonce does not match.')
        // We fall through and push to the referrer, which will mean we attempt log in again.
        // Possibly we could add an error message here, so the user can understand why they
        // are being asked to log in again.
      }
      dispatch(push(referrer))
    })
  }
}
