package tree

import (
	"github.com/micro/go-micro/client"
	"github.com/micro/go-micro/errors"
	"github.com/pydio/services/common"
	"github.com/pydio/services/common/proto/jobs"
	"github.com/pydio/services/common/proto/tree"
	"github.com/pydio/services/common/views"
	"golang.org/x/net/context"
	"strconv"
)

type CopyMoveAction struct {
	Client            views.Handler
	Move              bool
	TargetPlaceholder string
	CreateFolder      bool
}

var (
	copyMoveActionName = "actions.tree.copymove"
)

// Unique identifier
func (c *CopyMoveAction) GetName() string {
	return copyMoveActionName
}

// Pass parameters
func (c *CopyMoveAction) Init(job *jobs.Job, cl client.Client, action *jobs.Action) error {

	c.Client = views.NewStandardRouter(true)

	if action.Parameters == nil {
		return errors.InternalServerError(common.SERVICE_JOBS, "Could not find parameters for CopyMove action")
	}
	var tOk bool
	c.TargetPlaceholder, tOk = action.Parameters["target"]
	if !tOk {
		return errors.InternalServerError(common.SERVICE_JOBS, "Could not find parameters for CopyMove action")
	}
	c.Move = false
	if actionType, ok := action.Parameters["type"]; ok && actionType == "move" {
		c.Move = true
	}

	if createParam, ok := action.Parameters["create"]; ok {
		c.CreateFolder, _ = strconv.ParseBool(createParam)
	}

	return nil
}

// Run the actual action code
func (c *CopyMoveAction) Run(ctx context.Context, input jobs.ActionMessage) (jobs.ActionMessage, error) {

	if len(input.Nodes) == 0 {
		return input.WithIgnore(), nil // Ignore
	}

	targetNode := &tree.Node{
		Path: c.TargetPlaceholder + "/" + input.Nodes[0].GetStringMeta("name"),
	}

	if targetNode.Path == input.Nodes[0].Path {
		// Do not copy on itself, ignore
		return input, nil
	}

	_, e := c.Client.CopyObject(ctx, input.Nodes[0], targetNode, &views.CopyRequestData{})
	if e != nil {
		return input.WithError(e), e
	}
	output := input.WithNode(targetNode)

	if c.Move {
		_, moveErr := c.Client.DeleteNode(ctx, &tree.DeleteNodeRequest{Node: input.Nodes[0]})
		if moveErr != nil {
			return output.WithError(moveErr), moveErr
		}
	}

	output.AppendOutput(&jobs.ActionOutput{
		Success: true,
	})

	return output, nil
}
