package tree

import (
	"github.com/pydio/services/common/proto/jobs"
	"github.com/pydio/services/common/proto/tree"
	"github.com/pydio/services/common/views"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
	"testing"
)

func TestCopyMoveAction_GetName(t *testing.T) {
	Convey("Test GetName", t, func() {
		metaAction := &CopyMoveAction{}
		So(metaAction.GetName(), ShouldEqual, copyMoveActionName)
	})
}

func TestCopyMoveAction_Init(t *testing.T) {
	Convey("", t, func() {
		action := &CopyMoveAction{}
		job := &jobs.Job{}
		// Test action without parameters
		e := action.Init(job, nil, &jobs.Action{})
		So(e, ShouldNotBeNil)

		// Test action without empty target parameters
		e = action.Init(job, nil, &jobs.Action{
			Parameters: map[string]string{
				"paramName": "paramValue",
			},
		})
		So(e, ShouldNotBeNil)

		// Test action with parameters
		e = action.Init(job, nil, &jobs.Action{
			Parameters: map[string]string{
				"target": "target/path",
				"type":   "move",
				"create": "true",
			},
		})
		So(e, ShouldBeNil)
		So(action.TargetPlaceholder, ShouldEqual, "target/path")
		So(action.CreateFolder, ShouldBeTrue)
		So(action.Move, ShouldBeTrue)

	})
}

func TestCopyMoveAction_RunCopy(t *testing.T) {

	Convey("", t, func() {

		action := &CopyMoveAction{}
		job := &jobs.Job{}
		action.Init(job, nil, &jobs.Action{
			Parameters: map[string]string{
				"target": "target/path",
				"type":   "copy",
				"create": "true",
			},
		})
		mock := &views.HandlerMock{
			Nodes: map[string]*tree.Node{},
		}
		action.Client = mock
		ignored, err := action.Run(context.Background(), jobs.ActionMessage{
			Nodes: []*tree.Node{},
		})
		So(ignored.GetLastOutput(), ShouldResemble, &jobs.ActionOutput{Ignored: true})

		output, err := action.Run(context.Background(), jobs.ActionMessage{
			Nodes: []*tree.Node{&tree.Node{
				Path:      "path/to/original",
				MetaStore: map[string]string{"name": `"original"`},
			}},
		})
		So(err, ShouldBeNil)
		So(output.Nodes, ShouldHaveLength, 1)
		So(output.Nodes[0].Path, ShouldEqual, "target/path/original")

		So(mock.Nodes, ShouldHaveLength, 2)
		So(mock.Nodes["from"].Path, ShouldEqual, "path/to/original")
		So(mock.Nodes["to"].Path, ShouldEqual, "target/path/original")

	})
}

func TestCopyMoveAction_RunCopyOnItself(t *testing.T) {

	Convey("", t, func() {

		action := &CopyMoveAction{}
		job := &jobs.Job{}
		action.Init(job, nil, &jobs.Action{
			Parameters: map[string]string{
				"target": "target/path",
				"type":   "copy",
				"create": "true",
			},
		})
		mock := &views.HandlerMock{
			Nodes: map[string]*tree.Node{},
		}
		action.Client = mock
		ignored, err := action.Run(context.Background(), jobs.ActionMessage{
			Nodes: []*tree.Node{},
		})
		So(ignored.GetLastOutput(), ShouldResemble, &jobs.ActionOutput{Ignored: true})

		output, err := action.Run(context.Background(), jobs.ActionMessage{
			Nodes: []*tree.Node{&tree.Node{
				Path:      "target/path/original",
				MetaStore: map[string]string{"name": `"original"`},
			}},
		})
		So(err, ShouldBeNil)
		So(output.Nodes, ShouldHaveLength, 1)
		So(output.Nodes[0].Path, ShouldEqual, "target/path/original")

		So(mock.Nodes, ShouldHaveLength, 0)

	})
}

func TestCopyMoveAction_RunMove(t *testing.T) {

	Convey("", t, func() {

		action := &CopyMoveAction{}
		job := &jobs.Job{}
		action.Init(job, nil, &jobs.Action{
			Parameters: map[string]string{
				"target": "target/path",
				"type":   "move",
				"create": "true",
			},
		})
		mock := &views.HandlerMock{
			Nodes: map[string]*tree.Node{},
		}
		action.Client = mock
		ignored, err := action.Run(context.Background(), jobs.ActionMessage{
			Nodes: []*tree.Node{},
		})
		So(ignored.GetLastOutput(), ShouldResemble, &jobs.ActionOutput{Ignored: true})

		output, err := action.Run(context.Background(), jobs.ActionMessage{
			Nodes: []*tree.Node{&tree.Node{
				Path:      "path/to/original",
				MetaStore: map[string]string{"name": `"original"`},
			}},
		})
		So(err, ShouldBeNil)
		So(output.Nodes, ShouldHaveLength, 1)
		So(output.Nodes[0].Path, ShouldEqual, "target/path/original")

		So(mock.Nodes, ShouldHaveLength, 3)
		So(mock.Nodes["from"].Path, ShouldEqual, "path/to/original")
		So(mock.Nodes["to"].Path, ShouldEqual, "target/path/original")
		// Deleted Node
		So(mock.Nodes["in"].Path, ShouldEqual, "path/to/original")

	})
}
