---
id: infer-types
title: Inferring Types
sidebar_label: Inferring Types
slug: /infer-types
---

<!-- Reusable snippet -->

```twoslash include server
// @module: esnext
// @filename: server.ts
import { initTRPC } from '@trpc/server';
import { z } from "zod";

const t = initTRPC.create();

const appRouter = t.router({
  post: t.router({
    byId: t.procedure
      .input(z.string())
      .query(({ input }) => {
        // imaginary db call
        return { id: 1, title: 'tRPC is the best!' };
    }),
    create: t.procedure
      .input(z.object({ title: z.string(), text: z.string(), }))
      .mutation(({ input }) => {
        // imaginary db call
        return { id: 1, ...input };
    }),
  }),
});

export type AppRouter = typeof appRouter;
```

It is often useful to wrap functionality of your `@trpc/client` or `@trpc/react` api within other functions. For this purpose, it's necessary to be able to infer input types and output types generated by your `@trpc/server` router.

## Inference Helpers

`@trpc/server` exports the following helper types to assist with inferring these types from the `AppRouter` exported by your `@trpc/server` router:

- `inferProcedureOutput<TProcedure>`
- `inferProcedureInput<TProcedure>`

Let's assume we have this example router:

```ts twoslash title='server.ts'
// @include: server
```

By traversing the router object, you can infer the types of the procedures. The following example shows how to infer the types of the procedures using the example `appRouter`:

```ts twoslash title="client.ts"
// @module: esnext
// @include: server
// ---cut---
// @filename: client.ts
import type { inferProcedureInput, inferProcedureOutput } from '@trpc/server';
import type { AppRouter } from './server';

// @noErrors
type PostCreateInput = inferProcedureInput<AppRouter['post']['create']>;
//   ^?
type PostCreateOutput = inferProcedureOutput<AppRouter['post']['create']>;
//   ^?
```

### Additional DX Helper Type

If you don't like the double-import from the above snippet, `@trpc/server` also exports a type `GetInferenceHelpers<TRouter>`. This lets you pass your router once at initialization, then import a single helper type when inferring types:

```ts twoslash title='utils/trpc.ts'
// @module: esnext
// @include: server
// @filename: index.ts
// ---cut---
import type { GetInferenceHelpers } from '@trpc/server';
import type { AppRouter } from './server';

export type InferProcedures = GetInferenceHelpers<AppRouter>;
```

<!-- FIXME: reuse above snippet -->

```ts twoslash
// @module: esnext
// @include: server
// @filename: index.ts
import type { GetInferenceHelpers } from '@trpc/server';
import type { AppRouter } from './server';

export type InferProcedures = GetInferenceHelpers<AppRouter>;

// ---cut---
export type AppRouterTypes = GetInferenceHelpers<AppRouter>;

type PostCreate = AppRouterTypes['post']['create'];

type PostCreateInput = PostCreate['input'];
//   ^?
type PostCreateOutput = PostCreate['output'];
//   ^?
```

## Infer `TRPClientError`s based on your router

```ts twoslash title='client.ts'
// @module: esnext
// @include: server

// @filename: trpc.ts
import { createTRPCProxyClient, httpBatchLink } from "@trpc/client";
import type { AppRouter } from "./server";

export const trpc = createTRPCProxyClient<AppRouter>({
  links: [
    httpBatchLink({
      url: "http://localhost:3000/api/trpc",
    }),
  ],
});

// ---cut---
// @filename: client.ts
import { TRPCClientError } from '@trpc/client';
import type { AppRouter } from './server';
import { trpc } from './trpc';

export function isTRPCClientError(
  cause: unknown,
): cause is TRPCClientError<AppRouter> {
  return cause instanceof TRPCClientError;
}

async function main() {
  try {
    await trpc.post.byId.query('1');
  } catch (cause) {
    if (isTRPCClientError(cause)) {
      // `cause` is now typed as your router's `TRPCClientError`
      console.log('data', cause.data);
      //                        ^?
    } else {
      // [...]
    }
  }
}

main();
```
