// Copyright 2019 Yunion
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package tasks

import (
	"context"

	"yunion.io/x/jsonutils"
	"yunion.io/x/pkg/errors"

	api "yunion.io/x/onecloud/pkg/apis/cloudid"
	"yunion.io/x/onecloud/pkg/cloudcommon/db"
	"yunion.io/x/onecloud/pkg/cloudcommon/db/taskman"
	"yunion.io/x/onecloud/pkg/cloudid/models"
)

type CloudpolicyUpdateTask struct {
	taskman.STask
}

func init() {
	taskman.RegisterTask(CloudpolicyUpdateTask{})
}

func (self *CloudpolicyUpdateTask) taskFailed(ctx context.Context, policy *models.SCloudpolicy, err error) {
	policy.SetStatus(self.GetUserCred(), api.CLOUD_POLICY_STATUS_SYNC_FAILE, err.Error())
	self.SetStageFailed(ctx, jsonutils.NewString(err.Error()))
}

func (self *CloudpolicyUpdateTask) OnInit(ctx context.Context, obj db.IStandaloneModel, body jsonutils.JSONObject) {
	policy := obj.(*models.SCloudpolicy)

	caches, err := policy.GetCloudpolicycaches()
	if err != nil {
		self.taskFailed(ctx, policy, errors.Wrapf(err, "GetCloudpolicycaches"))
		return
	}
	for i := range caches {
		if len(caches[i].ExternalId) > 0 {
			provider, err := caches[i].GetProvider()
			if err != nil {
				self.taskFailed(ctx, policy, errors.Wrapf(err, "GetProvider for cache %s(%s)", caches[i].Name, caches[i].Id))
				return
			}
			policies, err := provider.GetICustomCloudpolicies()
			if err != nil {
				self.taskFailed(ctx, policy, errors.Wrapf(err, "GetICustomCloudpolicies for account %s provider %s", caches[i].CloudaccountId, caches[i].CloudproviderId))
				return
			}
			for i := range policies {
				if policies[i].GetGlobalId() == caches[i].ExternalId {
					err = policies[i].UpdateDocument(policy.Document)
					if err != nil {
						self.taskFailed(ctx, policy, errors.Wrapf(err, "UpdateDocument for %s", policies[i].GetName()))
						return
					}
				}
			}
		}
	}

	policy.SetStatus(self.GetUserCred(), api.CLOUD_POLICY_STATUS_AVAILABLE, "")
	self.SetStageComplete(ctx, nil)
}
