import {Gate, GateBuilder} from "src/circuit/Gate.js"
import {GatePainting} from "src/draw/GatePainting.js"
import {Matrix} from "src/math/Matrix.js"
import {Point} from "src/math/Point.js"
import {ketArgs, ketShader, ketShaderPermute} from "src/circuit/KetShaderUtil.js"

/**
 * Gates that correspond to 180 degree rotations around the Bloch sphere, so they're their own inverses.
 */
let HalfTurnGates = {};

/**
 * The X gate is drawn as a crossed circle when it has controls.
 * @param {!GateDrawParams} args
 */
function NOT_DRAWER(args) {
    let hasSingleWireControl =
        args.positionInCircuit !== undefined &&
        args.stats.circuitDefinition.colHasSingleWireControl(args.positionInCircuit.col);
    let hasDoubleWireControl =
        args.positionInCircuit !== undefined &&
        args.stats.circuitDefinition.colHasDoubleWireControl(args.positionInCircuit.col);
    if ((!hasSingleWireControl && !hasDoubleWireControl) || args.isHighlighted) {
        GatePainting.DEFAULT_DRAWER(args);
        return;
    }

    let drawArea = args.rect.scaledOutwardBy(0.6);
    args.painter.fillCircle(drawArea.center(), drawArea.w / 2);
    args.painter.strokeCircle(drawArea.center(), drawArea.w / 2);
    if (hasSingleWireControl) {
        args.painter.strokeLine(drawArea.topCenter(), drawArea.bottomCenter());
    }
    if (hasDoubleWireControl) {
        args.painter.strokeLine(drawArea.topCenter().offsetBy(-1, 0), drawArea.bottomCenter().offsetBy(-1, 0));
        args.painter.strokeLine(drawArea.topCenter().offsetBy(+1, 0), drawArea.bottomCenter().offsetBy(+1, 0));
    }
    let isMeasured = args.stats.circuitDefinition.locIsMeasured(
        new Point(args.positionInCircuit.col, args.positionInCircuit.row));
    if (isMeasured) {
        args.painter.strokeLine(drawArea.centerLeft().offsetBy(0, -1), drawArea.centerRight().offsetBy(0, -1));
        args.painter.strokeLine(drawArea.centerLeft().offsetBy(0, +1), drawArea.centerRight().offsetBy(0, +1));
    } else {
        args.painter.strokeLine(drawArea.centerLeft(), drawArea.centerRight());
    }
}

let xShader = ketShaderPermute('', 'return 1.0-out_id;', 1);
/** @type {!Gate} */
HalfTurnGates.X = new GateBuilder().
    setSerializedIdAndSymbol("X").
    setTitle("Pauli X Gate").
    setBlurb("The NOT gate.\nToggles between ON and OFF.").
    setDrawer(NOT_DRAWER).
    setActualEffectToShaderProvider(ctx => xShader.withArgs(...ketArgs(ctx))).
    setKnownEffectToMatrix(Matrix.PAULI_X).
    gate;

let yShader = ketShader('', 'vec2 v = inp(1.0-out_id); return (out_id*2.0 - 1.0)*vec2(-v.y, v.x);', 1);
/** @type {!Gate} */
HalfTurnGates.Y = new GateBuilder().
    setSerializedIdAndSymbol("Y").
    setTitle("Pauli Y Gate").
    setBlurb("A combination of the X and Z gates.").
    setActualEffectToShaderProvider(ctx => yShader.withArgs(...ketArgs(ctx))).
    setKnownEffectToMatrix(Matrix.PAULI_Y).
    gate;

let zShader = ketShader('', 'return amp*(1.0 - out_id*2.0);', 1);
/** @type {!Gate} */
HalfTurnGates.Z = new GateBuilder().
    setSerializedIdAndSymbol("Z").
    setTitle("Pauli Z Gate").
    setBlurb("The phase flip gate.\nNegates phases when the qubit is ON.").
    setActualEffectToShaderProvider(ctx => zShader.withArgs(...ketArgs(ctx))).
    setKnownEffectToMatrix(Matrix.PAULI_Z).
    gate;

let hShader = ketShader('', 'return 0.7071067811865476*(amp*(1.0-2.0*out_id) + inp(1.0-out_id));', 1);
/** @type {!Gate} */
HalfTurnGates.H = new GateBuilder().
    setSerializedIdAndSymbol("H").
    setTitle("Hadamard Gate").
    setBlurb("Creates simple superpositions.\n" +
        "Maps ON to ON + OFF.\n" +
        "Maps OFF to ON - OFF.").
    setActualEffectToShaderProvider(ctx => hShader.withArgs(...ketArgs(ctx))).
    setKnownEffectToMatrix(Matrix.HADAMARD).
    gate;

HalfTurnGates.all = [
    HalfTurnGates.X,
    HalfTurnGates.Y,
    HalfTurnGates.Z,
    HalfTurnGates.H
];

export {HalfTurnGates}
