---
title: ServiceStack Tutorial
name: ServiceStack
image: /media/platforms/service-stack.png
tags:
  - quickstart
snippets:
  dependencies: server-platforms/servicestack/dependencies
  setup: server-platforms/servicestack/setup
  use: server-platforms/servicestack/use
---

# ServiceStack Tutorial

At the end of this tutorial you will have a working web site that calls a ServiceStack API with authenticated users.

## Before you start

We assume you are familiar with [ServiceStack](http://www.servicestack.net/)

### 1. Create a simple MVC3 website and install ServiceStack through NuGet

For this example, we will use the standard template that ships with Visual Studio 2012. Select __"FILE -> New project -> MVC 3 Web Application -> Empty"__

Once the default template unfolds, use NuGet to install the **ServiceStack.Host.Mvc** nuget, running the command:

${snippet(meta.snippets.dependencies)}

![](/media/articles/server-platforms/servicestack/install-servicestack-nuget.png)

Add the following line to your `Global.asax` file (this is required for ServiceStack):

```
routes.IgnoreRoute("api/{*pathInfo}");
```

Add a `HomeController` to return the `default.htm` page. Under the __Controllers__ folder add:

```c#
public class HomeController : Controller
{
    public ActionResult Index()
    {
        return Redirect("default.htm");
    }
}
```

### 2. Install Auth0 OAuthProvider

We provide a [Nuget package](http://nuget.org/packages/Auth0-ServiceStack-OAuthProvider/) to simplify integration of Auth0 with ServiceStack based applications.

Run this command on the __Package Manager Console__:

	Install-Package Auth0-ServiceStack-OAuthProvider

This command will add two classes to your project under the __App_Start__ folder: `Auth0Provider` and `Auth0UserSession`.

`Auth0Provider` extends ServiceStack's `OAuthProvider` and handles the authentication transaction for you.

### 3. Enable Authentication and plug in Auth0's Provider

Open the `AppHost.cs` file (also under the __App_Start__ folder) generated by the ServiceStack NuGet. Uncomment the following call under the `Configure` method:

```
ConfigureAuth(container);
```

And then uncomment and edit the `ConfigureAuth` method to look like this:

${snippet(meta.snippets.dependencies)}

> In this sample we are not interested in user registration. So we are leaving that section out.

### 4. Setting up the callback URL in Auth0

<div class="setup-callback">
<p>After authenticating the user on Auth0, we will do a POST to a URL on your web site. For security purposes, you have to register this URL on the <a href="${uiAppSettingsURL}">Application Settings</a> section on Auth0 Admin app (make sure to change the port).</p>

<pre><code>http://localhost:PORT/api/auth/auth0/</pre></code>
</div>

### 5. Enter configuration settings

Open your `web.config` file and change the three Auth0's parameters under `<appSettings>`:

```
<add key="oauth.auth0.AppId" value="${account.clientId}" />
<add key="oauth.auth0.AppSecret" value="${account.clientSecret}" />
<add key="oauth.auth0.OAuthServerUrl" value="https://${account.namespace}" />
```


### 6. Add backend code to test the app

Open the the `WebServiceExamples.cs` file.

#### 6.1. Add the `Authenticate` attribute to the `Hello` DTO:

```c#
[Authenticate]
public class Hello
{
	public string Name { get; set; }
}
```

#### 6.2. Add an `IAuthSession` property to the `HelloResponse` DTO:

```c#
public class HelloResponse
{
    public IAuthSession UserInfo { get; set; }
	public string Result { get; set; }
	public ResponseStatus ResponseStatus { get; set; } //Where Exceptions get auto-serialized
}
```

#### 6.3. Modify the `HelloService` to return the currently logged in user's `UserInfo` object

```c#
public class HelloService : ServiceBase<Hello>
{
	protected override object Run(Hello request)
	{
        IAuthSession session = this.GetSession();
        var sb = new StringBuilder();
        sb.AppendLine("Id: " + session.Id);
        sb.AppendLine("DisplayName: " + session.DisplayName);

        var auth0Session = session as Auth0UserSession;
        var pic = auth0Session.ExtraData["picture"];

        return new HelloResponse { UserInfo = session };
	}
}
```
> Notice we are not doing anything useful with these properties. You can place a breakpoint here and explore the session object.

### 7. Triggering login manually or integrating the Auth0Lock

${lockSDK}

### 8. Add UI code to Login and invoke the `HelloService`

Open `default.htm` and add the following statement in the `jQuery.ready` body:

```js
// get user info from hello endpoint
$.getJSON('/api/hello', function (data) {
    $('#userInfo').text(JSON.stringify(data.userInfo, 1, 2));
});
```

Add a section to display the `UserInfo`:

```html
<div>User Info:
     <pre><code id="userInfo">Not logged in</code></pre>
</div>
```

### 9. Run the app

After successful authentication, the `UserProfile` will be displayed on the page.

**Congratulations!**
