# Database: Seeding

[[toc]]

## Introduction

Goravel includes the ability to seed your database with data using seed struct. All seed structs are stored in the `database/seeders` directory. By default, a `DatabaseSeeder` struct is defined for you.

## Writing Seeders

To generate a seeder, run the `make:seeder` [Artisan command](../digging-deeper/artisan-console.md). All seeders generated by the framework will be stored in the `database/seeders` directory:

```shell
go run . artisan make:seeder UserSeeder
```

By default, a seeder struct has two methods: `Signature` and `Run`. The `Signature` method sets the name of the seeder, while the `Run` method is triggered when the `db:seed` Artisan command is executed. You can use the `Run` method to insert data into your database in any way you prefer.

To illustrate, we can customize the `DatabaseSeeder` struct by adding a database insert statement to the `Run` method.

```go
package seeders

import (
	"github.com/goravel/framework/contracts/database/seeder"
	"github.com/goravel/framework/facades"

	"goravel/app/models"
)

type DatabaseSeeder struct {
}

// Signature The name and signature of the seeder.
func (s *DatabaseSeeder) Signature() string {
	return "DatabaseSeeder"
}

// Run executes the seeder logic.
func (s *DatabaseSeeder) Run() error {
	user := models.User{
		Name: "goravel",
	}
	return facades.Orm().Query().Create(&user)
}
```

## Calling Additional Seeders

Within the `DatabaseSeeder` struct, you may use the `Call` method to execute additional seed structs. Using the `Call` method allows you to break up your database seeding into multiple files so that no single seeder struct becomes too large. The `Call` method accepts an array of seeder structs that should be executed:

```go
// Run executes the seeder logic.
func (s *DatabaseSeeder) Run() error {
	return facades.Seeder().Call([]seeder.Seeder{
		&UserSeeder{},
	})
}
```

Framework also provides a `CallOnce` method, a seeder will be executed only once in the `db:seed` command:

```go
// Run executes the seeder logic.
func (s *DatabaseSeeder) Run() error {
	return facades.Seeder().CallOnce([]seeder.Seeder{
		&UserSeeder{},
	})
}
```

## Running Seeders

You may run the `db:seed` Artisan command to seed your database. By default, the `db:seed` command runs the `database/seeders/database_seeder.go` file, which may in turn invoke other seed classes. However, you may use the `--seeder` option to specify a specific seeder class to run individually:

```shell
go run . artisan db:seed
```

If you want to execute other seeders when running the `db:seed` command, you can register the seeder in `app/providers/database_service_provider.go`:

```go
// app/providers/database_service_provider.go
func (receiver *DatabaseServiceProvider) Boot(app foundation.Application) {
	facades.Seeder().Register([]seeder.Seeder{
		&seeders.DatabaseSeeder{},
        &seeders.UserSeeder{},
        &seeders.PhotoSeeder{},
	})
}

go run . artisan db:seed --seeder=UserSeeder PhotoSeeder // The signature of seeder
```

You may also seed your database using the `migrate:fresh` and `migrate:refresh` command in combination with the `--seed` option, which will drop all tables and re-run all of your migrations. This command is useful for completely re-building your database. The `--seeder` option may be used to specify a specific seeder to run:

```shell
go run . artisan migrate:fresh --seed

go run . artisan migrate:fresh --seed --seeder=UserSeeder

go run . artisan migrate:refresh --seed

go run . artisan migrate:refresh --seed --seeder=UserSeeder
```

### Forcing Seeders To Run In Production

Some seeding operations may cause you to alter or lose data. In order to protect you from running seeding commands against your production database, you will be prompted for confirmation before the seeders are executed in the `production` environment. To force the seeders to run without a prompt, use the `--force` flag:

```shell
go run . artisan db:seed --force
```
