package tests

import (
	"fmt"
	"testing"

	"github.com/goal-web/application"
	"github.com/goal-web/config"
	"github.com/goal-web/contracts"
	"github.com/goal-web/database"
	"github.com/goal-web/database/table"
	"github.com/goal-web/supports/exceptions"
	"github.com/goal-web/supports/utils"
	"github.com/stretchr/testify/assert"
)

type User struct {
	Id        int    `json:"id" db:"id"`
	Name      string `json:"name" db:"name"`
	Age       int    `json:"age" db:"age"`
	CreatedAt string `json:"created_at"`
	UpdatedAt string `json:"updated_at"`
}

var tableName string

func init() {
	app := application.Default()

	tableName = "test_" + utils.RandStr(10)

	app.RegisterServices(
		exceptions.NewService([]contracts.Exception{}),
		config.NewService(config.NewDotEnv(config.File("")), map[string]contracts.ConfigProvider{
			"app": func(env contracts.Env) any {
				return application.Config{
					Name:     env.GetString("app.name"),
					Debug:    env.GetBool("app.debug"),
					Timezone: env.GetString("app.timezone"),
					Env:      env.GetString("app.env"),
					Locale:   env.GetString("app.locale"),
					Key:      env.GetString("app.key"),
				}
			},
			"database": func(env contracts.Env) any {
				return database.Config{
					Default: "mysql",
					Connections: map[string]contracts.Fields{
						"mysql": {
							"driver":          "mysql",
							"host":            "localhost",
							"port":            "3306",
							"database":        "goal",
							"username":        "root",
							"password":        "root",
							"charset":         env.StringOptional("db.charset", "utf8mb4"),
							"collation":       env.StringOptional("db.collation", "utf8mb4_unicode_ci"),
							"prefix":          env.GetString("db.prefix"),
							"strict":          env.GetBool("db.struct"),
							"max_connections": env.GetInt("db.max_connections"),
							"max_idles":       env.GetInt("db.max_idles"),
						},
					},
				}
			},
		}),
		database.NewService(),
	)

	app.Start()
	app.Call(func(connection contracts.DBConnection) {
		_, err := connection.Exec("CREATE TABLE IF NOT EXISTS " + tableName +
			"(" +
			"    `id`       INT UNSIGNED AUTO_INCREMENT," +
			"    name       varchar(20)," +
			"    age        int," +
			"    created_at timestamp," +
			"    updated_at timestamp," +
			"    PRIMARY KEY (`id`)" +
			") ENGINE = InnoDB" +
			"  DEFAULT CHARSET = utf8mb4;")
		if err != nil {
			panic(err)
		}
	})
}

var f = func(fields contracts.Fields) *User {
	return &User{
		Id:        utils.GetIntField(fields, "id"),
		Name:      utils.GetStringField(fields, "name"),
		Age:       utils.GetIntField(fields, "age"),
		CreatedAt: utils.GetStringField(fields, "created_at"),
		UpdatedAt: utils.GetStringField(fields, "updated_at"),
	}
}

func UserQuery(name string) *table.Table[User] {
	return table.Query[User](name).SetFactory(f)
}

func TestMysqlDatabaseService(t *testing.T) {

	assert.True(t, UserQuery(tableName).Count() == 0)

	user := UserQuery(tableName).SetFactory(f).Create(contracts.Fields{
		"name": "testing",
	})
	assert.NotNil(t, user)
	assert.True(t, user.Name == "testing")
	assert.True(t, UserQuery(tableName).Count() == 1)
	assert.True(t, UserQuery(tableName).Get().Count() == 1)
	UserQuery(tableName).Where("name", "testing").Delete()
	assert.True(t, UserQuery(tableName).Count() == 0)

}

func TestMysqlDatabaseWithoutApplication(t *testing.T) {
	// 实例化数据库工厂
	factory := database.NewFactory(
		database.Config{
			Default: "mysql",
			Connections: map[string]contracts.Fields{
				"mysql": {
					"driver":    "mysql",
					"host":      "localhost",
					"port":      "3306",
					"database":  "goal",
					"username":  "root",
					"password":  "root",
					"charset":   "utf8mb4",
					"collation": "utf8mb4_unicode_ci",
				},
			},
		},
		nil, // 第二个参数是一个 goal 的事件实例，非 goal 环境的情况下，允许为 nil
	)

	// 为 table 包设置数据库工厂
	table.SetFactory(factory)

	assert.True(t, table.ArrayQuery(tableName).Count() == 0)

	user := *table.ArrayQuery(tableName).Create(contracts.Fields{
		"name": "testing",
	})
	assert.NotNil(t, user)
	assert.True(t, user["name"] == "testing")
	assert.True(t, table.ArrayQuery(tableName).Count() == 1)
	table.ArrayQuery(tableName).Where("name", "testing").Delete()
	assert.True(t, table.ArrayQuery(tableName).Count() == 0)

}

func TestMysqlDatabaseFeature(t *testing.T) {
	// 实例化数据库工厂
	factory := database.NewFactory(
		database.Config{
			Default: "mysql",
			Connections: map[string]contracts.Fields{
				"mysql": {
					"driver":    "mysql",
					"host":      "localhost",
					"port":      "3306",
					"database":  "goal",
					"username":  "root",
					"password":  "root",
					"charset":   "utf8mb4",
					"collation": "utf8mb4_unicode_ci",
				},
			},
		},
		nil, // 第二个参数是一个 goal 的事件实例，非 goal 环境的情况下，允许为 nil
	)

	// 为 table 包设置数据库工厂
	table.SetFactory(factory)

	_, err := table.ArrayQuery(tableName).DeleteE()
	assert.NoError(t, err, err)

	count, err := table.ArrayQuery(tableName).CountE()
	assert.NoError(t, err, err)
	assert.True(t, count == 0)

	user, exception := UserQuery(tableName).CreateE(contracts.Fields{"name": "testing", "age": 18})
	assert.NotNil(t, user)
	assert.NoError(t, exception, exception)
	assert.True(t, user.Name == "testing")
	assert.True(t, user.Age == 18)

	ageSum, err := table.ArrayQuery(tableName).SumE("age")
	assert.NoError(t, err, err)
	assert.True(t, ageSum == 18)

	ageAvg, err := table.ArrayQuery(tableName).AvgE("age")
	assert.NoError(t, err, err)
	assert.True(t, ageAvg == 18)

	ageMin, err := table.ArrayQuery(tableName).MinE("age")
	assert.NoError(t, err, err)
	assert.True(t, ageMin == 18)

	ageMax, err := table.ArrayQuery(tableName).MaxE("age")
	assert.NoError(t, err, err)
	assert.True(t, ageMax == 18)

	err = table.ArrayQuery(tableName).Chunk(10, func(collection contracts.Collection[*contracts.Fields], page int) contracts.Exception {
		assert.True(t, page == 1)
		assert.True(t, collection.Count() == 1)

		collection.Each(func(i int, fields *contracts.Fields) *contracts.Fields {
			fmt.Println(i, *fields)
			assert.True(t, (*fields)["name"] == "testing")
			return nil
		})

		return nil
	})
	assert.NoError(t, err, err)

	user, exception = UserQuery(tableName).CreateE(contracts.Fields{"name": "testing2", "age": 18})
	assert.NotNil(t, user)
	assert.NoError(t, exception, exception)

	err = table.ArrayQuery(tableName).ChunkById(1, func(collection contracts.Collection[*contracts.Fields], page int) (any, contracts.Exception) {
		var p, _ = collection.First()
		result := *p
		switch page {
		case 1:
			assert.True(t, result["name"] == "testing")
		case 2:
			assert.True(t, result["name"] == "testing2")
		}

		return result["id"], nil
	})
	assert.NoError(t, err, err)

	err = table.ArrayQuery(tableName).ChunkByIdDesc(1, func(collection contracts.Collection[*contracts.Fields], page int) (any, contracts.Exception) {
		var p, _ = collection.First()
		result := *p
		switch page {
		case 2:
			assert.True(t, result["name"] == "testing")
		case 1:
			assert.True(t, result["name"] == "testing2")
		}

		return result["id"], nil
	})
	assert.NoError(t, err, err)

	assert.NoError(t, UserQuery(tableName).InsertE(contracts.Fields{"name": "testing3", "age": 18}), err)

	id, err := UserQuery(tableName).InsertGetIdE(contracts.Fields{"name": "testing4", "age": 18})
	assert.NoError(t, err, err)
	assert.True(t, id == int64(user.Id+2))

	num, err := UserQuery(tableName).InsertOrIgnoreE(contracts.Fields{"name": "testing5", "age": 18})
	assert.NoError(t, err, err)
	assert.True(t, num > 0)

	num, err = UserQuery(tableName).InsertOrReplaceE(contracts.Fields{
		"name": "testing6", "age": 18, "id": user.Id,
	})
	assert.NoError(t, err, err)
	assert.True(t, num > 0)
	user = UserQuery(tableName).Find(user.Id)
	assert.NotNil(t, user)
	assert.True(t, user.Name == "testing6")

	num, err = UserQuery(tableName).UpdateE(contracts.Fields{"age": 10})
	assert.NoError(t, err, err)
	assert.True(t, num > 0)
	user = UserQuery(tableName).Find(user.Id)
	assert.NotNil(t, user)
	assert.True(t, user.Age == 10)

	var lastId = user.Id
	num, err = UserQuery(tableName).Where("id", user.Id).DeleteE()
	assert.NoError(t, err, err)
	assert.True(t, num == 1)
	user = UserQuery(tableName).Find(user.Id)
	assert.Nil(t, user)

	err = UserQuery(tableName).UpdateOrInsertE(contracts.Fields{
		"id": lastId,
	}, contracts.Fields{
		"name": "testing6",
		"age":  8,
	})
	assert.NoError(t, err, err)
	user = UserQuery(tableName).Find(lastId)
	assert.NotNil(t, user)
	assert.True(t, user.Id == lastId)
	assert.True(t, user.Age == 8)
	assert.True(t, user.Name == "testing6")

	user, err = UserQuery(tableName).UpdateOrCreateE(contracts.Fields{
		"id": lastId,
	}, contracts.Fields{
		"name": "testing6",
		"age":  18,
	})
	assert.NoError(t, err, err)
	assert.NotNil(t, user)
	assert.True(t, user.Id == lastId)
	assert.True(t, user.Age == 18)
	assert.True(t, user.Name == "testing6")

	users, err := UserQuery(tableName).SelectForUpdateE()
	assert.NoError(t, err, err)
	assert.NotNil(t, users)
	lastUser, _ := users.Last()
	assert.True(t, lastUser.Age == 10)

	assert.True(t, UserQuery(tableName).Where("id", 0).FirstOr(func() *User {
		return user
	}).Name == "testing6")

	assert.Error(t, utils.NoPanic(func() {
		assert.True(t, UserQuery(tableName).Find(user.Id).Name == user.Name)
		assert.True(t, UserQuery(tableName).FirstWhere("id", user.Id).Name == user.Name)
		tmpUser, tmpErr := UserQuery(tableName).FirstWhereE("id", user.Id)
		assert.NoError(t, tmpErr, tmpErr)
		assert.True(t, tmpUser.Name == user.Name)

		UserQuery(tableName).Where("id", 0).FirstOrFail()
	}))

	assert.True(t, table.ArrayQuery(tableName).Count() == 5)
	list, total := UserQuery(tableName).Paginate(2, 1)
	assert.True(t, table.ArrayQuery(tableName).Count() == total)
	assert.True(t, list.Len() == 2)
	assert.True(t, UserQuery(tableName).SimplePaginate(2, 3).Len() == 1)

	table.ArrayQuery(tableName).Delete()
	assert.True(t, table.ArrayQuery(tableName).Count() == 0)

}
