import pytest

from goblet import Goblet
from goblet.backends.backend import Backend
from goblet.backends import CloudFunctionV1, CloudFunctionV2, CloudRun
from goblet.errors import GobletValidationError


class TestBackend:
    def test_custom_files(self, monkeypatch):
        monkeypatch.setenv("G_HTTP_TEST", "REPLAY")

        test_custom_files = {
            "custom_files": {"include": ["*.yaml"], "exclude": [".secret"]}
        }
        goblet = Goblet(config=test_custom_files)
        backend = Backend(goblet, None, None)

        assert "*.yaml" in backend.zip_config["include"]
        assert "*.py" in backend.zip_config["include"]
        assert ".secret" in backend.zip_config["exclude"]
        assert "build" in backend.zip_config["exclude"]

    def test_get_env_cloudfunction_v1(self):
        test_env = {"cloudfunction": {"environmentVariables": {"TEST": "VALUE"}}}
        goblet = Goblet(config=test_env)
        backend = CloudFunctionV1(goblet)

        assert backend.get_environment_vars() == {"TEST": "VALUE"}

    def test_get_env_cloudfunction_v2(self):
        test_env = {
            "cloudfunction_v2": {
                "serviceConfig": {"environmentVariables": {"TEST": "VALUE"}}
            }
        }
        goblet = Goblet(config=test_env)
        backend = CloudFunctionV2(goblet)

        assert backend.get_environment_vars() == {"TEST": "VALUE"}

    def test_get_env_cloudrun(self):
        test_env = {"cloudrun_container": {"env": [{"name": "TEST", "value": "VALUE"}]}}
        goblet = Goblet(config=test_env)
        backend = CloudRun(goblet)

        assert backend.get_environment_vars() == {"TEST": "VALUE"}

    def test_get_env_cloudrun_secret(self, monkeypatch):
        monkeypatch.setenv("GOOGLE_CLOUD_PROJECT", "goblet")
        monkeypatch.setenv("GOOGLE_LOCATION", "us-central1")
        monkeypatch.setenv("G_TEST_NAME", "get-secrets")
        monkeypatch.setenv("G_HTTP_TEST", "REPLAY")

        test_env = {
            "cloudrun_container": {
                "env": [
                    {
                        "name": "TESTSECRET",
                        "valueSource": {
                            "secretKeyRef": {"secret": "TESTSECRET", "version": "1"}
                        },
                    }
                ]
            }
        }
        goblet = Goblet(config=test_env)
        backend = CloudRun(goblet)

        assert backend.get_environment_vars() == {"TESTSECRET": "testtesttest"}

    def test_get_env_cloudfunction_secret(self, monkeypatch):
        monkeypatch.setenv("GOOGLE_CLOUD_PROJECT", "goblet")
        monkeypatch.setenv("GOOGLE_LOCATION", "us-central1")
        monkeypatch.setenv("G_TEST_NAME", "get-secrets")
        monkeypatch.setenv("G_HTTP_TEST", "REPLAY")

        test_env = {
            "cloudfunction": {
                "secretEnvironmentVariables": [
                    {"key": "TESTSECRET", "secret": "TESTSECRET", "version": "1"},
                ]
            }
        }
        goblet = Goblet(config=test_env)
        backend = CloudFunctionV1(goblet)

        assert backend.get_environment_vars() == {"TESTSECRET": "testtesttest"}

    def test_cloudrun_valid_name(self):
        with pytest.raises(GobletValidationError):
            CloudRun(Goblet(function_name="INVALID"))

        with pytest.raises(GobletValidationError):
            CloudRun(Goblet(function_name="in_valid"))

        CloudRun(Goblet(function_name="valid"))

    def test_cloudfunctionv2_valid_name(self):
        with pytest.raises(GobletValidationError):
            CloudFunctionV2(Goblet(function_name="INVALID"))

        with pytest.raises(GobletValidationError):
            CloudFunctionV2(Goblet(function_name="in_valid"))

        CloudFunctionV2(Goblet(function_name="valid"))
