module commandr.completion.bash;

import commandr.program;
import commandr.option;
import std.algorithm : map, filter;
import std.array : Appender, join;
import std.string : format;
import std.range : empty, chain;


/**
 * Creates bash completion script.
 *
 * Creates completion script for specified program, returning the script contents.
 * You need to create it once, and save the script in directory like
 * `/etc/bash_completion.d/` during installation.
 *
 * Params:
 *   program - Program to create completion script.
 *
 * Returns:
 *   Generated completion script contents.
 *
 * Examples:
 * ---
 * import std.file : write;
 * import std.string : format;
 * import commandr;
 *
 * auto prog = new Program("test");
 * std.file.write("%s.bash".format(prog.binaryName), createBashCompletionScript(prog));
 * ---
 */
string createBashCompletionScript(Program program) {
    Appender!string builder;

    builder ~= "#!/usr/bin/env bash\n";
    builder ~= "# This file is autogenerated. DO NOT EDIT.\n";

    builder ~= `
__get_args() {
    local max opts args name arg i count
    max=$1; shift
    opts=$1; shift
    args=($@)
    let i=0
    let count=0

    while [ $i -le ${#args[@]} ]; do
        arg=${args[i]}
        name="${arg%=*}"

        if [[ $name = -* ]]; then
            if [[ " $opts " = *"$name"* ]]; then
                if ! [[ $name = *"="* ]]; then
                    let i+=1
                fi
            fi
        else
            let count+=1
            echo $arg
        fi
        let i+=1

        [ $count -ge $max ] && break
    done

    return $i
}

__function_exists() {
    declare -f -F $1 > /dev/null
    return $?
}

`;
    // works by creating completion functions for commands recursively
    completionFunc(program, builder);

    builder ~= "complete -F _%s_completion_main %s\n".format(program.binaryName, program.binaryName);

    return builder.data;
}

private void completionFunc(Command command, Appender!string builder) {
    foreach(command; command.commands) {
        completionFunc(command, builder);
    }

    auto argumentCount = command.arguments.length;
    auto commands = command.commands.keys.join(" ");

    auto shorts = command.abbrevations.map!(s => "-" ~ s);
    auto longs = command.fullNames.map!(l => "--" ~ l);

    auto options = command.options
        .map!(o => [o.abbrev ? "-" ~ o.abbrev : null, o.full ? "--" ~ o.full : null])
        .join().filter!`a && a.length`.join(" ");

    builder ~= "_%s_completion() {\n".format(command.chain.join("_"));
    builder ~= "    local args target\n\n";

    builder ~= "    __args=( $(__get_args %s \"%s\" \"${COMP_WORDS[@]:__args_start}\") )\n"
                    .format(argumentCount + command.commands.length ? 1 : 0, options);
    builder ~= "    args=$?\n\n";

    builder ~= "    if [ $COMP_CWORD -lt $(( $__args_start + $args )) ]; then\n";
    builder ~= "        if [[ \"$curr\" = -* ]]; then\n";
    builder ~= "            COMPREPLY=( $(compgen -W \"%s\" -- \"$curr\") )\n".format(chain(shorts, longs).join(" "));

    if (command.commands.length > 0) {
        builder ~= "        elif [ ${#__args[@]} -ge %s ]; then\n".format(command.arguments.length);
        builder ~= "            COMPREPLY=( $(compgen -W \"%s\" -- \"$curr\") )\n".format(commands);
        builder ~= "        fi\n";

        builder ~= "    elif [ ${#__args[@]} -gt 0 ] && [[ \" %s \" = *\" ${__args[@]: -1:1} \"* ]]; then\n".format(commands);
        builder ~= "        target=\"_%s_${__args[@]: -1:1}_completion\"\n".format(command.chain.join("_"));
        builder ~= "        let __args_start+=$args\n";
        builder ~= "        __function_exists $target && $target\n";
    }
    else {
        builder ~= "        fi\n";
    }
    builder ~= "    fi\n";
    builder ~= "}\n\n";
}

private void completionFunc(Program program, Appender!string builder) {
    foreach(command; program.commands) {
        completionFunc(command, builder);
    }

    completionFunc(cast(Command)program, builder);

    builder ~= "_%s_completion_main() {\n".format(program.binaryName);
    builder ~= "    COMPREPLY=()\n";
    builder ~= "    __args_start=1\n";
    builder ~= "    curr=${COMP_WORDS[COMP_CWORD]}\n";
    builder ~= "    prev=${COMP_WORDS[COMP_CWORD-1]}\n\n";
    builder ~= "    _%s_completion\n".format(program.binaryName);

    builder ~= "    unset __args __args_start curr prev\n";
    builder ~= "}\n\n";
}
