/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */

// Code generated by the Google Gen AI SDK generator DO NOT EDIT.

import {ApiClient} from './_api_client.js';
import * as common from './_common.js';
import {BaseModule} from './_common.js';
import * as converters from './converters/_batches_converters.js';
import {PagedItem, Pager} from './pagers.js';
import * as types from './types.js';

export class Batches extends BaseModule {
  constructor(private readonly apiClient: ApiClient) {
    super();
  }

  /**
   * Create batch job.
   *
   * @param params - The parameters for create batch job request.
   * @return The created batch job.
   *
   * @example
   * ```ts
   * const response = await ai.batches.create({
   *   model: 'gemini-2.0-flash',
   *   src: {gcsUri: 'gs://bucket/path/to/file.jsonl', format: 'jsonl'},
   *   config: {
   *     dest: {gcsUri: 'gs://bucket/path/output/directory', format: 'jsonl'},
   *   }
   * });
   * console.log(response);
   * ```
   */
  create = async (
    params: types.CreateBatchJobParameters,
  ): Promise<types.BatchJob> => {
    if (this.apiClient.isVertexAI()) {
      // Format destination if not provided
      // Cast params.src as Vertex AI path does not handle InlinedRequest[]
      params.config = this.formatDestination(
        params.src as string | types.BatchJobSource,
        params.config,
      );
    }
    return this.createInternal(params);
  };

  /**
   * **Experimental** Creates an embedding batch job.
   *
   * @param params - The parameters for create embedding batch job request.
   * @return The created batch job.
   *
   * @example
   * ```ts
   * const response = await ai.batches.createEmbeddings({
   *   model: 'text-embedding-004',
   *   src: {fileName: 'files/my_embedding_input'},
   * });
   * console.log(response);
   * ```
   */
  createEmbeddings = async (
    params: types.CreateEmbeddingsBatchJobParameters,
  ): Promise<types.BatchJob> => {
    console.warn(
      'batches.createEmbeddings() is experimental and may change without notice.',
    );

    if (this.apiClient.isVertexAI()) {
      throw new Error('Vertex AI does not support batches.createEmbeddings.');
    }

    // MLDEV
    const src = params.src as types.EmbeddingsBatchJobSource;
    const is_inlined = src.inlinedRequests !== undefined;

    if (!is_inlined) {
      return this.createEmbeddingsInternal(params); // Fixed typo here
    }

    // Inlined embed content requests handling
    const result = this.createInlinedEmbedContentRequest(params);
    const path = result.path;
    const requestBody = result.body;
    const queryParams =
      (converters.createEmbeddingsBatchJobParametersToMldev(
        this.apiClient,
        params,
      )['_query'] as Record<string, string>) || {};

    const response = this.apiClient
      .request({
        path: path,
        queryParams: queryParams,
        body: JSON.stringify(requestBody),
        httpMethod: 'POST',
        httpOptions: params.config?.httpOptions,
        abortSignal: params.config?.abortSignal,
      })
      .then((httpResponse) => {
        return httpResponse.json();
      }) as Promise<types.BatchJob>;

    return response.then((apiResponse) => {
      const resp = converters.batchJobFromMldev(apiResponse);
      return resp as types.BatchJob;
    });
  };

  /**
   * Lists batch job configurations.
   *
   * @param params - The parameters for the list request.
   * @return The paginated results of the list of batch jobs.
   *
   * @example
   * ```ts
   * const batchJobs = await ai.batches.list({config: {'pageSize': 2}});
   * for await (const batchJob of batchJobs) {
   *   console.log(batchJob);
   * }
   * ```
   */
  list = async (
    params: types.ListBatchJobsParameters = {},
  ): Promise<Pager<types.BatchJob>> => {
    return new Pager<types.BatchJob>(
      PagedItem.PAGED_ITEM_BATCH_JOBS,
      (x: types.ListBatchJobsParameters) => this.listInternal(x),
      await this.listInternal(params),
      params,
    );
  };

  // Helper function to handle inlined generate content requests
  private createInlinedGenerateContentRequest(
    params: types.CreateBatchJobParameters,
  ): {path: string; body: Record<string, unknown>} {
    const body = converters.createBatchJobParametersToMldev(
      this.apiClient, // Use instance apiClient
      params,
    );

    const urlParams = body['_url'] as Record<string, unknown>;
    const path = common.formatMap('{model}:batchGenerateContent', urlParams);

    const batch = body['batch'] as {[key: string]: unknown};
    const inputConfig = batch['inputConfig'] as {[key: string]: unknown};
    const requestsWrapper = inputConfig['requests'] as {
      [key: string]: unknown;
    };
    const requests = requestsWrapper['requests'] as Array<{
      [key: string]: unknown;
    }>;
    const newRequests = [];

    for (const request of requests) {
      const requestDict = {...request}; // Clone
      if (requestDict['systemInstruction']) {
        const systemInstructionValue = requestDict['systemInstruction'];
        delete requestDict['systemInstruction'];
        const requestContent = requestDict['request'] as {
          [key: string]: unknown;
        };
        requestContent['systemInstruction'] = systemInstructionValue;
        requestDict['request'] = requestContent;
      }
      newRequests.push(requestDict);
    }
    requestsWrapper['requests'] = newRequests;

    delete body['config'];
    delete body['_url'];
    delete body['_query'];

    return {path, body};
  }

  // Helper function to handle inlined embedding requests
  private createInlinedEmbedContentRequest(
    params: types.CreateEmbeddingsBatchJobParameters,
  ): {path: string; body: Record<string, unknown>} {
    const body = converters.createEmbeddingsBatchJobParametersToMldev(
      this.apiClient, // Use instance apiClient
      params,
    );

    const urlParams = body['_url'] as Record<string, unknown>;
    const path = common.formatMap('{model}:asyncBatchEmbedContent', urlParams);

    const batch = body['batch'] as {[key: string]: unknown};
    const inputConfig = batch['inputConfig'] as {[key: string]: unknown};
    const requestsWrapper = inputConfig['requests'] as {
      [key: string]: unknown;
    };
    const requests = requestsWrapper['requests'] as Array<{
      [key: string]: unknown;
    }>;
    const newRequests = [];

    delete requestsWrapper['config']; // Remove top-level config

    for (const request of requests) {
      const requestDict = {...request}; // Clone
      const innerRequest = requestDict['request'] as {
        [key: string]: unknown;
      };
      for (const key in requestDict) {
        if (key !== 'request') {
          innerRequest[key] = requestDict[key];
          delete requestDict[key];
        }
      }
      newRequests.push(requestDict);
    }
    requestsWrapper['requests'] = newRequests;

    delete body['config'];
    delete body['_url'];
    delete body['_query'];

    return {path, body};
  }

  // Helper function to get the first GCS URI
  private getGcsUri(src: string | types.BatchJobSource): string | undefined {
    if (typeof src === 'string') {
      return src.startsWith('gs://') ? src : undefined;
    }
    if (!Array.isArray(src) && src.gcsUri && src.gcsUri.length > 0) {
      return src.gcsUri[0];
    }
    return undefined;
  }

  // Helper function to get the BigQuery URI
  private getBigqueryUri(
    src: string | types.BatchJobSource,
  ): string | undefined {
    if (typeof src === 'string') {
      return src.startsWith('bq://') ? src : undefined;
    }
    if (!Array.isArray(src)) {
      return src.bigqueryUri;
    }
    return undefined;
  }

  // Function to format the destination configuration for Vertex AI
  private formatDestination(
    src: string | types.BatchJobSource,
    config?: types.CreateBatchJobConfig,
  ): types.CreateBatchJobConfig {
    const newConfig = config ? {...config} : {};

    const timestampStr = Date.now().toString();

    if (!newConfig.displayName) {
      newConfig.displayName = `genaiBatchJob_${timestampStr}`;
    }

    if (newConfig.dest === undefined) {
      const gcsUri = this.getGcsUri(src);
      const bigqueryUri = this.getBigqueryUri(src);

      if (gcsUri) {
        if (gcsUri.endsWith('.jsonl')) {
          // For .jsonl files, remove suffix and add /dest
          newConfig.dest = `${gcsUri.slice(0, -6)}/dest`;
        } else {
          // Fallback for other GCS URIs
          newConfig.dest = `${gcsUri}_dest_${timestampStr}`;
        }
      } else if (bigqueryUri) {
        newConfig.dest = `${bigqueryUri}_dest_${timestampStr}`;
      } else {
        throw new Error(
          'Unsupported source for Vertex AI: No GCS or BigQuery URI found.',
        );
      }
    }
    return newConfig;
  }

  /**
   * Internal method to create batch job.
   *
   * @param params - The parameters for create batch job request.
   * @return The created batch job.
   *
   */
  private async createInternal(
    params: types.CreateBatchJobParameters,
  ): Promise<types.BatchJob> {
    let response: Promise<types.BatchJob>;

    let path: string = '';
    let queryParams: Record<string, string> = {};
    if (this.apiClient.isVertexAI()) {
      const body = converters.createBatchJobParametersToVertex(
        this.apiClient,
        params,
      );
      path = common.formatMap(
        'batchPredictionJobs',
        body['_url'] as Record<string, unknown>,
      );
      queryParams = body['_query'] as Record<string, string>;
      delete body['_url'];
      delete body['_query'];

      response = this.apiClient
        .request({
          path: path,
          queryParams: queryParams,
          body: JSON.stringify(body),
          httpMethod: 'POST',
          httpOptions: params.config?.httpOptions,
          abortSignal: params.config?.abortSignal,
        })
        .then((httpResponse) => {
          return httpResponse.json();
        }) as Promise<types.BatchJob>;

      return response.then((apiResponse) => {
        const resp = converters.batchJobFromVertex(apiResponse);

        return resp as types.BatchJob;
      });
    } else {
      const body = converters.createBatchJobParametersToMldev(
        this.apiClient,
        params,
      );
      path = common.formatMap(
        '{model}:batchGenerateContent',
        body['_url'] as Record<string, unknown>,
      );
      queryParams = body['_query'] as Record<string, string>;
      delete body['_url'];
      delete body['_query'];

      response = this.apiClient
        .request({
          path: path,
          queryParams: queryParams,
          body: JSON.stringify(body),
          httpMethod: 'POST',
          httpOptions: params.config?.httpOptions,
          abortSignal: params.config?.abortSignal,
        })
        .then((httpResponse) => {
          return httpResponse.json();
        }) as Promise<types.BatchJob>;

      return response.then((apiResponse) => {
        const resp = converters.batchJobFromMldev(apiResponse);

        return resp as types.BatchJob;
      });
    }
  }

  /**
   * Internal method to create batch job.
   *
   * @param params - The parameters for create batch job request.
   * @return The created batch job.
   *
   */
  private async createEmbeddingsInternal(
    params: types.CreateEmbeddingsBatchJobParameters,
  ): Promise<types.BatchJob> {
    let response: Promise<types.BatchJob>;

    let path: string = '';
    let queryParams: Record<string, string> = {};
    if (this.apiClient.isVertexAI()) {
      throw new Error(
        'This method is only supported by the Gemini Developer API.',
      );
    } else {
      const body = converters.createEmbeddingsBatchJobParametersToMldev(
        this.apiClient,
        params,
      );
      path = common.formatMap(
        '{model}:asyncBatchEmbedContent',
        body['_url'] as Record<string, unknown>,
      );
      queryParams = body['_query'] as Record<string, string>;
      delete body['_url'];
      delete body['_query'];

      response = this.apiClient
        .request({
          path: path,
          queryParams: queryParams,
          body: JSON.stringify(body),
          httpMethod: 'POST',
          httpOptions: params.config?.httpOptions,
          abortSignal: params.config?.abortSignal,
        })
        .then((httpResponse) => {
          return httpResponse.json();
        }) as Promise<types.BatchJob>;

      return response.then((apiResponse) => {
        const resp = converters.batchJobFromMldev(apiResponse);

        return resp as types.BatchJob;
      });
    }
  }

  /**
   * Gets batch job configurations.
   *
   * @param params - The parameters for the get request.
   * @return The batch job.
   *
   * @example
   * ```ts
   * await ai.batches.get({name: '...'}); // The server-generated resource name.
   * ```
   */
  async get(params: types.GetBatchJobParameters): Promise<types.BatchJob> {
    let response: Promise<types.BatchJob>;

    let path: string = '';
    let queryParams: Record<string, string> = {};
    if (this.apiClient.isVertexAI()) {
      const body = converters.getBatchJobParametersToVertex(
        this.apiClient,
        params,
      );
      path = common.formatMap(
        'batchPredictionJobs/{name}',
        body['_url'] as Record<string, unknown>,
      );
      queryParams = body['_query'] as Record<string, string>;
      delete body['_url'];
      delete body['_query'];

      response = this.apiClient
        .request({
          path: path,
          queryParams: queryParams,
          body: JSON.stringify(body),
          httpMethod: 'GET',
          httpOptions: params.config?.httpOptions,
          abortSignal: params.config?.abortSignal,
        })
        .then((httpResponse) => {
          return httpResponse.json();
        }) as Promise<types.BatchJob>;

      return response.then((apiResponse) => {
        const resp = converters.batchJobFromVertex(apiResponse);

        return resp as types.BatchJob;
      });
    } else {
      const body = converters.getBatchJobParametersToMldev(
        this.apiClient,
        params,
      );
      path = common.formatMap(
        'batches/{name}',
        body['_url'] as Record<string, unknown>,
      );
      queryParams = body['_query'] as Record<string, string>;
      delete body['_url'];
      delete body['_query'];

      response = this.apiClient
        .request({
          path: path,
          queryParams: queryParams,
          body: JSON.stringify(body),
          httpMethod: 'GET',
          httpOptions: params.config?.httpOptions,
          abortSignal: params.config?.abortSignal,
        })
        .then((httpResponse) => {
          return httpResponse.json();
        }) as Promise<types.BatchJob>;

      return response.then((apiResponse) => {
        const resp = converters.batchJobFromMldev(apiResponse);

        return resp as types.BatchJob;
      });
    }
  }

  /**
   * Cancels a batch job.
   *
   * @param params - The parameters for the cancel request.
   * @return The empty response returned by the API.
   *
   * @example
   * ```ts
   * await ai.batches.cancel({name: '...'}); // The server-generated resource name.
   * ```
   */
  async cancel(params: types.CancelBatchJobParameters): Promise<void> {
    let path: string = '';
    let queryParams: Record<string, string> = {};
    if (this.apiClient.isVertexAI()) {
      const body = converters.cancelBatchJobParametersToVertex(
        this.apiClient,
        params,
      );
      path = common.formatMap(
        'batchPredictionJobs/{name}:cancel',
        body['_url'] as Record<string, unknown>,
      );
      queryParams = body['_query'] as Record<string, string>;
      delete body['_url'];
      delete body['_query'];

      await this.apiClient.request({
        path: path,
        queryParams: queryParams,
        body: JSON.stringify(body),
        httpMethod: 'POST',
        httpOptions: params.config?.httpOptions,
        abortSignal: params.config?.abortSignal,
      });
    } else {
      const body = converters.cancelBatchJobParametersToMldev(
        this.apiClient,
        params,
      );
      path = common.formatMap(
        'batches/{name}:cancel',
        body['_url'] as Record<string, unknown>,
      );
      queryParams = body['_query'] as Record<string, string>;
      delete body['_url'];
      delete body['_query'];

      await this.apiClient.request({
        path: path,
        queryParams: queryParams,
        body: JSON.stringify(body),
        httpMethod: 'POST',
        httpOptions: params.config?.httpOptions,
        abortSignal: params.config?.abortSignal,
      });
    }
  }

  private async listInternal(
    params: types.ListBatchJobsParameters,
  ): Promise<types.ListBatchJobsResponse> {
    let response: Promise<types.ListBatchJobsResponse>;

    let path: string = '';
    let queryParams: Record<string, string> = {};
    if (this.apiClient.isVertexAI()) {
      const body = converters.listBatchJobsParametersToVertex(params);
      path = common.formatMap(
        'batchPredictionJobs',
        body['_url'] as Record<string, unknown>,
      );
      queryParams = body['_query'] as Record<string, string>;
      delete body['_url'];
      delete body['_query'];

      response = this.apiClient
        .request({
          path: path,
          queryParams: queryParams,
          body: JSON.stringify(body),
          httpMethod: 'GET',
          httpOptions: params.config?.httpOptions,
          abortSignal: params.config?.abortSignal,
        })
        .then((httpResponse) => {
          return httpResponse.json().then((jsonResponse) => {
            const response = jsonResponse as types.ListBatchJobsResponse;
            response.sdkHttpResponse = {
              headers: httpResponse.headers,
            } as types.HttpResponse;
            return response;
          });
        }) as Promise<types.ListBatchJobsResponse>;

      return response.then((apiResponse) => {
        const resp = converters.listBatchJobsResponseFromVertex(apiResponse);
        const typedResp = new types.ListBatchJobsResponse();
        Object.assign(typedResp, resp);
        return typedResp;
      });
    } else {
      const body = converters.listBatchJobsParametersToMldev(params);
      path = common.formatMap(
        'batches',
        body['_url'] as Record<string, unknown>,
      );
      queryParams = body['_query'] as Record<string, string>;
      delete body['_url'];
      delete body['_query'];

      response = this.apiClient
        .request({
          path: path,
          queryParams: queryParams,
          body: JSON.stringify(body),
          httpMethod: 'GET',
          httpOptions: params.config?.httpOptions,
          abortSignal: params.config?.abortSignal,
        })
        .then((httpResponse) => {
          return httpResponse.json().then((jsonResponse) => {
            const response = jsonResponse as types.ListBatchJobsResponse;
            response.sdkHttpResponse = {
              headers: httpResponse.headers,
            } as types.HttpResponse;
            return response;
          });
        }) as Promise<types.ListBatchJobsResponse>;

      return response.then((apiResponse) => {
        const resp = converters.listBatchJobsResponseFromMldev(apiResponse);
        const typedResp = new types.ListBatchJobsResponse();
        Object.assign(typedResp, resp);
        return typedResp;
      });
    }
  }

  /**
   * Deletes a batch job.
   *
   * @param params - The parameters for the delete request.
   * @return The empty response returned by the API.
   *
   * @example
   * ```ts
   * await ai.batches.delete({name: '...'}); // The server-generated resource name.
   * ```
   */
  async delete(
    params: types.DeleteBatchJobParameters,
  ): Promise<types.DeleteResourceJob> {
    let response: Promise<types.DeleteResourceJob>;

    let path: string = '';
    let queryParams: Record<string, string> = {};
    if (this.apiClient.isVertexAI()) {
      const body = converters.deleteBatchJobParametersToVertex(
        this.apiClient,
        params,
      );
      path = common.formatMap(
        'batchPredictionJobs/{name}',
        body['_url'] as Record<string, unknown>,
      );
      queryParams = body['_query'] as Record<string, string>;
      delete body['_url'];
      delete body['_query'];

      response = this.apiClient
        .request({
          path: path,
          queryParams: queryParams,
          body: JSON.stringify(body),
          httpMethod: 'DELETE',
          httpOptions: params.config?.httpOptions,
          abortSignal: params.config?.abortSignal,
        })
        .then((httpResponse) => {
          return httpResponse.json().then((jsonResponse) => {
            const response = jsonResponse as types.DeleteResourceJob;
            response.sdkHttpResponse = {
              headers: httpResponse.headers,
            } as types.HttpResponse;
            return response;
          });
        }) as Promise<types.DeleteResourceJob>;

      return response.then((apiResponse) => {
        const resp = converters.deleteResourceJobFromVertex(apiResponse);

        return resp as types.DeleteResourceJob;
      });
    } else {
      const body = converters.deleteBatchJobParametersToMldev(
        this.apiClient,
        params,
      );
      path = common.formatMap(
        'batches/{name}',
        body['_url'] as Record<string, unknown>,
      );
      queryParams = body['_query'] as Record<string, string>;
      delete body['_url'];
      delete body['_query'];

      response = this.apiClient
        .request({
          path: path,
          queryParams: queryParams,
          body: JSON.stringify(body),
          httpMethod: 'DELETE',
          httpOptions: params.config?.httpOptions,
          abortSignal: params.config?.abortSignal,
        })
        .then((httpResponse) => {
          return httpResponse.json().then((jsonResponse) => {
            const response = jsonResponse as types.DeleteResourceJob;
            response.sdkHttpResponse = {
              headers: httpResponse.headers,
            } as types.HttpResponse;
            return response;
          });
        }) as Promise<types.DeleteResourceJob>;

      return response.then((apiResponse) => {
        const resp = converters.deleteResourceJobFromMldev(apiResponse);

        return resp as types.DeleteResourceJob;
      });
    }
  }
}
