/*
 *  BEGIN_COPYRIGHT
 *
 *  Copyright (C) 2011-2013 deCODE genetics Inc.
 *  Copyright (C) 2013-2019 WuXi NextCode Inc.
 *  All Rights Reserved.
 *
 *  GORpipe is free software: you can redistribute it and/or modify
 *  it under the terms of the AFFERO GNU General Public License as published by
 *  the Free Software Foundation.
 *
 *  GORpipe is distributed "AS-IS" AND WITHOUT ANY WARRANTY OF ANY KIND,
 *  INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 *  NON-INFRINGEMENT, OR FITNESS FOR A PARTICULAR PURPOSE. See
 *  the AFFERO GNU General Public License for the complete license terms.
 *
 *  You should have received a copy of the AFFERO GNU General Public License
 *  along with GORpipe.  If not, see <http://www.gnu.org/licenses/agpl-3.0.html>
 *
 *  END_COPYRIGHT
 */

package gorsat.Regression;

import org.junit.Assert;
import org.junit.Test;

import java.util.Arrays;

public class UTestLinearRegression {

    @Test
    public void testConstructor_IllegalArguments() {
        boolean success = false;
        try {
            new LinearRegression(1, 1);
        } catch (IllegalArgumentException e) {
            success = true;
        }
        Assert.assertTrue(success);
    }

    @Test
    public void testSetData_IllegalArguments() {
        boolean success = false;
        final LinearRegression lr = new LinearRegression(10, 1);
        try {
            lr.setData(new double[1][10], 1);
        } catch (IllegalArgumentException e) {
            success = true;
        }
        Assert.assertTrue(success);
    }

    @Test
    public void testGeneral_OneCov() {
        final double[][] A = {{-30.3103, 6.9958, -20.0497, 28.6698, -60.8050, -98.4559, 34.2474, 57.5631, -1.5624, 73.1597, -67.1765, 64.5470, -61.6020, -25.2137, 6.4554, 28.0240, 97.2975, -12.5694, 26.7219, -79.2484, 75.2375, -48.9837, -63.2864, -50.1669, -68.8241, -83.1687, -79.2802, 81.1928, 88.0202, -40.1488, 63.6741, -88.1831, 21.6204, -60.7807, -98.8606, 3.0431, 59.4414, -54.8358, 10.6132, 38.1773, 64.6970, 20.6372, 44.9360, 17.7211, 31.7272, -4.0502, 58.5818, -44.5098, 68.3983, -75.3068, 27.6143, -11.0951, 91.3091, 59.2206, -85.9124, 66.7348, 9.5832, -46.0262, 75.6757, 27.5777, 23.0289, -65.0646, 71.3081, -67.6451, -89.5520, 23.4875, -36.4028, -35.8959, 33.8754, 52.9585, 68.8349, 58.3504, 34.5287, -33.8607, -44.6583, 77.3377, 46.2523, -67.9502, -59.0506, -8.5992, -97.7659, -29.7628, -9.6016, -19.2399, 60.7775, 20.8871, -36.5701, 21.0198, 25.9107, 49.0292, 14.3564, -34.3490, 88.6126, 21.1183, -6.1106, -11.2591, 64.4386, 42.2721, -45.9925, -98.7104}};
        final double[] y = {76.8574, 42.6630, 25.1124, -85.1546, -57.0220, -12.7491, -84.1371, 36.8249, -83.8078, 17.8130, -2.6491, -45.5213, -37.3673, 12.0057, 28.0982, 77.8639, 14.0646, 4.8673, 72.0206, 42.3451, 82.0800, -86.7056, 40.6571, 83.0763, -1.1311, 9.1638, 9.2389, -38.3547, 91.5950, -39.3690, 38.6305, 89.4087, -52.5910, -88.8648, 85.1731, -57.9493, 76.1301, -84.3559, 30.4624, -90.1041, -72.3717, -96.3319, -86.9084, -21.4919, -29.0641, -11.4236, 94.3912, 84.1061, -7.8774, -7.2554, -59.8037, 38.3265, 17.9638, -91.9329, 50.6729, -33.1321, -77.5671, 61.3639, -79.7645, 56.5437, 23.0504, 4.2139, -40.4151, 55.2784, 24.1197, -3.2542, 61.8996, 19.5757, -86.1183, 4.9174, -54.7885, 53.9923, 72.6189, -19.1846, -47.0038, 37.1462, 98.7403, 45.5061, 45.0107, -21.2073, -90.1212, 43.9689, -39.5918, -97.5020, -73.9677, -8.3568, -14.8378, -1.5573, -8.2216, -17.8793, 5.8315, 40.8854, -55.4035, -24.4860, -18.3041, -85.1814, -13.6736, -62.9680, 17.2955, -53.6523};
        final double[] beta = {-4.0971, -0.1036};
        final double[] se = {5.5843, 0.1018};
        final double[] tStats = {-0.734, -1.018};
        final double[] pValues = {0.465, 0.311};
        runGeneralTest(A, y, beta, se, tStats, pValues, y.length, A.length);
    }

    @Test
    public void testGeneral_TwoCovs() {
        final double[][] A = {{-46.6744, 45.2952, -95.6652, -77.3575, 36.9821, 6.7644, -60.2550, 44.5283, -81.4679, -81.6511, 92.3519, -68.8381, 30.7701, -85.5692, 33.7216, 27.0140, -23.5361, 2.4599, 97.0458, 40.5264, 93.9294, 2.1654, 90.0472, 17.0386, 78.6618, -55.6845, -72.6189, 20.9531, -79.6295, 1.1136, -34.5899, -65.5164, -8.4823, -16.9861, -16.6329, 52.3970, -81.0274, 68.1871, -65.2009, 92.9844, 84.6479, -68.9402, 91.8222, -44.9379, 33.9946, 5.2979, 22.0976, 29.1290, -77.0104, 54.2987, -25.4281, -82.1273, 4.7556, -71.2973, -95.0906, 37.0391, 46.2146, -34.3686, -54.8252, -47.7490, 73.1797, -71.6996, 14.7086, 60.6792, -30.9472, 97.3562, 80.2706, -51.4569, -2.2022, -93.2727, -96.5918, 46.2355, 86.1215, -33.0222, -38.3577, 62.4376, 82.8009, 38.0935, -2.5592, -93.0896, 14.8509, -98.6685, -64.9108, 8.7205, -19.2614, -63.7964, -89.5546, -97.5257, -39.8963, -66.1373, 82.2010, 89.7111, 77.4563, 72.1068, -64.3461, 85.9202, -65.9216, 39.9073, -48.8705, 35.7210},
                {64.7947, -93.3550, -18.6759, 49.8314, -43.5361, 62.2827, 27.9048, 69.0572, -9.2810, -44.0855, 17.1913, -17.7962, -36.7596, 49.3271, 66.4025, 76.5820, -33.5841, -88.4429, 47.1430, 61.1627, -43.4593, 31.3869, 92.6389, 27.9221, -84.5993, -75.7632, 58.6685, 36.1552, -40.7145, 49.6297, 12.4247, -12.4019, 8.0462, 38.4196, -57.5098, 27.5805, 55.1341, -69.2712, 70.8053, -6.4324, 76.6279, -53.9524, 3.9491, 37.3280, -16.9246, -53.7917, 86.2544, 70.6009, -47.4524, 9.9534, -88.7818, -73.4840, -13.3783, 14.5740, 10.3684, 90.1546, 3.0823, 15.6059, 70.1891, -68.4778, 19.9476, -70.1769, 27.9057, -27.1656, -5.0002, -40.2171, 65.8738, 39.4033, 63.3186, -50.0722, 62.0126, -20.4571, 62.3861, 34.7635, 60.1672, -6.2060, -49.8465, -71.1150, -0.5177, 87.4046, 6.0285, 16.6342, 85.8123, -47.6186, -41.9365, 25.9277, 0.8888, 96.5631, 1.6964, -83.6969, 26.4139, 67.0481, 66.2117, 75.4895, -89.5603, -36.0573, 44.6661, 1.8142, -77.6577, -92.2231}};
        final double[] y = {-7.5534, 67.4033, 97.3590, -60.0801, -7.8127, -79.6176, 22.1190, -40.4140, -2.6879, -78.2833, 47.5731, -94.9659, 5.9288, 91.9641, -22.8588, 81.4777, 4.5081, 25.8890, -28.6567, 78.1370, -31.2055, -90.8771, -55.0671, 67.0042, 96.5589, 95.1326, -40.5437, -89.4175, -61.1376, 63.3145, 53.0032, 73.7346, 51.3826, 94.8558, -61.8261, 9.9711, 60.9863, 87.5704, 7.2946, 36.4555, 35.1166, -15.7589, -89.3297, -75.3028, -47.3063, -88.9318, 19.1735, -84.6518, -94.8880, -86.3560, -53.2762, 15.7170, 4.5501, -62.5543, -8.7785, 57.5566, 44.1903, 28.1567, 54.2567, 80.3782, -58.9186, 24.1995, -79.5779, -38.0094, 97.2556, 79.6796, -83.6830, 43.1570, 38.3497, -9.4844, -72.6640, -12.7919, 31.3572, 99.8455, -72.9041, 78.1461, 44.2757, -69.1268, -30.1407, -7.7816, -9.7051, 36.1389, 10.5747, -21.8644, 97.8010, 61.7655, -49.8427, -49.9468, 64.8622, -96.4444, -44.6053, 35.1333, -7.0152, 78.8833, -26.6311, 23.9459, -50.1800, 54.1448, 49.5437, -52.8343};
        final double[] beta = {2.41747, 0.05227, -0.02617};
        final double[] se = {6.10837, 0.09777, 0.11321};
        final double[] tStats = {0.396, 0.535, -0.231};
        final double[] pValues = {0.693, 0.594, 0.818};
        runGeneralTest(A, y, beta, se, tStats, pValues, y.length, A.length);
    }

    @Test
    public void testGeneral_ThreeCovs() {
        final double[][] A = {{-82.1358, -48.4816, -33.3257, -84.9069, -49.4417, 48.4541, -18.8659, -90.2682, -82.6790, 78.8426, -60.0289, -39.8956, -36.0856, -58.5925, 38.2000, 85.7212, -64.6850, -19.2806, -19.8396, 22.3037, 19.5678, 57.8076, 83.1362, -35.3381, -63.8288, -9.6974, -24.4932, -70.8971, -31.0069, 62.1105, -10.5717, 35.3707, -56.2812, -97.3819, -51.9530, 8.1165, 49.8291, 35.4295, 47.2435, -2.0981, 7.1110, -3.3658, -68.0911, 21.7313, 76.6987, -77.3943, 66.7510, -76.8825, 16.6459, -92.6679, -98.3671, -48.3231, 82.0298, 29.2777, -68.5577, 62.2572, -30.4139, -52.2242, -49.5081, 33.5959, 6.8752, -31.0474, 1.4516, -93.4906, 85.9797, 81.1827, -81.1886, 41.6145, 41.7331, 60.0758, -33.1240, 96.5806, -25.5200, -57.0060, 72.2900, 22.9127, 17.3672, -26.8984, -48.9044, -17.7334, 48.1663, -87.0552, 30.3198, 82.2944, 45.6274, 27.5636, 64.0943, 32.0221, -9.6359, 37.1345, -55.6937, 16.7578, 10.2697, -60.8554, 5.7799, 92.5316, 70.4650, 0.0970, -74.0105, 0.0661},
                {31.3376, 52.1027, -80.4362, -92.1691, 16.7790, -15.1194, 84.7903, -64.3778, 12.9331, 40.7426, 9.4941, -20.3217, -97.0872, 20.5923, -15.2088, 86.9253, 76.7444, 7.5532, -46.8772, -92.4185, -98.3632, -53.1915, 53.9159, 62.8880, -31.9672, 26.2228, -4.9754, -36.3792, 11.4413, -42.2558, -50.0465, -20.6163, 10.0508, 59.5885, 80.8614, 58.6833, -41.8756, 39.2043, -12.6532, 99.7842, 15.1753, -22.9810, -18.9851, -27.6539, -69.0751, 10.9667, 85.8250, 89.8422, 46.0404, 21.7970, 64.2199, 32.7585, -31.3447, 19.1650, -30.9636, -77.4255, 26.6700, -71.4973, -78.8033, 46.5253, -4.1657, 78.6715, 13.1815, 33.1446, 3.6108, 90.3962, 38.2840, 78.5894, -91.1803, 65.4798, -54.3545, -76.2544, 57.0649, -23.1999, -78.9282, 82.1149, -88.0587, -40.9126, -6.3687, 18.4830, 62.0231, 0.4830, 68.0341, -21.6306, -96.3663, -65.1207, 82.7924, -31.7425, 99.3969, 35.0415, -5.8381, -13.3927, -87.4310, -93.4901, 85.8653, 5.6478, -82.8477, -17.7163, -95.4041, 3.9523},
                {-15.8717, -10.9293, -89.9296, -91.3049, -57.5497, -98.5881, -21.9788, -75.4596, 95.0492, -82.5026, 96.2091, 6.3689, 9.4024, -8.2049, -90.2207, -91.5513, 76.2446, 94.3900, -54.2690, 58.5777, -21.4812, 84.8183, -2.1155, -39.5657, 17.3594, 67.9331, 18.8792, -72.0421, -4.8942, -68.1992, 19.2032, 51.7613, -44.6523, -32.3708, 73.2544, 97.5749, 80.0014, 71.7271, -67.5679, 36.7206, 43.4178, 29.4940, -54.7392, -91.3812, -76.4793, -18.5574, -15.4560, -84.6747, -30.0222, -28.7485, 81.5509, 48.0998, -70.5988, 75.8492, 67.2327, -53.9940, 27.5809, 17.5299, -79.5287, 86.5427, -91.2518, -41.9000, -76.6413, 8.3062, 31.8708, -24.9438, -76.4922, 65.6834, 23.9797, -80.2051, 3.3943, 15.5774, 90.5266, -42.1478, 56.7305, -63.8369, -17.2275, 91.1762, -28.1309, -22.3253, 22.2085, -37.5707, -77.7605, 45.2244, 82.0067, 41.3232, 63.8148, -86.3481, 25.5030, 92.9839, -60.3900, 74.4061, -98.2886, 28.3484, 91.2114, 14.4569, -2.1316, 36.5930, -22.6625, 27.2390}};
        final double[] y = {30.8188, 30.3928, 45.9660, -75.1835, 92.3729, -83.7946, 47.0600, -25.1211, -12.0502, 29.7694, 10.9030, -47.9335, -90.1465, -32.0252, 64.5724, 70.1173, -69.4071, 47.1145, -24.1731, 0.8715, -41.5948, 71.1035, 90.2712, -63.3684, 81.0612, -87.0206, -93.1989, -84.1971, -2.0488, 9.6834, -8.6692, -88.1841, 40.4527, 70.1234, -71.1994, -14.6706, 80.5788, 96.5180, 13.4665, -0.2847, 27.9082, 19.0213, -22.5296, -0.1755, 14.7967, 43.6302, 57.6923, -45.9660, -30.1308, 13.3714, -99.1170, -40.4250, 79.7934, 82.7706, 7.0785, 11.1429, 39.3352, 5.6030, 11.2959, 32.7880, -10.3056, -64.0754, -96.4902, 73.0876, 45.1223, 42.3848, 86.9103, -36.4007, -92.1796, 35.8206, -25.5573, -59.4041, 71.1204, -89.8680, 23.6278, -69.9191, -89.0549, -3.5620, -90.3363, -3.9141, -34.4487, 30.5147, -63.1909, 52.2490, -96.4472, -66.4856, -11.9529, -63.3261, 80.2263, 5.9654, -86.9686, -49.9284, -17.9036, 35.0541, 54.3126, -42.8181, 59.8042, 15.1870, 11.8464, -48.5232};
        final double[] beta = {-3.04739, 0.13653, 0.16412, 0.03213};
        final double[] se = {5.60806, 0.10193, 0.09848, 0.09317};
        final double[] tStats = {-0.543, 1.339, 1.667, 0.345};
        final double[] pValues = {0.5881, 0.1836, 0.0989, 0.7310};
        runGeneralTest(A, y, beta, se, tStats, pValues, y.length, A.length);
    }

    @Test
    public void testGeneral_FourCovs() {
        final double[][] A = {{-92.2232, -81.7136, 99.4315, -16.8546, -53.1288, 18.2590, 66.9407, 78.1924, 94.3834, 61.2773, -2.3262, 54.1522, -50.4859, 41.9069, -18.5915, 27.1942, 55.3289, 56.3998, 91.7770, 44.0258, -28.5156, -36.6379, 12.1419, -31.5794, -37.8895, -94.2547, 65.8501, 32.4798, 72.2665, -13.0343, -25.6155, 58.8084, -0.6017, 16.0258, -9.4392, -2.5145, 30.4828, -60.1684, -91.1968, 40.6509, 11.0501, -34.5281, 40.6631, -22.2715, 39.0358, 41.0441, -38.8690, 35.7492, -65.5795, -89.3177, 79.9192, 11.9330, -75.1274, -12.2369, 71.7829, 49.6359, 8.1194, -0.1646, -88.9271, -66.0397, -96.4734, -50.3585, 98.1011, 89.7752, -11.6233, 8.6648, 38.5309, -47.4130, -71.4063, -87.0375, 41.8859, 51.6459, 59.1685, 65.6094, 41.4074, -5.1298, -53.7015, 65.9020, -95.1318, -23.3853, -28.7375, -17.4601, -95.4112, 42.0049, 15.3077, -33.6773, -55.2194, -64.4952, -4.6619, -42.5281, -94.5247, -63.4196, -98.0546, 83.5063, -45.3170, 41.7669, -75.6632, 65.6695, -95.9465, -9.8575},
                {-16.2102, 99.5921, -54.2170, -90.2521, -10.3883, 34.2836, -49.6402, 9.5642, -98.0930, 59.9839, -57.9114, 60.9964, 73.1025, -95.9737, 71.1546, -70.1534, 21.9635, -61.1764, 70.8929, 10.4390, 37.3913, -50.9904, 2.9952, -88.4966, -29.1185, -73.6571, 73.9857, 54.3085, -20.2154, 14.9134, 95.3606, 60.3706, -50.6731, -53.4267, 41.2300, 64.0711, -31.5757, 55.1501, 19.9960, -25.2260, -37.3614, 37.8799, 20.5476, 33.5408, -2.8025, 43.6310, 18.6475, -38.6977, 53.8206, 9.8749, 35.6302, -21.6512, 57.1621, 43.9898, -70.7722, -10.7367, -19.9969, 49.3664, 65.2850, 72.9261, -9.1047, -67.0790, -30.6823, 93.9806, 23.7793, -2.0013, -11.6807, -39.8462, 46.1701, -74.7127, -94.0901, -10.9126, 80.8235, 74.2417, -52.1292, 18.8388, 95.3669, 98.6537, 90.3030, 97.9726, 15.1980, 62.3837, 39.0905, 24.1694, -62.9733, -35.0960, 87.0494, 64.5359, -76.2464, 41.1858, 74.0831, 86.3097, -54.0490, -13.4038, 75.8653, 53.4091, 1.0306, -25.1479, 28.0126, 2.2858},
                {49.0101, -18.1646, 98.2792, 41.4417, -11.5909, -82.4622, 38.1783, -12.4486, 73.6864, -46.1783, -44.9287, -58.5102, 4.1113, 26.8136, -79.3812, -52.9908, 84.8961, -17.7767, 8.1361, 75.9355, 34.5599, -4.3822, -43.9993, 95.0795, -25.1162, -21.4923, -35.9439, 79.2564, -93.7319, -59.8967, 52.4850, 42.7084, 82.4413, -0.2109, 64.3838, 52.0482, -59.1528, 51.8132, 76.2787, -44.0908, 18.6745, -28.6195, 95.0787, 19.8100, -10.6269, -79.7121, 65.4731, -10.6899, -9.3661, -39.5687, -11.4654, 20.3554, -17.8049, 14.9612, -39.7004, -87.2199, -29.5346, 29.5592, -21.6462, -38.7449, -33.1571, -19.7473, 34.9604, 97.5397, -92.2626, -50.6204, -82.9764, 38.7175, 56.6877, 47.7620, 66.1906, 78.1632, 32.6003, 98.8409, 16.6519, -40.6865, -36.9859, -80.5287, -76.2934, -43.1222, -52.5652, -67.7291, -77.0811, 55.9524, 33.1588, -74.1378, -82.3010, -33.4334, 54.6452, 42.0414, 82.3992, 39.5131, -32.6274, -87.1456, 88.5244, -41.1183, -81.7462, 95.7557, 13.4129, -22.4599},
                {49.2552, -80.6959, -50.2067, -95.4205, 58.6603, -82.6829, -85.1581, 41.8180, -30.8574, 97.1981, -82.5958, -28.8795, -85.4085, 4.1487, -66.5393, 10.3298, 1.6701, -5.6475, 27.4476, -43.0301, -76.3096, -50.2332, -78.3696, 94.8636, -23.3943, -12.5253, -0.7261, -22.2998, 75.6843, 26.4727, 89.4912, 58.7736, -59.6224, -79.7282, -83.7753, -27.7563, -34.9129, -59.5483, -90.3698, -61.3443, 24.9823, -95.6877, -24.7911, -33.9130, -21.2824, -20.9725, -5.6511, 48.1722, -76.1615, -34.7881, 89.5121, 21.3902, -64.1337, 64.5694, 28.1176, -63.6191, 78.4931, -30.6986, -38.7517, -78.3101, -89.9835, -24.7913, -3.3755, -26.7747, -69.9146, -84.0696, 89.8581, 88.8403, 11.0283, 42.5199, 3.8255, 84.8410, 39.4760, -69.4343, 33.0611, 79.8494, -43.3108, 86.3245, 18.4006, 11.3358, -70.0537, -56.1613, 86.6678, 61.2669, -87.4609, -3.2283, -58.9770, -90.3010, -43.6246, 91.6989, -0.3419, -91.2213, -71.1247, 23.8534, 33.8477, -25.5041, -45.2000, -78.1703, -9.0687, 29.6663}};
        final double[] y = {16.7472, -78.6970, 55.6269, 81.0493, -74.2367, 35.2803, -47.9342, 17.9189, 31.1192, -62.2628, 26.8560, 40.8794, 45.8709, -2.9343, 95.4711, -44.8028, -43.0432, 91.6748, 34.4285, 2.0237, 90.8166, -84.8920, 48.6440, 81.7356, 19.7079, -11.2201, 7.4781, -50.2725, 86.2425, -62.9504, -44.7818, 55.8809, -37.0799, 57.6691, -68.3776, 92.8757, -66.0538, -19.7837, -0.7157, 93.7062, 88.3785, 45.9402, 66.9602, 75.2111, -53.3467, -35.4818, -78.3654, -37.8808, 6.8023, -93.4543, 36.9790, 45.0599, 92.1773, 75.1468, 11.9977, -73.8175, 9.2572, -34.0833, -88.9701, -12.5273, 82.1229, -28.5230, -75.0823, 79.5513, 19.4386, -84.7566, 53.0255, -27.3148, 90.2895, -91.4218, -83.0654, -63.0655, -63.0166, 16.3185, -40.9833, 24.8072, -64.0708, -12.2585, -47.2775, 17.4870, 31.9656, -36.6499, -43.6935, -35.3558, 17.0193, 2.4296, -22.1589, -1.9623, -99.2679, 92.1499, 86.7539, 30.5582, -71.2991, -3.9449, -94.2419, 65.0603, -56.7646, 78.1036, -53.0729, -68.7246};
        final double[] beta = {-2.07341, 0.17468, 0.08943, 0.06622, -0.10331};
        final double[] se = {6.16950, 0.10777, 0.10973, 0.10612, 0.10445};
        final double[] tStats = {-0.336, 1.621, 0.815, 0.624, -0.989};
        final double[] pValues = {0.738, 0.108, 0.417, 0.534, 0.325};
        runGeneralTest(A, y, beta, se, tStats, pValues, y.length, A.length);
    }

    private void runGeneralTest(double[][] A, double[] y, double[] beta, double[] se, double[] tStats, double[] pValues, int numberOfSamples, int numberOfCovs) {
        final LinearRegression lr = runRegression(A, y, numberOfSamples, numberOfCovs);
        Assert.assertArrayEquals(beta, lr.beta, 1e-4);
        Assert.assertArrayEquals(se, lr.betaError, 1e-4);
        Assert.assertArrayEquals(tStats, lr.tStats, 1e-3);
        Assert.assertArrayEquals(pValues, lr.pValues, 1e-2);
    }

    @Test
    public void testRunRegression() {
        final double a = -10;
        final double b = 10;

        for (int numberOfSamples = 1; numberOfSamples < 25; ++numberOfSamples) {
            for (int numberOfCovars = 0; numberOfCovars < numberOfSamples; ++numberOfCovars) {
                testRunRegression(a, b, numberOfSamples, numberOfCovars);
            }
        }
    }

    private void testRunRegression(double a, double b, int numberOfSamples, int numberOfCovars) {
        final double[][] A = ComputationTestUtils.getRandomMatrix(numberOfCovars, numberOfSamples, a, b);
        final double[][] cA = copyArray(A);

        final double[] y = ComputationTestUtils.getRandomVector(numberOfSamples, a, b);
        final double[] cy = Arrays.copyOf(y, y.length);

        final LinearRegression lr = runRegression(cA, cy, numberOfSamples, numberOfCovars);

        final double[] error = getError(A, lr.beta, y, numberOfSamples, numberOfCovars);
        validateOrthogonality(A, error, numberOfSamples, numberOfCovars);
    }

    private static void validateOrthogonality(double[][] A, double[] x, int numberOfSamples, int numberOfCovars) {
        final double delta = 1e-8;
        Assert.assertEquals(0, LinearAlgebra.sum(x, numberOfSamples), delta);
        for (int i = 0; i < numberOfCovars; ++i) {
            final double dotProd = LinearAlgebra.dotProd(A[i], x, numberOfSamples);
            Assert.assertEquals(0, dotProd, delta);
        }
    }

    private static double[] getError(double[][] A, double[] beta, double[] y, int numberOfSamples, int numberOfCovars) {
        final double[] yProj = getYInSpace(A, beta, numberOfSamples, numberOfCovars);
        final double[] error = Arrays.copyOf(y, numberOfSamples);
        LinearAlgebra.addMultipleOf(error, -1, yProj, 0, numberOfSamples);
        return error;
    }

    private static double[][] copyArray(double[][] a) {
        final double[][] ca = new double[a.length][];
        for (int i = 0; i < a.length; ++i) {
            ca[i] = new double[a[i].length];
            System.arraycopy(a[i], 0, ca[i], 0, a[i].length);
        }
        return ca;
    }

    private static LinearRegression runRegression(double[][] A, double[] y, int numberOfSamples, int numberOfCovars) {
        final LinearRegression lr = new LinearRegression(numberOfSamples, numberOfCovars);
        lr.setData(A, numberOfSamples);
        lr.runRegression(y);
        return lr;
    }

    private static double[] getYInSpace(double[][] A, double[] beta, int m, int n) {
        final double[] y = new double[m];
        Arrays.fill(y, beta[0]);
        for (int i = 0; i < n; ++i) {
            LinearAlgebra.addMultipleOf(y, beta[i + 1], A[i], 0, m);
        }
        return y;
    }
}
