/*
 *  BEGIN_COPYRIGHT
 *
 *  Copyright (C) 2011-2013 deCODE genetics Inc.
 *  Copyright (C) 2013-2019 WuXi NextCode Inc.
 *  All Rights Reserved.
 *
 *  GORpipe is free software: you can redistribute it and/or modify
 *  it under the terms of the AFFERO GNU General Public License as published by
 *  the Free Software Foundation.
 *
 *  GORpipe is distributed "AS-IS" AND WITHOUT ANY WARRANTY OF ANY KIND,
 *  INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 *  NON-INFRINGEMENT, OR FITNESS FOR A PARTICULAR PURPOSE. See
 *  the AFFERO GNU General Public License for the complete license terms.
 *
 *  You should have received a copy of the AFFERO GNU General Public License
 *  along with GORpipe.  If not, see <http://www.gnu.org/licenses/agpl-3.0.html>
 *
 *  END_COPYRIGHT
 */

package org.gorpipe.gor.binsearch;

import junit.framework.TestCase;
import org.gorpipe.gor.util.IntHashMap;
import org.gorpipe.gor.util.StringUtil;
import org.gorpipe.gor.util.Util;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.util.*;

/**
 * Test the BinarySearchItertor
 *
 * @version $Id$
 */
public class UTestBlockPacker extends TestCase {

    private static final Logger log = LoggerFactory.getLogger(UTestBlockPacker.class);

    /**
     * Test converting lookup maps to bytes and read back works
     */
    public void testLookupMapByteConverstion() {
        // Create a map, convert to bytes and back to the map
        Map<Integer, Map<String, Integer>> in = new LinkedHashMap<>();
        in.put(0, createLookupHashMap("kalli", "palli", "nalli", "salli"));
        in.put(3, createLookupHashMap("joe", "smoe", "loe", "foe", "doe", null));
        in.put(100, createLookupHashMap("with", "empty", "char\u0000", "\u0000"));
        in.put(133, createLookupHashMap("1nn", "dkdkd", "dfdsf", "dfjkddk", "dkdkddkkk", "2dd", "3ddd", "sdfsfsdfsd", "888383"));
        in.put(140, createLookupHashMap());
        final byte[] bytes = BlockPacker.bytesFromLookupMap(in);

        final Map<Integer, Map<Integer, byte[]>> out = new HashMap<>();
        BlockPacker.lookupMapFromBytes(out, bytes);

        // Ensure the in and out maps are equal
        assertEquals(in.size() - 1, out.size());
        for (Map.Entry<Integer, Map<String, Integer>> entry : in.entrySet()) {
            final Map<String, Integer> expectedColMap = entry.getValue();
            if (expectedColMap.size() > 0) {
                final Map<Integer, byte[]> foundColMap = out.get(entry.getKey());
                assertEquals(expectedColMap.size(), foundColMap.size());
                for (Map.Entry<String, Integer> lookup : expectedColMap.entrySet()) {
                    final byte[] foundText = foundColMap.get(lookup.getValue());
                    assertEquals(Util.nvl(lookup.getKey(), "").replace("\u0000", ""), new String(foundText));
                }
            }
        }
    }

    private static HashMap<String, Integer> createLookupHashMap(String... keys) {
        final HashMap<String, Integer> map = new HashMap<String, Integer>();
        for (int i = 0; i < keys.length; i++) {
            map.put(keys[i], i);
        }
        return map;
    }

    void compareFiles(String filePath1, String filePath2) {
        try {
            File f1 = new File(filePath1);
            File f2 = new File(filePath2);
            if (f1.length() != f2.length()) {
                fail("Length of files is different for " + filePath1 + " and " + filePath2);
            }
            BufferedInputStream bis1 = new BufferedInputStream(new FileInputStream(f1));
            BufferedInputStream bis2 = new BufferedInputStream(new FileInputStream(f2));

            try {
                while (true) {
                    int a = bis1.read();
                    int b = bis2.read();
                    if (a != b) {
                        fail("Files are different: " + filePath1 + " and " + filePath2);
                    }
                    if (a == -1) {
                        return; // Done, all matches
                    }
                }
            } finally {
                bis1.close();
                bis2.close();
            }
        } catch (Exception e) {
            fail("Exception during compare for " + filePath1 + " and " + filePath2 + " is: " + e.getMessage());
        }
    }

    /**
     * @throws Exception
     */
    public void testSomeZips() throws Exception {
        final byte[][] data = {
                {1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0},
                {1, 1, 1, 1, 1, 1, 1, 1, 1, -9, 1, 1, 1, 1, 1, 1, 1, 1, 1, -9, 1, 1, 1, 1, 1, 1, 1, 1, -9, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, -9, 1, 1, 1, 1, 1, 1, 1, 1, 1, -9},
                {1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2},
                {1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1},
                {1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 8, 8, 9},
                {1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1},
                {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 0, 0, 0, 0, 0, 9},
                {0, 0, 0, 0, 0, 0, 0, 125, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1},
                {125, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1},
                {1, 3, 9, 2, 8, 4, 7, 6, 2, 0, 1, 4, 2, 6, 4, 8, 4, 6, 2, 0, 1, 9, 5, 3, 2, 6, 3, 6, 3, 5, 1, 8, 4, 2, 7, 3, 9, 0, 1, 2, 6, 3, 7, 9, 2, 7, 3, 4, 5, 8},
                {0, 0, 0, 0, 0, 0, 0, 0, 0}
        };
        for (int i = 0; i < data.length; i++) {
            log.info("i = " + i + " size is " + data[i].length + " compresses to " + BlockPacker.zip(data[i]).length);
        }
    }

    /**
     * Test integration with gor
     */
    public void testGorUsage() {
    }

    /**
     * @throws Exception
     */
    public void testKeyValueParsing() throws Exception {
        String[] s = {"RSPOS=10145;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=10229;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000020005000002000200;GENEINFO=LOC100652771:100652771;WGT=1;VC=DIV;R5;ASP;OTHERKG", "RSPOS=10234;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000020005000002000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;OTHERKG", "RSPOS=10248;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000020005000002000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;OTHERKG", "RSPOS=10255;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000020005000002000200;GENEINFO=LOC100652771:100652771;WGT=1;VC=DIV;R5;ASP;OTHERKG", "RSPOS=10291;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000020005000002000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;OTHERKG", "RSPOS=10327;dbSNPBuildID=132;SSR=0;SAO=0;VP=050000020005000002000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;OTHERKG", "RSPOS=10330;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000020005000002000200;GENEINFO=LOC100652771:100652771;WGT=1;VC=DIV;R5;ASP;OTHERKG", "RSPOS=10352;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000020005000002000210;GENEINFO=LOC100652771:100652771;WGT=1;VC=DIV;R5;ASP;OTHERKG;NOC", "RSPOS=10352;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000020005000002000210;GENEINFO=LOC100652771:100652771;WGT=1;VC=DIV;R5;ASP;OTHERKG;NOC", "RSPOS=10383;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000020005000002000210;GENEINFO=LOC100652771:100652771;WGT=1;VC=DIV;R5;ASP;OTHERKG;NOC", "RSPOS=10383;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000020005000002000210;GENEINFO=LOC100652771:100652771;WGT=1;VC=DIV;R5;ASP;OTHERKG;NOC", "RSPOS=10433;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000020005000002000200;GENEINFO=LOC100652771:100652771;WGT=1;VC=DIV;R5;ASP;OTHERKG", "RSPOS=10440;dbSNPBuildID=132;SSR=0;SAO=0;VP=050100020015000102000200;GENEINFO=LOC100652771:100652771;WGT=1;VC=DIV;SLO;R5;OTH;ASP;GNO;OTHERKG", "RSPOS=10440;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000020005000002000200;GENEINFO=LOC100652771:100652771;WGT=1;VC=DIV;R5;ASP;OTHERKG", "RSPOS=10440;dbSNPBuildID=132;SSR=0;SAO=0;VP=050000020015000002000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;OTH;ASP;OTHERKG", "RSPOS=10492;GMAF=0.0617001828153565;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000020005040002000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;VLD;OTHERKG", "RSPOS=10519;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000020005000002000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;OTHERKG", "RSPOS=10583;GMAF=0.270566727605119;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000020005040016000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;VLD;KGPhase1;KGPROD;OTHERKG", "RSPOS=10611;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000020005000014000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;KGPhase1;KGPROD", "RSPOS=10828;dbSNPBuildID=119;SSR=0;SAO=0;VP=050000020005000002000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;OTHERKG", "RSPOS=10904;dbSNPBuildID=119;SSR=0;SAO=0;VP=050000020005000102000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;GNO;OTHERKG", "RSPOS=10927;dbSNPBuildID=119;SSR=0;SAO=0;VP=050000020005000002000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;OTHERKG", "RSPOS=10938;dbSNPBuildID=125;SSR=0;SAO=0;VP=050000020005000002000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;OTHERKG", "RSPOS=11014;dbSNPBuildID=125;SSR=0;SAO=0;VP=050000020005000002000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;OTHERKG", "RSPOS=11022;dbSNPBuildID=125;SSR=0;SAO=0;VP=050000020005000002000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;OTHERKG", "RSPOS=11081;dbSNPBuildID=119;SSR=0;SAO=0;VP=050000020009000102000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;CFL;GNO;OTHERKG", "RSPOS=11863;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000020005000002000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R5;ASP;OTHERKG", "RSPOS=13302;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000080005000014000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;INT;ASP;KGPhase1;KGPROD", "RSPOS=13327;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000080005000014000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;INT;ASP;KGPhase1;KGPROD", "RSPOS=13684;RV;dbSNPBuildID=130;SSR=0;SAO=0;VP=050100040009000102000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;SLO;R3;CFL;GNO;OTHERKG", "RSPOS=13980;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000040005000014000100;GENEINFO=LOC100652771:100652771;WGT=1;VC=SNV;R3;ASP;KGPhase1;KGPROD", "RSPOS=14889;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=14907;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000009040102000100;WGT=1;VC=SNV;CFL;VLD;GNO;OTHERKG", "RSPOS=14930;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000009040102000100;WGT=1;VC=SNV;CFL;VLD;GNO;OTHERKG", "RSPOS=14976;RV;dbSNPBuildID=130;SSR=0;SAO=0;VP=05010000000d000102000100;WGT=1;VC=SNV;SLO;CFL;ASP;GNO;OTHERKG", "RSPOS=15061;RV;dbSNPBuildID=130;SSR=0;SAO=0;VP=05010000000d000102000200;WGT=1;VC=DIV;SLO;CFL;ASP;GNO;OTHERKG", "RSPOS=15118;RV;dbSNPBuildID=130;SSR=0;SAO=0;VP=05010000000d000102000100;WGT=1;VC=SNV;SLO;CFL;ASP;GNO;OTHERKG", "RSPOS=15211;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=15211;dbSNPBuildID=131;SSR=0;SAO=0;VP=05000000000d040102000100;WGT=1;VC=SNV;CFL;ASP;VLD;GNO;OTHERKG", "RSPOS=16257;dbSNPBuildID=131;SSR=0;SAO=0;VP=05000000000d000102000100;WGT=1;VC=SNV;CFL;ASP;GNO;OTHERKG", "RSPOS=16378;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=16495;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=16497;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=17519;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=19226;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=20141;dbSNPBuildID=129;SSR=0;SAO=0;VP=050100000001000402000100;WGT=1;VC=SNV;SLO;HD;OTHERKG", "RSPOS=20144;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=20206;RV;dbSNPBuildID=130;SSR=0;SAO=0;VP=050100000001000102000100;WGT=1;VC=SNV;SLO;GNO;OTHERKG", "RSPOS=20245;RV;GMAF=0.256398537477148;dbSNPBuildID=130;SSR=0;SAO=0;VP=050100000001000102000100;WGT=1;VC=SNV;SLO;GNO;OTHERKG", "RSPOS=20304;RV;GMAF=0.338208409506399;dbSNPBuildID=130;SSR=0;SAO=0;VP=050100000001000102000100;WGT=1;VC=SNV;SLO;GNO;OTHERKG", "RSPOS=26999;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=29436;dbSNPBuildID=100;SSR=0;SAO=0;VP=050000000001000102000100;WGT=1;VC=SNV;GNO;OTHERKG", "RSPOS=30923;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=33487;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000005000102000100;WGT=1;VC=SNV;ASP;GNO;OTHERKG", "RSPOS=33495;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000005040102000100;WGT=1;VC=SNV;ASP;VLD;GNO;OTHERKG", "RSPOS=33505;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000005040102000100;WGT=1;VC=SNV;ASP;VLD;GNO;OTHERKG", "RSPOS=33508;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000005040102000100;WGT=1;VC=SNV;ASP;VLD;GNO;OTHERKG", "RSPOS=33521;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000001040102000100;WGT=1;VC=SNV;VLD;GNO;OTHERKG", "RSPOS=33593;dbSNPBuildID=83;SSR=0;SAO=0;VP=050000000001000002000100;WGT=1;VC=SNV;OTHERKG", "RSPOS=33648;RV;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000001000002000100;WGT=1;VC=SNV;OTHERKG", "RSPOS=33656;RV;dbSNPBuildID=132;SSR=0;SAO=0;VP=050000000001000002000100;WGT=1;VC=SNV;OTHERKG", "RSPOS=51476;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=51479;GMAF=0.113802559414991;dbSNPBuildID=132;SSR=0;SAO=0;VP=050000000005070014000100;WGT=1;VC=SNV;ASP;VLD;G5A;G5;KGPhase1;KGPROD", "RSPOS=51803;GMAF=0.468921389396709;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000001040002000100;WGT=1;VC=SNV;VLD;OTHERKG", "RSPOS=51898;GMAF=0.0731261425959781;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000001000102000100;WGT=1;VC=SNV;GNO;OTHERKG", "RSPOS=51914;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=51928;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000001000102000100;WGT=1;VC=SNV;GNO;OTHERKG", "RSPOS=51935;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=51954;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=52058;GMAF=0.0342778793418647;dbSNPBuildID=129;SSR=1;SAO=0;VP=050000000001040016000140;WGT=1;VC=SNV;VLD;KGPhase1;KGPROD;OTHERKG", "RSPOS=52144;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=52238;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=54353;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=54421;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=54490;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=54676;GMAF=0.191956124314442;dbSNPBuildID=100;SSR=0;SAO=0;VP=050000000005040516000100;WGT=1;VC=SNV;ASP;VLD;HD;GNO;KGPhase1;KGPROD;OTHERKG", "RSPOS=54753;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=54789;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=54789;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=54795;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=55164;GMAF=0.145338208409506;dbSNPBuildID=103;SSR=0;SAO=0;VP=050100000001030116000100;WGT=1;VC=SNV;SLO;G5A;G5;GNO;KGPhase1;KGPROD;OTHERKG", "RSPOS=55299;GMAF=0.278793418647166;dbSNPBuildID=119;SSR=0;SAO=0;VP=050100000001030117000100;WGT=1;VC=SNV;SLO;G5A;G5;GNO;KGPhase1;KGPROD;OTHERKG;PH3", "RSPOS=55302;dbSNPBuildID=103;SSR=0;SAO=0;VP=050000000001000002000100;WGT=1;VC=SNV;OTHERKG", "RSPOS=55313;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=55322;dbSNPBuildID=103;SSR=0;SAO=0;VP=050000000001000002000100;WGT=1;VC=SNV;OTHERKG", "RSPOS=55326;dbSNPBuildID=103;SSR=0;SAO=0;VP=050000000001000516000100;WGT=1;VC=SNV;HD;GNO;KGPhase1;KGPROD;OTHERKG", "RSPOS=55330;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=55367;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=55388;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=55394;dbSNPBuildID=101;SSR=0;SAO=0;VP=050000000001000117000100;WGT=1;VC=SNV;GNO;KGPhase1;KGPROD;OTHERKG;PH3", "RSPOS=55416;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=55427;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=55545;dbSNPBuildID=125;SSR=0;SAO=0;VP=050000000001000102000100;WGT=1;VC=SNV;GNO;OTHERKG", "RSPOS=55816;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=55850;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=55852;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=56644;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=57952;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=58771;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=58814;GMAF=0.0982632541133455;dbSNPBuildID=132;SSR=0;SAO=0;VP=050000000005050014000100;WGT=1;VC=SNV;ASP;VLD;G5;KGPhase1;KGPROD", "RSPOS=59040;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=60718;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000009000102000100;WGT=1;VC=SNV;CFL;GNO;OTHERKG", "RSPOS=60726;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=60791;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000009000102000100;WGT=1;VC=SNV;CFL;GNO;OTHERKG", "RSPOS=61442;GMAF=0.076782449725777;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000009000116000100;WGT=1;VC=SNV;CFL;GNO;KGPhase1;KGPROD;OTHERKG", "RSPOS=61462;GMAF=0.0383912248628885;dbSNPBuildID=129;SSR=0;SAO=0;VP=050100000009030116000100;WGT=1;VC=SNV;SLO;CFL;G5A;G5;GNO;KGPhase1;KGPROD;OTHERKG", "RSPOS=61480;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000009000102000100;WGT=1;VC=SNV;CFL;GNO;OTHERKG", "RSPOS=61499;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000009000102000100;WGT=1;VC=SNV;CFL;GNO;OTHERKG", "RSPOS=61743;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=61920;GMAF=0.0255941499085923;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000009040002000100;WGT=1;VC=SNV;CFL;VLD;OTHERKG", "RSPOS=61987;GMAF=0.292961608775137;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000009000116000100;WGT=1;VC=SNV;CFL;GNO;KGPhase1;KGPROD;OTHERKG", "RSPOS=61989;GMAF=0.309414990859232;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000009000116000100;WGT=1;VC=SNV;CFL;GNO;KGPhase1;KGPROD;OTHERKG", "RSPOS=61993;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=62156;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=62157;GMAF=0.00228519195612431;dbSNPBuildID=119;SSR=0;SAO=0;VP=050000000009040016000100;WGT=1;VC=SNV;CFL;VLD;KGPhase1;KGPROD;OTHERKG", "RSPOS=62162;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=62203;dbSNPBuildID=125;SSR=0;SAO=0;VP=05000000000900000e000100;WGT=1;VC=SNV;CFL;KGPilot123;KGPROD;OTHERKG", "RSPOS=62271;GMAF=0.138482632541133;dbSNPBuildID=125;SSR=0;SAO=0;VP=050000000009040002000100;WGT=1;VC=SNV;CFL;VLD;OTHERKG", "RSPOS=63268;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000009000102000100;WGT=1;VC=SNV;CFL;GNO;OTHERKG", "RSPOS=63276;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=63297;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=63671;GMAF=0.170018281535649;dbSNPBuildID=132;SSR=0;SAO=0;VP=050000000005050014000100;WGT=1;VC=SNV;ASP;VLD;G5;KGPhase1;KGPROD", "RSPOS=63738;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000009000002000210;WGT=1;VC=DIV;CFL;OTHERKG;NOC", "RSPOS=63738;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000009000002000210;WGT=1;VC=DIV;CFL;OTHERKG;NOC", "RSPOS=64649;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=66008;dbSNPBuildID=100;SSR=0;SAO=0;VP=050100000005000102000100;WGT=1;VC=SNV;SLO;ASP;GNO;OTHERKG", "RSPOS=66162;GMAF=0.320383912248629;dbSNPBuildID=129;SSR=0;SAO=0;VP=050100000005000116000100;WGT=1;VC=SNV;SLO;ASP;GNO;KGPhase1;KGPROD;OTHERKG", "RSPOS=66176;GMAF=0.0484460694698355;dbSNPBuildID=125;SSR=0;SAO=0;VP=050000000005040016000100;WGT=1;VC=SNV;ASP;VLD;KGPhase1;KGPROD;OTHERKG", "RSPOS=66219;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=66238;dbSNPBuildID=132;SSR=0;SAO=0;VP=050100000005000102000100;WGT=1;VC=SNV;SLO;ASP;GNO;OTHERKG", "RSPOS=66314;dbSNPBuildID=125;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=66331;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=66334;dbSNPBuildID=125;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=66442;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=66457;GMAF=0.0795246800731261;dbSNPBuildID=121;SSR=0;SAO=0;VP=050000000005040016000100;WGT=1;VC=SNV;ASP;VLD;KGPhase1;KGPROD;OTHERKG", "RSPOS=66503;dbSNPBuildID=132;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=66507;GMAF=0.479890310786106;dbSNPBuildID=120;SSR=0;SAO=0;VP=050000000005040016000100;WGT=1;VC=SNV;ASP;VLD;KGPhase1;KGPROD;OTHERKG", "RSPOS=66651;dbSNPBuildID=100;SSR=0;SAO=0;VP=050000000005000102000100;WGT=1;VC=SNV;ASP;GNO;OTHERKG", "RSPOS=67179;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000020005000014000100;GENEINFO=OR4F5:79501;WGT=1;VC=SNV;R5;ASP;KGPhase1;KGPROD", "RSPOS=67181;GMAF=0.0470749542961609;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000020005070116000100;GENEINFO=OR4F5:79501;WGT=1;VC=SNV;R5;ASP;VLD;G5A;G5;GNO;KGPhase1;KGPROD;OTHERKG", "RSPOS=67223;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000020005000102000100;GENEINFO=OR4F5:79501;WGT=1;VC=SNV;R5;ASP;GNO;OTHERKG", "RSPOS=69428;dbSNPBuildID=134;SSR=0;SAO=0;VP=050200000a05040002000100;GENEINFO=OR4F5:79501;WGT=1;VC=SNV;S3D;NSM;REF;ASP;VLD;OTHERKG", "RSPOS=69453;dbSNPBuildID=134;SSR=0;SAO=0;VP=050200000305000002000100;GENEINFO=OR4F5:79501;WGT=1;VC=SNV;S3D;REF;SYN;ASP;OTHERKG", "RSPOS=69476;dbSNPBuildID=134;SSR=0;SAO=0;VP=050200000a05040002000100;GENEINFO=OR4F5:79501;WGT=1;VC=SNV;S3D;NSM;REF;ASP;VLD;OTHERKG", "RSPOS=69496;dbSNPBuildID=134;SSR=0;SAO=0;VP=050200000a05040002000100;GENEINFO=OR4F5:79501;WGT=1;VC=SNV;S3D;NSM;REF;ASP;VLD;OTHERKG", "RSPOS=69511;GMAF=0.193784277879342;dbSNPBuildID=131;SSR=0;SAO=0;VP=050200000a01000116000100;GENEINFO=OR4F5:79501;WGT=1;VC=SNV;S3D;NSM;REF;GNO;KGPhase1;KGPROD;OTHERKG", "RSPOS=69534;dbSNPBuildID=135;SSR=0;SAO=0;VP=050200000305000014000100;GENEINFO=OR4F5:79501;WGT=1;VC=SNV;S3D;REF;SYN;ASP;KGPhase1;KGPROD", "RSPOS=69552;dbSNPBuildID=129;SSR=0;SAO=0;VP=050300000301040402000100;GENEINFO=OR4F5:79501;WGT=1;VC=SNV;S3D;SLO;REF;SYN;VLD;HD;OTHERKG", "RSPOS=69590;dbSNPBuildID=134;SSR=0;SAO=0;VP=050200000a05000002000100;GENEINFO=OR4F5:79501;WGT=1;VC=SNV;S3D;NSM;REF;ASP;OTHERKG", "RSPOS=69594;dbSNPBuildID=134;SSR=0;SAO=0;VP=050200000305000002000100;GENEINFO=OR4F5:79501;WGT=1;VC=SNV;S3D;REF;SYN;ASP;OTHERKG", "RSPOS=72148;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=73841;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=74651;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000001000002000100;WGT=1;VC=SNV;OTHERKG", "RSPOS=74681;GMAF=0.286106032906764;dbSNPBuildID=121;SSR=0;SAO=0;VP=050000000009040002000100;WGT=1;VC=SNV;CFL;VLD;OTHERKG", "RSPOS=74709;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000009000002000100;WGT=1;VC=SNV;CFL;OTHERKG", "RSPOS=74771;dbSNPBuildID=121;SSR=0;SAO=0;VP=050000000009000002000100;WGT=1;VC=SNV;CFL;OTHERKG", "RSPOS=74790;dbSNPBuildID=121;SSR=0;SAO=0;VP=050000000009000002000100;WGT=1;VC=SNV;CFL;OTHERKG", "RSPOS=74792;dbSNPBuildID=121;SSR=0;SAO=0;VP=050000000009000002000100;WGT=1;VC=SNV;CFL;OTHERKG", "RSPOS=77462;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=77470;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=77874;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=77961;GMAF=0.125685557586837;dbSNPBuildID=131;SSR=0;SAO=0;VP=050000000001040016000100;WGT=1;VC=SNV;VLD;KGPhase1;KGPROD;OTHERKG", "RSPOS=79033;GMAF=0.438299817184644;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000001000516000100;WGT=1;VC=SNV;HD;GNO;KGPhase1;KGPROD;OTHERKG", "RSPOS=79050;GMAF=0.224405850091408;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000001000116000100;WGT=1;VC=SNV;GNO;KGPhase1;KGPROD;OTHERKG", "RSPOS=79137;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=79417;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=79418;RV;GMAF=0.0178244972577697;dbSNPBuildID=100;SSR=0;SAO=0;VP=050100000001040002000100;WGT=1;VC=SNV;SLO;VLD;OTHERKG", "RSPOS=79538;RV;dbSNPBuildID=100;SSR=0;SAO=0;VP=050000000001000002000100;WGT=1;VC=SNV;OTHERKG", "RSPOS=79772;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=79872;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=80323;RV;dbSNPBuildID=108;SSR=0;SAO=0;VP=050100000009000102000100;WGT=1;VC=SNV;SLO;CFL;GNO;OTHERKG", "RSPOS=80386;RV;GMAF=0.0118829981718464;dbSNPBuildID=108;SSR=0;SAO=0;VP=050100000001040002000100;WGT=1;VC=SNV;SLO;VLD;OTHERKG", "RSPOS=80454;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=81836;RV;dbSNPBuildID=100;SSR=0;SAO=0;VP=050100000005000102000100;WGT=1;VC=SNV;SLO;ASP;GNO;OTHERKG", "RSPOS=81949;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=81962;RV;dbSNPBuildID=108;SSR=0;SAO=0;VP=050100000005000002000200;WGT=1;VC=DIV;SLO;ASP;OTHERKG", "RSPOS=82102;RV;dbSNPBuildID=108;SSR=0;SAO=0;VP=050100000005000002000100;WGT=1;VC=SNV;SLO;ASP;OTHERKG", "RSPOS=82103;RV;dbSNPBuildID=92;SSR=0;SAO=0;VP=050100000005000002000100;WGT=1;VC=SNV;SLO;ASP;OTHERKG", "RSPOS=82126;RV;dbSNPBuildID=92;SSR=0;SAO=0;VP=050100000005000002000100;WGT=1;VC=SNV;SLO;ASP;OTHERKG", "RSPOS=82136;RV;dbSNPBuildID=108;SSR=0;SAO=0;VP=050100000005000002000200;WGT=1;VC=DIV;SLO;ASP;OTHERKG", "RSPOS=82136;RV;dbSNPBuildID=108;SSR=0;SAO=0;VP=050100000005000002000200;WGT=1;VC=DIV;SLO;ASP;OTHERKG", "RSPOS=82154;dbSNPBuildID=111;SSR=0;SAO=0;VP=050000000005000402000100;WGT=1;VC=SNV;ASP;HD;OTHERKG", "RSPOS=82162;RV;GMAF=0.0351919561243144;dbSNPBuildID=92;SSR=0;SAO=0;VP=050100000005000102000100;WGT=1;VC=SNV;SLO;ASP;GNO;OTHERKG", "RSPOS=82163;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=82196;dbSNPBuildID=132;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=82249;RV;GMAF=0.0452468007312614;dbSNPBuildID=92;SSR=0;SAO=0;VP=050100000005040016000100;WGT=1;VC=SNV;SLO;ASP;VLD;KGPhase1;KGPROD;OTHERKG", "RSPOS=82282;RV;dbSNPBuildID=108;SSR=0;SAO=0;VP=050100000005000002000100;WGT=1;VC=SNV;SLO;ASP;OTHERKG", "RSPOS=82303;RV;dbSNPBuildID=108;SSR=0;SAO=0;VP=050100000005000002000100;WGT=1;VC=SNV;SLO;ASP;OTHERKG", "RSPOS=82316;RV;dbSNPBuildID=108;SSR=0;SAO=0;VP=050100000005000102000100;WGT=1;VC=SNV;SLO;ASP;GNO;OTHERKG", "RSPOS=82609;dbSNPBuildID=134;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=82676;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=82734;RV;GMAF=0.261882998171846;dbSNPBuildID=108;SSR=0;SAO=0;VP=050100000005040016000100;WGT=1;VC=SNV;SLO;ASP;VLD;KGPhase1;KGPROD;OTHERKG", "RSPOS=82957;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=83084;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=83088;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=83107;dbSNPBuildID=111;SSR=0;SAO=0;VP=050000000005000002000100;WGT=1;VC=SNV;ASP;OTHERKG", "RSPOS=83120;RV;dbSNPBuildID=108;SSR=0;SAO=0;VP=050100000005000002000200;WGT=1;VC=DIV;SLO;ASP;OTHERKG", "RSPOS=83120;RV;dbSNPBuildID=108;SSR=0;SAO=0;VP=050100000005000002000200;WGT=1;VC=DIV;SLO;ASP;OTHERKG", "RSPOS=83771;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=83794;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=83827;dbSNPBuildID=129;SSR=0;SAO=0;VP=050100000005000102000700;WGT=1;VC=MIXED;SLO;ASP;GNO;OTHERKG", "RSPOS=83827;dbSNPBuildID=129;SSR=0;SAO=0;VP=050100000005000102000700;WGT=1;VC=MIXED;SLO;ASP;GNO;OTHERKG", "RSPOS=83857;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=83873;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=83873;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=83885;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=83899;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=83903;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=83923;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=83931;RV;dbSNPBuildID=130;SSR=0;SAO=0;VP=050100000005000102000200;WGT=1;VC=DIV;SLO;ASP;GNO;OTHERKG", "RSPOS=83931;RV;dbSNPBuildID=130;SSR=0;SAO=0;VP=050100000005000102000200;WGT=1;VC=DIV;SLO;ASP;GNO;OTHERKG", "RSPOS=83935;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=83935;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=83977;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=84002;GMAF=0.138939670932358;dbSNPBuildID=125;SSR=0;SAO=0;VP=050000000005040016000100;WGT=1;VC=SNV;ASP;VLD;KGPhase1;KGPROD;OTHERKG", "RSPOS=84010;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=84020;dbSNPBuildID=129;SSR=0;SAO=0;VP=050000000005000002000200;WGT=1;VC=DIV;ASP;OTHERKG", "RSPOS=84079;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=84139;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=84156;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=84244;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=84295;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=84346;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=84453;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=84705;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=85063;dbSNPBuildID=135;SSR=0;SAO=0;VP=050000000005000014000100;WGT=1;VC=SNV;ASP;KGPhase1;KGPROD", "RSPOS=85200;RV;GMAF=0.0680987202925046;dbSNPBuildID=108;SSR=0;SAO=0;VP=050100000005040002000100;WGT=1;VC=SNV;SLO;ASP;VLD;OTHERKG"};
        final HashMap<String, Integer> map = new HashMap<String, Integer>();
//        final IntArray counts = new IntArray();
        final IntHashMap counts = new IntHashMap();
        final ArrayList<String> parts = new ArrayList<>();
        int totalLen = 0;
        for (String value : s) {
            totalLen += s.length;
            final ArrayList<String> list = StringUtil.split(value, ';');
            for (String l : list) {
                Integer idx = map.get(l);
                if (idx == null) {
                    idx = map.size();
                    counts.put(idx, 1);
                    map.put(l, idx);
                    parts.add(l);
                } else {
                    counts.increment(idx);
                }
            }
        }

        map.clear();
        int[] keys = counts.keysToArraySortByValue(true);
        for (int key : keys) {
            if (counts.get(key) > 1 && map.size() < 255) {
                map.put(parts.get(key), map.size() + 1);
            }
        }

        final StringBuilder kalli = new StringBuilder();
        String[] a = new String[map.size()];
        for (int i = 0; i < map.size(); i++) {
            kalli.append(parts.get(keys[i]));
            kalli.append('\0');
            a[i] = parts.get(keys[i]);
        }
        Arrays.sort(a);
        final StringBuilder palli = new StringBuilder();
        for (String value : a) {
            palli.append(value);
            palli.append('\0');
        }

        log.info("map.size = " + map.size());
        log.info("encoded size = " + kalli.length());
        log.info("original size = " + totalLen);
        log.info("zip = " + BlockPacker.zip(kalli.toString().getBytes()).length);
        log.info("sorted zip = " + BlockPacker.zip(palli.toString().getBytes()).length);

        Random rand = new Random();
        for (int i = 0; i < s.length; i++) {
            kalli.append(9);
            palli.append(9);
            for (int j = 0; j < 9; j++) {
                int r = rand.nextInt(255);
                kalli.append((byte) r);
                palli.append((byte) r);
            }
        }

        log.info("zip = " + BlockPacker.zip(kalli.toString().getBytes()).length);
        log.info("sorted zip = " + BlockPacker.zip(palli.toString().getBytes()).length);
    }

    /**
     * Test usage of externally stored lookup tables
     *
     * @throws Exception
     */
    public void testExternalTables() throws Exception {
        // Create two blocks to encode
        final String in1 = gorsat.TestUtils.runGorPipe("gor 1.mem | where pos < 2000");
        final String in2 = gorsat.TestUtils.runGorPipe("gor 1.mem | where pos > 2000");
        final byte[] buf1 = new byte[in1.length()];
        final byte[] buf2 = new byte[in2.length()];

        // Encode data in blocks, capture sizes for comparison
        final int in1encodelen = BlockPacker.encode(in1.getBytes(), buf1);
        final int in2encodelen = BlockPacker.encode(in2.getBytes(), buf2);
        final int in1ziplen = BlockPacker.zip(buf1, 0, in1encodelen).length;
        final int in2ziplen = BlockPacker.zip(buf2, 0, in2encodelen).length;

        // Check that a valid decode can be carried out
        final byte[] dest1 = new byte[in1.length()];
        final byte[] dest2 = new byte[in2.length()];
        int len = BlockPacker.decode(buf1, 0, dest1, 0);
        assertEquals(in1, new String(dest1, 0, len));
        len = BlockPacker.decode(buf2, 0, dest2, 0);
        assertEquals(in2, new String(dest2, 0, len));

        // Trigger storage of 5-th column in an externally managed map
        final Map<Integer, Map<String, Integer>> encodeMap = new HashMap<>();
        encodeMap.put(4, new HashMap<>());

        // Encode and observer the storage of blocks to shrink
        final int extin1encodelen = BlockPacker.encode(in1.getBytes(), buf1, encodeMap, true, Integer.MAX_VALUE);
        final int extin2encodelen = BlockPacker.encode(in2.getBytes(), buf2, encodeMap, true, Integer.MAX_VALUE);
        final int extin1ziplen = BlockPacker.zip(buf1, 0, extin1encodelen).length;
        final int extin2ziplen = BlockPacker.zip(buf2, 0, extin2encodelen).length;
        log.info("1 len = {}, encoded len = {}, zip = {}, external table encoded len {}, zip = {}", in1.length(), in1encodelen, in1ziplen, extin1encodelen, extin1ziplen);
        log.info("2 len = {}, encoded len = {}, zip = {}, external table encoded len {}, zip = {}", in2.length(), in2encodelen, in2ziplen, extin2encodelen, extin2ziplen);

        // Create decode table check that a valid decode can be carried out
        final Map<Integer, Map<Integer, byte[]>> decodeMap = new HashMap<>();
        for (Map.Entry<Integer, Map<String, Integer>> column : encodeMap.entrySet()) {
            final Map<Integer, byte[]> map = new HashMap<>();
            decodeMap.put(column.getKey(), map);
            for (Map.Entry<String, Integer> entry : column.getValue().entrySet()) {
                map.put(entry.getValue(), entry.getKey().getBytes());
            }
        }
        len = BlockPacker.decode(buf1, 0, dest1, 0, decodeMap);
        assertEquals(in1, new String(dest1, 0, len));
        len = BlockPacker.decode(buf2, 0, dest2, 0, decodeMap);
        assertEquals(in2, new String(dest2, 0, len));
    }

    /**
     * Test simple rounds of encode and decode
     *
     * @throws Exception
     */
    public void testEncodeDecode() throws Exception {
        long seed = new Random().nextLong();
        Random rand = new Random(seed);
        // Create data with different poolsizes (as it results in different code paths)
        final String input = mergeColumns(
                generateColumnData("string", 23, 150, false, null),
                generateColumnData("integer", 4000, 1, false, null),
                generateColumnData("string", 50, 1, false, rand),
                generateColumnData("integer", 50, 1, false, rand),
                generateColumnData("string", 140, 1, false, rand),
                generateColumnData("integer", 140, 1, false, rand),
                generateColumnData("string", 500, 1, false, rand),
                generateColumnData("integer", 500, 1, false, rand));

        final byte[] buf = new byte[input.length()];
        final int size = BlockPacker.encode(input, buf);
        log.info("seed = {}, strlen = {}, encoded len = {}", seed, input.length(), size);

        final byte[] bytes = new byte[input.length() + 1];
        final int len = BlockPacker.decode(buf, 0, bytes, 0);

        final String output = new String(bytes, 0, len);
        assertEquals(input, output);
        assertEquals(input.length(), len);

        byte[] zipped = BlockPacker.zip(buf, 0, size);
        log.info("encoded 2 block len is {}, encoded 2 block zipped len is {}, total is {}",
                size, zipped.length, (input.length() / (zipped.length * 1.0f)));
    }

    /**
     * Test simple rounds of encode and decode with some fields empty
     *
     * @throws Exception
     */
    public void testWithEmptyFields() throws Exception {
        long seed = new Random().nextLong();
        Random rand = new Random(seed);

        // Create data with different poolsizes (as it results in different code paths)
        final String input = mergeColumns(
                generateColumnData("string", 23, 150, false, null),
                generateColumnData("integer", 4000, 1, false, null),
                generateColumnData("string", 50, 1, true, rand),
                generateColumnData("integer", 50, 1, true, rand),
                generateColumnData("string", 140, 1, true, rand),
                generateColumnData("integer", 140, 1, true, rand),
                generateColumnData("string", 500, 1, true, rand),
                generateColumnData("integer", 500, 1, true, rand));

        final byte[] buf = new byte[input.length()];
        final int size = BlockPacker.encode(input, buf);
        log.info("seed = {}, strlen = {}, encoded len = {}", seed, input.length(), size);

        final byte[] bytes = new byte[input.length() + 1];
        final int len = BlockPacker.decode(buf, 0, bytes, 0);

        final String output = new String(bytes, 0, len);
        assertEquals(input, output);
        assertEquals(input.length(), len);

        byte[] zipped = BlockPacker.zip(buf, 0, size);
        log.info("encoded 2 block len is {}, encoded 2 block zipped len is {}, total is {}",
                size, zipped.length, (input.length() / (zipped.length * 1.0f)));
    }

    /**
     * Generate column data.
     *
     * @param type        String or Integer
     * @param poolSize    size of the pool we draw values from.
     * @param repeats     of often is each value repeated.
     * @param includeNull should the data include nulls
     * @param random      should it be ordered or random (ordered if random = null).
     * @return
     */
    private String[] generateColumnData(String type, int poolSize, int repeats, boolean includeNull, Random random) {
        if (repeats < 1) return new String[0];
        String[] pool = new String[poolSize];
        for (int i = 0; i < pool.length; i++) {
            pool[i] = (type.toLowerCase() == "string" ? "data" : "") + i;
        }
        String[] data = new String[4000];
        int i = 0;
        while (i < data.length) {
            String val = null;
            if (!includeNull || i % 5 != 0) {
                val = data[i] = random == null ? pool[i % pool.length] : pool[random.nextInt(pool.length)];
            }
            for (int r = 0; r < repeats; r++) {
                data[i++] = val;
                if (i == data.length) break;
            }
        }
        return data;
    }

    /**
     * Transfrom array of columns into
     *
     * @param columns the columns to merge, must be all of the same length.
     * @return csv table of the column data.
     */
    private String mergeColumns(String[]... columns) {
        StringBuffer buffer = new StringBuffer();
        for (int i = 0; i < columns[0].length; i++) {
            for (String[] col : columns) {
                buffer.append(col[i] != null ? col[i] : "");
                buffer.append("\t");
            }
            buffer.append('\n');
        }
        return buffer.toString();
    }
}
