package albedo

import (
	"github.com/genshinsim/gcsim/pkg/core/attacks"
	"github.com/genshinsim/gcsim/pkg/core/attributes"
	"github.com/genshinsim/gcsim/pkg/core/combat"
	"github.com/genshinsim/gcsim/pkg/core/player/character"
	"github.com/genshinsim/gcsim/pkg/core/player/shield"
	"github.com/genshinsim/gcsim/pkg/modifier"
)

const (
	// Transient Blossoms generated by Abiogenesis: Solar Isotoma grant Albedo Fatal Reckoning for 30s:
	// • Unleashing Rite of Progeniture: Tectonic Tide consumes all stacks of Fatal Reckoning. Each stack of Fatal Reckoning consumed increases the DMG dealt by Fatal Blossoms and Rite of Progeniture: Tectonic Tide's burst DMG by 30% of Albedo's DEF.
	// • This effect stacks up to 4 times.
	c2key = "albedo-c2"
)

// C4:
// Active party members within the Solar Isotoma field have their Plunging Attack DMG increased by 30%.
func (c *char) c4(lastConstruct int) func() {
	return func() {
		// don't apply / check again if skill was recast
		if c.lastConstruct != lastConstruct {
			return
		}
		// don't apply / check again if skill isn't active anymore
		if !c.skillActive {
			return
		}

		// apply C4 buff to active char for 1s if within skill area
		inSkillArea := c.Core.Combat.Player().IsWithinArea(c.skillArea)
		if inSkillArea {
			active := c.Core.Player.ActiveChar()
			m := make([]float64, attributes.EndStatType)
			m[attributes.DmgP] = 0.3
			active.AddAttackMod(character.AttackMod{
				Base: modifier.NewBaseWithHitlag("albedo-c4", 60), // 1s
				Amount: func(atk *combat.AttackEvent, _ combat.Target) ([]float64, bool) {
					if atk.Info.AttackTag != attacks.AttackTagPlunge {
						return nil, false
					}
					return m, true
				},
			})
		}

		// check again in 0.3s
		c.Core.Tasks.Add(c.c4(lastConstruct), 18)
	}
}

// C6:
// Active party members within the Solar Isotoma field who are protected by a shield created by Crystallize have their DMG increased by 17%.
func (c *char) c6(lastConstruct int) func() {
	return func() {
		// don't apply / check again if skill was recast
		if c.lastConstruct != lastConstruct {
			return
		}
		// don't apply / check again if skill isn't active anymore
		if !c.skillActive {
			return
		}

		// apply C6 buff to active char for 1s if they are protected by crystallize and within the skill area
		crystallizeShield := c.Core.Player.Shields.Get(shield.Crystallize) != nil
		inSkillArea := c.Core.Combat.Player().IsWithinArea(c.skillArea)
		if crystallizeShield && inSkillArea {
			active := c.Core.Player.ActiveChar()
			m := make([]float64, attributes.EndStatType)
			m[attributes.DmgP] = 0.17
			active.AddStatMod(character.StatMod{
				Base:         modifier.NewBase("albedo-c6", 60), // 1s
				AffectedStat: attributes.DmgP,
				Amount: func() ([]float64, bool) {
					return m, true
				},
			})
		}

		// check again in 0.3s
		c.Core.Tasks.Add(c.c6(lastConstruct), 18)
	}
}
