package character

import (
	"bytes"
	"fmt"
	"go/format"
	"os"
	"text/template"
)

type icdData struct {
	TagData   []string
	GroupData []ICDGroupData
}

func (g *Generator) GenerateICDData(dir string) error {
	t, err := template.New("icd_groups").Parse(icdGroupTmpl)
	if err != nil {
		return fmt.Errorf("failed to build template: %w", err)
	}
	var data icdData
	grpDupCheck := make(map[string]bool)
	tagDupCheck := make(map[string]bool)

	for i := range g.chars {
		for _, v := range g.chars[i].ICDGroups {
			if _, ok := grpDupCheck[v.GroupName]; ok {
				return fmt.Errorf("duplicated group tag %v", v.GroupName)
			}
			grpDupCheck[v.GroupName] = true
			data.GroupData = append(data.GroupData, v)
		}
		for _, v := range g.chars[i].ICDTags {
			if _, ok := tagDupCheck[v]; ok {
				return fmt.Errorf("duplicated icd tag: %v", v)
			}
			tagDupCheck[v] = true
			data.TagData = append(data.TagData, v)
		}
	}

	buf := new(bytes.Buffer)
	err = t.Execute(buf, data)
	if err != nil {
		return fmt.Errorf("failed to execute icd data template: %w", err)
	}
	src := buf.Bytes()
	dst, err := format.Source(src)
	if err != nil {
		fmt.Println(string(src))
		return fmt.Errorf("failed to gofmt on generated icd data: %w", err)
	}
	err = os.WriteFile(fmt.Sprintf("%v/icd_groups_gen.go", dir), dst, 0o644)
	if err != nil {
		return fmt.Errorf("failed to write icd_groups_gen.go: %w", err)
	}

	// tags
	t, err = template.New("icd_tags").Parse(icdTagTmpl)
	if err != nil {
		return fmt.Errorf("failed to build template: %w", err)
	}
	buf = new(bytes.Buffer)
	err = t.Execute(buf, data)
	if err != nil {
		return fmt.Errorf("failed to execute icd data template: %w", err)
	}
	src = buf.Bytes()
	dst, err = format.Source(src)
	if err != nil {
		fmt.Println(string(src))
		return fmt.Errorf("failed to gofmt on generated icd data: %w", err)
	}
	err = os.WriteFile(fmt.Sprintf("%v/icd_tags_gen.go", dir), dst, 0o644)
	if err != nil {
		return fmt.Errorf("failed to write icd_groups_gen.go: %w", err)
	}

	return nil
}

const icdGroupTmpl = `// Code generated by "pipeline"; DO NOT EDIT.
package attacks

const (
	ICDGroupGeneratedStart ICDGroup = iota + EndDefaultICDGroups + 1
{{ range $index, $ele := .GroupData }}
	{{- $ele.GroupName}}
{{ end -}}
	ICDGroupLength
)

func init() {
{{- range $index, $ele := .GroupData }}
	ICDGroupResetTimer[{{$ele.GroupName}}] = {{$ele.ResetTimer}}
	ICDGroupEleApplicationSequence[{{$ele.GroupName}}] = []float64{ {{- range $val := $ele.EleAppSequence -}}{{$val}},{{ end}} } 
	ICDGroupDamageSequence[{{$ele.GroupName}}] = []float64{ {{- range $val := $ele.DamageSequence -}}{{$val}},{{ end}} } 
{{end -}}
}
`

const icdTagTmpl = `// Code generated by "pipeline"; DO NOT EDIT.
package attacks

const (
	ICDTagGeneratedStart ICDTag = iota + EndDefaultICDTags + 1
{{ range $index, $ele := .TagData }}
	{{- $ele}}
{{ end -}}
	ICDTagLength
)
`
