package character

import (
	"bytes"
	"fmt"
	"go/format"
	"log"
	"os"
	"text/template"

	"github.com/genshinsim/gcsim/pkg/model"
	"golang.org/x/text/cases"
	"golang.org/x/text/language"
)

type keyData struct {
	VarName    string
	Key        string
	PrettyName string
	Element    string
}

func (g *Generator) GenerateKeys(dir string) error {
	t, err := template.New("char_keys").Parse(keysTmpl)
	if err != nil {
		return fmt.Errorf("failed to build template: %w", err)
	}
	var data []keyData

	for i := range g.chars {
		v := g.chars[i]
		dm, ok := g.data[v.Key]
		if !ok {
			log.Printf("No data found for %v; skipping", v.Key)
			continue
		}
		if dm.SubId > 0 {
			// skip traveler
			continue
		}
		if v.KeyVarName == "" {
			v.KeyVarName = cases.Title(language.AmericanEnglish).String(v.Key)
		}
		if v.PrettyName == "" {
			v.PrettyName = cases.Title(language.AmericanEnglish).String(v.Key)
		}

		data = append(data, keyData{
			VarName:    v.KeyVarName,
			Key:        v.Key,
			PrettyName: v.PrettyName,
			Element:    eleAttrString(dm.Element),
		})
	}

	buf := new(bytes.Buffer)
	err = t.Execute(buf, data)
	if err != nil {
		return fmt.Errorf("failed to execute keys data template: %w", err)
	}
	src := buf.Bytes()
	dst, err := format.Source(src)
	if err != nil {
		fmt.Println(string(src))
		return fmt.Errorf("failed to gofmt on generated keys data: %w", err)
	}
	err = os.WriteFile(fmt.Sprintf("%v/keys_char_gen.go", dir), dst, 0o644)
	if err != nil {
		return fmt.Errorf("failed to write keys_char_gen.go: %w", err)
	}

	return nil
}

func eleAttrString(ele model.Element) string {
	switch ele {
	case model.Element_Electric:
		return "Electro"
	case model.Element_Fire:
		return "Pyro"
	case model.Element_Ice:
		return "Cryo"
	case model.Element_Water:
		return "Hydro"
	case model.Element_Grass:
		return "Dendro"
	case model.Element_Wind:
		return "Anemo"
	case model.Element_Rock:
		return "Geo"
	default:
		return "UnknownElement"
	}
}

const keysTmpl = `// Code generated by "pipeline"; DO NOT EDIT.
package keys

import (
	"github.com/genshinsim/gcsim/pkg/core/attributes"
)

const (
	TestCharDoNotUse Char = iota + TravelerDelim + 1
{{ range $index, $ele := . }}
	{{- $ele.VarName}}
{{ end -}}
    EndCharKeys
)

func init() {
{{- range $index, $ele := . }}
	charNames[{{$ele.VarName}}] = "{{$ele.Key}}"
	charPrettyName[{{$ele.VarName}}] = "{{$ele.PrettyName}}"
	CharKeyToEle[{{$ele.VarName}}] = attributes.{{$ele.Element}}
{{end -}}
}
`
