package weapon

import (
	"bytes"
	"fmt"
	"go/format"
	"log"
	"os"
	"strings"
	"text/template"

	"github.com/genshinsim/gcsim/pkg/model"
	"google.golang.org/protobuf/encoding/prototext"
)

type weaponData struct {
	Config
	Data *model.WeaponData
}

func (g *Generator) GenerateTemplate() error {
	t, err := template.New("chartemplate").Parse(tmpl)
	if err != nil {
		return err
	}
	for _, v := range g.weaps {
		dm, ok := g.data[v.Key]
		if !ok {
			log.Printf("No data found for %v; skipping", v.Key)
			continue
		}
		// write the data to []byte
		opt := prototext.MarshalOptions{
			Multiline: true,
			Indent:    "   ",
		}
		b, err := opt.Marshal(dm)
		if err != nil {
			log.Printf("error marshalling %v data to proto\n", v.Key)
			return err
		}
		// hack to work around stupid prototext not stable (on purpose - google u suck)
		b = []byte(strings.ReplaceAll(string(b), ":  ", ": "))
		os.WriteFile(fmt.Sprintf("%v/data_gen.textproto", v.RelativePath), b, 0o644)

		buff := new(bytes.Buffer)
		d := weaponData{
			Config: v,
			Data:   dm,
		}
		if d.StructName == "" {
			d.StructName = "Weapon"
		}
		t.Execute(buff, d)
		src := buff.Bytes()
		dst, err := format.Source(src)
		if err != nil {
			return err
		}
		os.WriteFile(fmt.Sprintf("%v/%v_gen.go", v.RelativePath, v.PackageName), dst, 0o644)
	}

	return nil
}

const tmpl = `// Code generated by "pipeline"; DO NOT EDIT.
package {{.PackageName}}

import (
	_ "embed"

    "github.com/genshinsim/gcsim/pkg/model"
	"google.golang.org/protobuf/encoding/prototext"
)

//go:embed data_gen.textproto
var pbData []byte
var base *model.WeaponData

func init() {
	base = &model.WeaponData{}
	err := prototext.Unmarshal(pbData, base)
	if err != nil {
		panic(err)
	}
}

{{if (not .SkipDataFunc) -}}
func (x *{{.StructName}}) Data() *model.WeaponData {
	return base
}
{{- end}}
`
