// Copyright 2014 Google Inc. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.devtools.build.lib.analysis;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.devtools.build.lib.actions.Artifact;
import com.google.devtools.build.lib.actions.FilesetTraversalParams;
import com.google.devtools.build.lib.packages.OutputFile;
import com.google.devtools.build.lib.rules.fileset.FilesetProvider;
import com.google.devtools.build.lib.vfs.PathFragment;

/**
 * A configured target for output files generated by {@code Fileset} rules. They are almost the
 * same thing as output files except that they implement {@link FilesetProvider} so that
 * {@code Fileset} can figure out the link tree behind them.
 *
 * <p>In an ideal world, this would not be needed: Filesets would depend on other Filesets and not
 * their output directories. However, sometimes a Fileset depends on the output directory of
 * another Fileset. Thus, we need this hack.
 */
public final class FilesetOutputConfiguredTarget extends OutputFileConfiguredTarget
    implements FilesetProvider {
  private final Artifact filesetInputManifest;
  private final PathFragment filesetLinkDir;
  private final ImmutableList<FilesetTraversalParams> filesetTraversals;

  FilesetOutputConfiguredTarget(
      TargetContext targetContext,
      OutputFile outputFile,
      TransitiveInfoCollection generatingRule,
      Artifact outputArtifact,
      ImmutableList<FilesetTraversalParams> traversals) {
    super(targetContext, outputFile, generatingRule, outputArtifact);
    FilesetProvider provider = generatingRule.getProvider(FilesetProvider.class);
    Preconditions.checkArgument(provider != null);
    filesetInputManifest = provider.getFilesetInputManifest();
    filesetLinkDir = provider.getFilesetLinkDir();
    filesetTraversals = traversals;
  }

  @Override
  public Artifact getFilesetInputManifest() {
    return filesetInputManifest;
  }

  @Override
  public PathFragment getFilesetLinkDir() {
    return filesetLinkDir;
  }

  @Override
  public ImmutableList<FilesetTraversalParams> getTraversals() {
    return filesetTraversals;
  }
}
