// Copyright 2014 Google Inc. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.devtools.build.lib.skyframe;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.devtools.build.lib.actions.Action;
import com.google.devtools.build.lib.analysis.Aspect;
import com.google.devtools.build.lib.analysis.AspectWithParameters;
import com.google.devtools.build.lib.analysis.ConfiguredAspectFactory;
import com.google.devtools.build.lib.analysis.config.BuildConfiguration;
import com.google.devtools.build.lib.events.Location;
import com.google.devtools.build.lib.packages.AspectParameters;
import com.google.devtools.build.lib.syntax.Label;
import com.google.devtools.build.skyframe.SkyFunctionName;
import com.google.devtools.build.skyframe.SkyKey;

import javax.annotation.Nullable;

/**
 * An aspect in the context of the Skyframe graph.
 */
public final class AspectValue extends ActionLookupValue {
  /**
   * The key of an action that is generated by an aspect.
   */
  public static final class AspectKey extends ActionLookupKey {
    private final Label label;
    private final BuildConfiguration configuration;
    private final AspectWithParameters aspect;

    private AspectKey(Label label, BuildConfiguration configuration,
        Class<? extends ConfiguredAspectFactory> aspectFactory,
        AspectParameters parameters) {
      Preconditions.checkNotNull(parameters);
      this.label = label;
      this.configuration = configuration;
      this.aspect = new AspectWithParameters(aspectFactory, parameters);
    }

    @Override
    public Label getLabel() {
      return label;
    }

    public BuildConfiguration getConfiguration() {
      return configuration;
    }

    public Class<? extends ConfiguredAspectFactory> getAspect() {
      return aspect.getAspectFactory();
    }

    @Nullable
    public AspectParameters getParameters() {
      return aspect.getParameters();
    }

    @Override
    SkyFunctionName getType() {
      return SkyFunctions.ASPECT;
    }

    @Override
    public int hashCode() {
      return Objects.hashCode(label, configuration, aspect);
    }

    @Override
    public boolean equals(Object other) {
      if (this == other) {
        return true;
      }

      if (!(other instanceof AspectKey)) {
        return false;
      }

      AspectKey that = (AspectKey) other;
      return Objects.equal(label, that.label)
          && Objects.equal(configuration, that.configuration)
          && Objects.equal(aspect, that.aspect);
    }

    @Override
    public String toString() {
      return label + "#" + aspect.getAspectFactory().getSimpleName() + " "
          + (configuration == null ? "null" : configuration.checksum()) + " "
          + aspect.getParameters();
    }
  }

  private final Label label;
  private final Location location;
  private final AspectKey key;
  private final Aspect aspect;

  public AspectValue(
      AspectKey key, Label label, Location location, Aspect aspect, Iterable<Action> actions) {
    super(actions);
    this.location = location;
    this.label = label;
    this.key = key;
    this.aspect = aspect;
  }

  public Aspect getAspect() {
    return aspect;
  }

  public Label getLabel() {
    return label;
  }

  public Location getLocation() {
    return location;
  }

  public AspectKey getKey() {
    return key;
  }

  public static SkyKey key(Label label, BuildConfiguration configuration,
      Class<? extends ConfiguredAspectFactory> aspectFactory,
      AspectParameters additionalConfiguration) {
    return new SkyKey(SkyFunctions.ASPECT,
        new AspectKey(label, configuration, aspectFactory, additionalConfiguration));
  }

  public static SkyKey key(AspectKey aspectKey) {
    return new SkyKey(SkyFunctions.ASPECT, aspectKey);
  }

  public static AspectKey createAspectKey(
      Label label,
      BuildConfiguration configuration,
      Class<? extends ConfiguredAspectFactory> aspectFactory) {
    return new AspectKey(label, configuration, aspectFactory, AspectParameters.EMPTY);
  }
}
