#!/usr/bin/python2.4
#
# Copyright 2007-2009 Google Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ========================================================================

"""
Generates IDL file for the OneClick ActiveX control from the passed-in IDL
template. The input template is a complete IDL file in all but one respect;
It has one replaceable entry for the CLSID for GoopdateOneClickControl.
We generate a GUID using UUIDGEN.EXE, and write out an IDL with a new CLSID.

"""

import sys
import os
import getopt
import commands


def _GetStatusOutput(cmd):
  """Return (status, output) of executing cmd in a shell."""
  if os.name == "nt":
    pipe = os.popen(cmd + " 2>&1", 'r')
    text = pipe.read()
    sts = pipe.close()
    if sts is None: sts = 0
    if text[-1:] == '\n': text = text[:-1]
    return sts, text
  else:
    return commands.getstatusoutput(cmd)


def _GenerateIDLText(idl_template):
  (status, guid) = _GetStatusOutput("uuidgen.exe")
  if status != 0:
    raise SystemExit("Failed to get GUID: %s" % guid)

  return idl_template % guid


def _GenerateIDLFile(idl_template_filename, idl_output_filename):
  f_in = open(idl_template_filename, 'r')
  idl_template = f_in.read()
  f_in.close()

  idl_output = _GenerateIDLText(idl_template)

  f_out = open(idl_output_filename, 'w')
  f_out.write("""
                 // ** AUTOGENERATED FILE. DO NOT HAND-EDIT **
              """)
  f_out.write(idl_output)
  f_out.close()


def _Usage():
  """Prints out script usage information."""
  print """
generate_oneclick_idl.py: Write out the given IDL file.

Usage:
  generate_oneclick_idl.py [--help
                            | --idl_template_file filename
                              --idl_output_file filename]

Options:
  --help                    Show this information.
  --idl_output_file filename     Path/name of output IDL filename.
  --idl_template_file filename      Path/name of input IDL template.
"""


def _Main():
  """Generates IDL file."""
  # use getopt to parse the option and argument list; this may raise, but
  # don't catch it
  _ARGUMENT_LIST = ["help", "idl_template_file=", "idl_output_file="]
  (opts, args) = getopt.getopt(sys.argv[1:], "", _ARGUMENT_LIST)
  if not opts or ("--help", "") in opts:
    _Usage()
    sys.exit()

  idl_template_filename = ""
  idl_output_filename = ""

  for (o, v) in opts:
    if o == "--idl_template_file":
      idl_template_filename = v
    if o == "--idl_output_file":
      idl_output_filename = v

  # make sure we have work to do
  if not idl_template_filename:
    raise SystemExit("no idl_template_filename specified")
  if not idl_output_filename:
    raise SystemExit("no idl_output_filename specified")

  _GenerateIDLFile(idl_template_filename, idl_output_filename)
  sys.exit()


if __name__ == "__main__":
  _Main()

