/*! \file timelib.h
	\brief timelib include file
	A library providing functions for handling various types of time.
*/

//! \ingroup support
//! \defgroup timelib Time handling library
//! Time systems support.
//!
//! The following time systems are supported withing COSMOS:
//! - Coordinated Universal Time (UTC)
//! - Universal Time (UT1)
//! - Terrestrial Time (TT)
//! - GPS Time
//! - Barycentric Dynamical Time (TDB)
//! - Greenwhich Mean Sidereal Time (GMST)
//! - Greenwhich Apparent Sidereal Time (GAST)
//! Except for Sidereal time, these are all represented internally as Modified Julian Day.
//! This library provides functions to convert between these systems, and to represent
//! MJD in various other forms.

#ifndef _TIMELIB_H
#define _TIMELIB_H 1

#include "configCosmos.h"

#include "mathlib.h"
#include "timeutils.hpp"

#include <cmath>
#include <time.h>
#include <errno.h>
#include <stdio.h>
#include <ctime>
#include <ratio>

//! \ingroup timelib
//! \defgroup timelib_constants Time handling constants
//! @{

#define MJD2JD(mjd) (double)((mjd) + 2400000.5)
#define JD2MJD(jd) (double)((jd) - 2400000.5)
//! @}

//! \ingroup timelib
//! \defgroup timelib_typedefs Time handling type definitions
//! @{
typedef struct
{
	double mjd;
	int32_t year;
	int32_t month;
	int32_t day;
	double fd;
	double tt_mjd;
	double gmst_rad;
} timestruc;

//! @}

//! \ingroup timelib
//! \defgroup timelib_functions Time handling functions
//! @{

double currentmjd(double offset);
double currentmjd();

// cal to another format
double  cal2mjd2(int32_t year, int32_t month, double day);
void    cal2mjd( int iy, int im, int id, double *djm, int *j );

// utc to another format
struct timeval utc2unix(double utc);
double utc2epsilon(double mjd);
double utc2depsilon(double mjd);
double utc2dpsi(double mjd);
double utc2L(double mjd);
double utc2Lp(double mjd);
double utc2F(double mjd);
double utc2D(double mjd);
double utc2omega(double mjd);
double utc2tt(double mjd);
double utc2gps(double utc);
double utc2ut1(double mjd);
double utc2dut1(double mjd);
double utc2tdb(double mjd);
double utc2tdb(double mjd);
double utc2gmst(double mjd);
double utc2gast(double mjd);
rvector utc2nuts(double mjd);
double utc2theta(double mjd);
double utc2jcen(double mjd);
string utc2iso8601(double mjd);

// gps to another format
double  gps2utc(double gps);
void    gps2week(double gps, uint32_t& week, double& seconds);
double  week2gps(uint32_t week, double seconds);

// mjd to another format
double  mjd2year(double mjd);
int32_t mjd2ymd(double mjd, int32_t *year, int32_t *month, double *day, double *doy);
void    mjd2cal( double djm, int *iy, int *im, int *id, double *fd, int *j);
string  mjd2human(double mjd);
string  mjd2human2(double mjd);
string  mjd2human3(double mjd);
string  mjd2iso8601(double mjd);
double  mjd2jd(double mjd);
double  jd2mjd(double jd);

// other
double unix2utc(struct timeval unixtime);
double unix2utc(double unixtime);
double  tt2utc(double mjd);
double  tt2tdb(double mjd);
double  julcen(double mjd);
cvector polar_motion(double mjd);
int32_t leap_seconds(double mjd);
double  ranrm(double angle);
int16_t isleap(int32_t year);
int32_t load_iers();
string  seconds2DDHHMMSS(double elapsed_seconds);

// moved to core/libraries/new/elapsedtime.cpp
//// profiling class
//// On windows using MinGw32 it does not get better than 1ms
//class ElapsedTime {
//    // old plain c
//    struct timeval time1, time2;
//    struct timezone x;
//    float timeDiff;

//    // new c++11
//    //std::chrono::high_resolution_clock::time_point time1, time2;

//public:
//    int timeval_subtract (struct timeval* result, struct timeval* x, struct timeval* y);
//    float elapsed_time(struct timeval a,struct timeval b);
//    void printElapsedTime();
//    void printElapsedTime(string text);
//    double getElapsedTimeMiliSeconds();
//    double getElapsedTime();

//    void tic();

//    double toc();
//    //double toc(bool print_flag);
//    double toc(string text);

//    // turn on/off printing
//    bool print = true; //
//    double elapsedTime = 0.;
//};


//! @}

#endif
