# Iterating

Throughout the examples we're going to use this dataset

```php
use Collection\Map;

$data = [
  [
      "id" => 70111470,
      "title" =>  "Die Hard",
      "boxart" => "http://cdn-0.nflximg.com/images/2891/DieHard.jpg",
      "uri" => "http://api.netflix.com/catalog/titles/movies/70111470",
      "rating" =>  [4.0],
      "bookmark" =>  []
  ],
  [
      "id" =>  654356453,
      "title" => "Bad Boys",
      "boxart" =>  "http://cdn-0.nflximg.com/images/2891/BadBoys.jpg",
      "uri" => "http://api.netflix.com/catalog/titles/movies/70111470",
      "rating" =>  [5.0],
      "bookmark" => [{ "id" => 432534, "time" => 65876586 }]
  ],
  [
      "id" => 65432445,
      "title" => "The Chamber",
      "boxart" => "http://cdn-0.nflximg.com/images/2891/TheChamber.jpg",
      "uri" => "http://api.netflix.com/catalog/titles/movies/70111470",
      "rating" => [4.0],
      "bookmark" => []
  ]
];
$videos = new Map($data);
```

## Projecting Collections

Applying a function to a value and creating a new value is called a projection. To project one iterable into another, we apply a projection function to each item in the collection and collect the results in a new collection.

### Each

```php
$videoAndTitlePairs = new Map();

$videos->each(function ($video) {
    $videoAndTitlePairs->addAll(["id" => $video["id"], "title" => $video["title"]]);
});
```

### Map

To make projections easier, lets add a `map()` function to the game. Map accepts the projection function to be applied to each item in the source collection, and returns the projected collection.

Lets use map to project a collection of videos into a collection of [id, title]:

```php
$new = $videos->map(function ($video) {
    return [
		"id" => $video["id"],
		"title" => $video["title"]
	];
});
```

The `map()` method will create a new iterator which lazily creates the resulting items when iterated.

### Filter

Like projection, filtering a collection is also a common operation. To filter a collection we apply a test to each item in the iterable and collect the items that pass into a new iterable.

Lets filter and map to collect the ids of videos that have a rating of 5.0

```php
use Collection\Map;

$topRatedVideos = $videos->filter(function ($video) {
    return $video["rating"] === 5.0;
})->map(function ($video) {
    return $video["rating"] ;
});
```

