# Copyright The IETF Trust 2020, All Rights Reserved
# -*- coding: utf-8 -*-
# Generated by Django 1.11.27 on 2020-01-17 12:32


from django.db import migrations


def create_or_delete_ipr_doc_events(apps, delete=False):
    """Create or delete DocEvents for IprEvents
    
    Mostly duplicates IprEvent.create_doc_events(). This is necessary
    because model methods, including custom save() methods, are not 
    available to migrations.
    """
    IprEvent = apps.get_model('ipr', 'IprEvent')
    DocEvent = apps.get_model('doc', 'DocEvent')
    
    # Map from self.type_id to DocEvent.EVENT_TYPES for types that
    # should be logged as DocEvents
    event_type_map = {
        'posted': 'posted_related_ipr',
        'removed': 'removed_related_ipr',
    }

    for ipr_event in IprEvent.objects.filter(type_id__in=event_type_map):
        related_docs = set()  # related docs, no duplicates
        for alias in ipr_event.disclosure.docs.all():
            related_docs.update(alias.docs.all())
        for doc in related_docs:
            kwargs = dict(
                type=event_type_map[ipr_event.type_id],
                time=ipr_event.time,
                by=ipr_event.by,
                doc=doc,
                rev='',
                desc='%s related IPR disclosure: <b>%s</b>' % (ipr_event.type.name,
                                                               ipr_event.disclosure.title),
            )
            events = DocEvent.objects.filter(**kwargs)  # get existing events
            if delete:
                events.delete()
            elif len(events) == 0:
                DocEvent.objects.create(**kwargs)  # create if did not exist

def forward(apps, schema_editor):
    """Create a DocEvent for each 'posted' or 'removed' IprEvent"""
    create_or_delete_ipr_doc_events(apps, delete=False)

def reverse(apps, schema_editor):
    """Delete DocEvents that would be created by the forward migration
    
    This removes data, but only data that can be regenerated by running
    the forward migration.
    """
    create_or_delete_ipr_doc_events(apps, delete=True)
    
class Migration(migrations.Migration):

    dependencies = [
        ('ipr', '0006_document_primary_key_cleanup'),
        # Ensure the DocEvent types we need exist
        ('doc', '0029_add_ipr_event_types'),
    ]

    operations = [
        migrations.RunPython(forward, reverse),
    ]
