package io.mosip.kernel.biometrics.test;

import static org.hamcrest.CoreMatchers.isA;
import static org.junit.Assert.assertThat;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.json.JSONException;
import org.junit.Test;

import io.mosip.kernel.biometrics.commons.BiometricsSignatureHelper;
import io.mosip.kernel.biometrics.entities.BIR;
import io.mosip.kernel.core.exception.BiometricSignatureValidationException;

public class BiometricsSignatureHelperTest {


	@Test(expected = BiometricSignatureValidationException.class)
	public void extractJWTTokenBiometricSignatureValidationExceptionTest() throws BiometricSignatureValidationException, JSONException {
		BIR bir = new BIR.BIRBuilder().build();
		BiometricsSignatureHelper.extractJWTToken(bir);
	}


	@Test
	public void extractJWTTokenTest() throws IOException, Exception {
		String payload="{\"digitalId\":\"eyJ4NWMiOlsibW9jayJdfQ.ZXlKNE5XTWlPbHNpVFVsSlJpOUVRME5CSzFOblFYZEpRa.JDUVZGelJrRkVRbVpOVVhOM1ExRlpSRlpSVVVkRmQwcEtWR3BGVFUxQmIwZEJNVlZGUTBGM1JGTXdSbE5OVVhkM1EyZFpSRlpSVVVoRVFVNURWRVpKZUVWRVFVOUNaMDVXUWtGdlRVSXdNVkJSTUhST1VrWk5\",\"deviceCode\":\"b692b595-3523-slap-99fc-bd76e35f190f\",\"deviceServiceVersion\":\"0.9.5\",\"bioType\":\"Finger\",\"bioSubType\":\"Right Thumb\",\"purpose\":\"Registration\",\"env\":\"Staging\",\"bioValue\":\"&lt;bioValue&gt;\",\"transactionId\":\"433c4e0e-fe65-463b-a5b3-476750cfd576\",\"timestamp\":\"2021-08-11T10:32:21Z\",\"requestedScore\":\"40\",\"qualityScore\":\"80\"}";
		String sb="eyJ4NWMiOlsibW9jayJdfQ..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";
		String bdb="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";
		/*Entry entry = new Entry("PAYLOAD", payload);
		List<Entry> entries = new ArrayList<>();
		entries.add(entry);*/
		BIR bir = new BIR.BIRBuilder().withOthers("PAYLOAD", payload).withBdb(bdb.getBytes()).withSb(sb.getBytes()).build();
		assertThat(BiometricsSignatureHelper.extractJWTToken(bir), isA(String.class));	
	}
	
	@Test(expected = BiometricSignatureValidationException.class)
	public void extractJWTTokenSignatureMismatch() throws IOException, Exception {
		String payload="{\"digitalId\":\"eyJ4NWMiOlsibW9jayJdfQ.ZXlKNE5XTWlPbHNpVFVsSlJpOUVRME5CSzFOblFYZEpRa.JDUVZGelJrRkVRbVpOVVhOM1ExRlpSRlpSVVVkRmQwcEtWR3BGVFUxQmIwZEJNVlZGUTBGM1JGTXdSbE5OVVhkM1EyZFpSRlpSVVVoRVFVNURWRVpKZUVWRVFVOUNaMDVXUWtGdlRVSXdNVkJSTUhST1VrWk5\",\"deviceCode\":\"b692b595-3523-slap-99fc-bd76e35f190f\",\"deviceServiceVersion\":\"0.9.5\",\"bioType\":\"Finger\",\"bioSubType\":\"Right Thumb\",\"purpose\":\"Registration\",\"env\":\"Staging\",\"bioValue\":\"&lt;bioValue&gt;\",\"transactionId\":\"433c4e0e-fe65-463b-a5b3-476750cfd576\",\"timestamp\":\"2021-08-11T10:32:21Z\",\"requestedScore\":\"40\",\"qualityScore\":\"80\"}";
		String sb="eyJ4NWMiOltdfQ..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";
		String bdb="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";
		/*Entry entry = new Entry("PAYLOAD", payload);
		List<Entry> entries = new ArrayList<>();
		entries.add(entry);*/
		BIR bir = new BIR.BIRBuilder().withOthers("PAYLOAD", payload).withBdb(bdb.getBytes()).withSb(sb.getBytes()).build();
		assertThat(BiometricsSignatureHelper.extractJWTToken(bir), isA(String.class));	
	}
	
	
}
