macro(add_to_alloptions _NEWNAME)
  list(APPEND ALLOPTIONS ${_NEWNAME})
endmacro()

macro(set_option _NAME _DESC)
  add_to_alloptions(${_NAME})
  if(${ARGC} EQUAL 3)
    set(_DEFLT ${ARGV2})
  else()
    set(_DEFLT OFF)
  endif()
  option(${_NAME} ${_DESC} ${_DEFLT})
endmacro()

macro(dep_option _NAME _DESC _DEFLT _DEPTEST _FAILDFLT)
  add_to_alloptions("${_NAME}")
  cmake_dependent_option("${_NAME}" "${_DESC}" "${_DEFLT}" "${_DEPTEST}" "${_FAILDFLT}")
endmacro()

macro(option_string _NAME _DESC _VALUE)
  add_to_alloptions(${_NAME})
  set(${_NAME} ${_VALUE} CACHE STRING "${_DESC}")
  set(HAVE_${_NAME} ${_VALUE})
ENDMACRO()

macro(message_bool_option _NAME _VALUE)
  set(_PAD "\t")
  if(${ARGC} EQUAL 3)
    set(_PAD ${ARGV2})
  endif()
  if(${_VALUE})
    message(STATUS "  ${_NAME}:${_PAD}ON")
  else()
    message(STATUS "  ${_NAME}:${_PAD}OFF")
  endif()
endmacro()

macro(message_tested_option _NAME)
  set(_REQVALUE ${${_NAME}})
  set(_PAD " ")
  if(${ARGC} EQUAL 2)
    set(_PAD ${ARGV1})
  endif()
  string(SUBSTRING "${_NAME}" 0 4 _NAMESTART)
  if(_NAMESTART STREQUAL "SDL_")
    string(SUBSTRING "${_NAME}" 4 -1 _STRIPPEDNAME)
  else()
    set(_STRIPPEDNAME "${_NAME}")
  endif()
  if(NOT HAVE_${_STRIPPEDNAME})
    set(HAVE_${_STRIPPEDNAME} OFF)
  elseif("${HAVE_${_STRIPPEDNAME}}" MATCHES "1|TRUE|YES|Y")
    set(HAVE_${_STRIPPEDNAME} ON)
  endif()
  message(STATUS "  ${_NAME}${_PAD}(Wanted: ${_REQVALUE}): ${HAVE_${_STRIPPEDNAME}}")
endmacro()

function(find_stringlength_longest_item inList outLength)
  set(maxLength 0)
  foreach(item IN LISTS ${inList})
    string(LENGTH "${item}" slen)
    if(slen GREATER maxLength)
      set(maxLength ${slen})
    endif()
  endforeach()
  set("${outLength}" ${maxLength} PARENT_SCOPE)
endfunction()

function(message_dictlist inList)
  find_stringlength_longest_item(${inList} maxLength)
  foreach(name IN LISTS ${inList})
    # Get the padding
    string(LENGTH ${name} nameLength)
    math(EXPR padLength "(${maxLength} + 1) - ${nameLength}")
    string(RANDOM LENGTH ${padLength} ALPHABET " " padding)
    message_tested_option(${name} ${padding})
  endforeach()
endfunction()

if(APPLE)
  include(CheckOBJCSourceCompiles)
  enable_language(OBJC)
endif()

function(SDL_detect_linker)
  if(CMAKE_VERSION VERSION_LESS 3.29)
    if(NOT DEFINED SDL_CMAKE_C_COMPILER_LINKER_ID)
      execute_process(COMMAND ${CMAKE_LINKER} -v OUTPUT_VARIABLE LINKER_OUTPUT ERROR_VARIABLE LINKER_OUTPUT)
      string(REGEX REPLACE "[\r\n]" " " LINKER_OUTPUT "${LINKER_OUTPUT}")
      if(LINKER_OUTPUT MATCHES ".*Microsoft.*")
        set(linker MSVC)
      else()
        set(linker GNUlike)
      endif()
      message(STATUS "Linker identification: ${linker}")
      set(SDL_CMAKE_C_COMPILER_LINKER_ID "${linker}" CACHE STRING "Linker identification")
      mark_as_advanced(SDL_CMAKE_C_COMPILER_LINKER_ID)
    endif()
    set(CMAKE_C_COMPILER_LINKER_ID "${SDL_CMAKE_C_COMPILER_LINKER_ID}" PARENT_SCOPE)
  endif()
endfunction()

function(read_absolute_symlink DEST PATH)
  file(READ_SYMLINK "${PATH}" p)
  if(NOT IS_ABSOLUTE "${p}")
    get_filename_component(pdir "${PATH}" DIRECTORY)
    set(p "${pdir}/${p}")
  endif()
  get_filename_component(p "${p}" ABSOLUTE)
  set("${DEST}" "${p}" PARENT_SCOPE)
endfunction()

function(win32_implib_identify_dll DEST IMPLIB)
  cmake_parse_arguments(ARGS "NOTFATAL" "" "" ${ARGN})
  if(CMAKE_DLLTOOL)
    execute_process(
            COMMAND "${CMAKE_DLLTOOL}" --identify "${IMPLIB}"
            RESULT_VARIABLE retcode
            OUTPUT_VARIABLE stdout
            ERROR_VARIABLE stderr)
    if(NOT retcode EQUAL 0)
      if(NOT ARGS_NOTFATAL)
        message(FATAL_ERROR "${CMAKE_DLLTOOL} failed.")
      else()
        set("${DEST}" "${DEST}-NOTFOUND" PARENT_SCOPE)
        return()
      endif()
    endif()
    string(STRIP "${stdout}" result)
    set(${DEST} "${result}" PARENT_SCOPE)
  elseif(MSVC)
    get_filename_component(CMAKE_C_COMPILER_DIRECTORY "${CMAKE_C_COMPILER}" DIRECTORY CACHE)
    find_program(CMAKE_DUMPBIN NAMES dumpbin PATHS "${CMAKE_C_COMPILER_DIRECTORY}")
    if(CMAKE_DUMPBIN)
      execute_process(
              COMMAND "${CMAKE_DUMPBIN}" "-headers" "${IMPLIB}"
              RESULT_VARIABLE retcode
              OUTPUT_VARIABLE stdout
              ERROR_VARIABLE stderr)
      if(NOT retcode EQUAL 0)
        if(NOT ARGS_NOTFATAL)
          message(FATAL_ERROR "dumpbin failed.")
        else()
          set(${DEST} "${DEST}-NOTFOUND" PARENT_SCOPE)
          return()
        endif()
      endif()
      string(REGEX MATCH "DLL name[ ]+:[ ]+([^\n]+)\n" match "${stdout}")
      if(NOT match)
        if(NOT ARGS_NOTFATAL)
          message(FATAL_ERROR "dumpbin did not find any associated dll for ${IMPLIB}.")
        else()
          set(${DEST} "${DEST}-NOTFOUND" PARENT_SCOPE)
          return()
        endif()
      endif()
      set(result "${CMAKE_MATCH_1}")
      set(${DEST} "${result}" PARENT_SCOPE)
    else()
      message(FATAL_ERROR "Cannot find dumpbin, please set CMAKE_DUMPBIN cmake variable")
    endif()
  else()
    if(NOT ARGS_NOTFATAL)
      message(FATAL_ERROR "Don't know how to identify dll from import library. Set CMAKE_DLLTOOL (for mingw) or CMAKE_DUMPBIN (for MSVC)")
    else()
      set(${DEST} "${DEST}-NOTFOUND")
    endif()
  endif()
endfunction()

function(get_actual_target)
  set(dst "${ARGV0}")
  set(target "${${dst}}")
  set(input "${target}")
  get_target_property(alias "${target}" ALIASED_TARGET)
  while(alias)
    set(target "${alias}")
    get_target_property(alias "${target}" ALIASED_TARGET)
  endwhile()
  message(DEBUG "get_actual_target(\"${input}\") -> \"${target}\"")
  set("${dst}" "${target}" PARENT_SCOPE)
endfunction()

function(target_get_dynamic_library DEST TARGET)
  set(result)
  get_actual_target(TARGET)
  if(WIN32)
    # Use the target dll of the import library
    set(props_to_check IMPORTED_IMPLIB)
    if(CMAKE_BUILD_TYPE)
      list(APPEND props_to_check IMPORTED_IMPLIB_${CMAKE_BUILD_TYPE})
    endif()
    list(APPEND props_to_check IMPORTED_LOCATION)
    if(CMAKE_BUILD_TYPE)
      list(APPEND props_to_check IMPORTED_LOCATION_${CMAKE_BUILD_TYPE})
    endif()
    foreach (config_type ${CMAKE_CONFIGURATION_TYPES} RELEASE DEBUG RELWITHDEBINFO MINSIZEREL)
      list(APPEND props_to_check IMPORTED_IMPLIB_${config_type})
      list(APPEND props_to_check IMPORTED_LOCATION_${config_type})
    endforeach()

    foreach(prop_to_check ${props_to_check})
      if(NOT result)
        get_target_property(propvalue "${TARGET}" ${prop_to_check})
        if(propvalue AND EXISTS "${propvalue}")
          win32_implib_identify_dll(result "${propvalue}" NOTFATAL)
        endif()
      endif()
    endforeach()
  else()
    # 1. find the target library a file might be symbolic linking to
    # 2. find all other files in the same folder that symolic link to it
    # 3. sort all these files, and select the 1st item on Linux, and last on Macos
    set(location_properties IMPORTED_LOCATION)
    if(CMAKE_BUILD_TYPE)
      list(APPEND location_properties IMPORTED_LOCATION_${CMAKE_BUILD_TYPE})
    endif()
    foreach (config_type ${CMAKE_CONFIGURATION_TYPES} RELEASE DEBUG RELWITHDEBINFO MINSIZEREL)
      list(APPEND location_properties IMPORTED_LOCATION_${config_type})
    endforeach()
    if(APPLE)
      set(valid_shared_library_regex "\\.[0-9]+\\.dylib$")
    else()
      set(valid_shared_library_regex "\\.so\\.([0-9.]+)?[0-9]")
    endif()
    foreach(location_property ${location_properties})
      if(NOT result)
        get_target_property(library_path "${TARGET}" ${location_property})
        message(DEBUG "get_target_property(${TARGET} ${location_propert}) -> ${library_path}")
        if(EXISTS "${library_path}")
          get_filename_component(library_path "${library_path}" ABSOLUTE)
          while (IS_SYMLINK "${library_path}")
            read_absolute_symlink(library_path "${library_path}")
          endwhile()
          message(DEBUG "${TARGET} -> ${library_path}")
          get_filename_component(libdir "${library_path}" DIRECTORY)
          file(GLOB subfiles "${libdir}/*")
          set(similar_files "${library_path}")
          foreach(subfile ${subfiles})
            if(IS_SYMLINK "${subfile}")
              read_absolute_symlink(subfile_target "${subfile}")
              while(IS_SYMLINK "${subfile_target}")
                read_absolute_symlink(subfile_target "${subfile_target}")
              endwhile()
              get_filename_component(subfile_target "${subfile_target}" ABSOLUTE)
              if(subfile_target STREQUAL library_path AND subfile MATCHES "${valid_shared_library_regex}")
                list(APPEND similar_files "${subfile}")
              endif()
            endif()
          endforeach()
          list(SORT similar_files)
          message(DEBUG "files that are similar to \"${library_path}\"=${similar_files}")
          if(APPLE)
            list(REVERSE similar_files)
          endif()
          list(GET similar_files 0 item)
          get_filename_component(result "${item}" NAME)
        endif()
      endif()
    endforeach()
  endif()
  if(result)
    string(TOLOWER "${result}" result_lower)
    if(WIN32 OR OS2)
      if(NOT result_lower MATCHES ".*dll")
        message(FATAL_ERROR "\"${result}\" is not a .dll library")
      endif()
    elseif(APPLE)
      if(NOT result_lower MATCHES ".*dylib.*")
        message(FATAL_ERROR "\"${result}\" is not a .dylib shared library")
      endif()
    else()
      if(NOT result_lower MATCHES ".*so.*")
        message(FATAL_ERROR "\"${result}\" is not a .so shared library")
      endif()
    endif()
  else()
    get_target_property(target_type ${TARGET} TYPE)
    if(target_type MATCHES "SHARED_LIBRARY|MODULE_LIBRARY")
      # OK
    elseif(target_type MATCHES "STATIC_LIBRARY|OBJECT_LIBRARY|INTERFACE_LIBRARY|EXECUTABLE")
      message(SEND_ERROR "${TARGET} is not a shared library, but has type=${target_type}")
    else()
      message(WARNING "Unable to extract dynamic library from target=${TARGET}, type=${target_type}.")
    endif()
    # TARGET_SONAME_FILE is not allowed for DLL target platforms.
    if(WIN32)
      set(result "$<TARGET_FILE_NAME:${TARGET}>")
    else()
      set(result "$<TARGET_SONAME_FILE_NAME:${TARGET}>")
    endif()
  endif()
  set(${DEST} ${result} PARENT_SCOPE)
endfunction()

function(check_linker_supports_version_file VAR)
  SDL_detect_linker()
  if(CMAKE_C_COMPILER_LINKER_ID MATCHES "^(MSVC)$")
    set(LINKER_SUPPORTS_VERSION_SCRIPT FALSE)
  else()
    cmake_push_check_state(RESET)
    file(WRITE "${CMAKE_CURRENT_BINARY_DIR}/dummy.sym" "n_0 {\n global:\n  func;\n local: *;\n};\n")
    list(APPEND CMAKE_REQUIRED_LINK_OPTIONS "-Wl,--version-script=${CMAKE_CURRENT_BINARY_DIR}/dummy.sym")
    check_c_source_compiles("int func(void) {return 0;} int main(int argc,char*argv[]){(void)argc;(void)argv;return func();}" LINKER_SUPPORTS_VERSION_SCRIPT FAIL_REGEX "(unsupported|syntax error|unrecognized option)")
    cmake_pop_check_state()
  endif()
  set(${VAR} "${LINKER_SUPPORTS_VERSION_SCRIPT}" PARENT_SCOPE)
endfunction()

if(CMAKE_VERSION VERSION_LESS 3.18)
  function(check_linker_flag LANG FLAG VAR)
    cmake_push_check_state(RESET)
    list(APPEND CMAKE_REQUIRED_LINK_OPTIONS ${FLAG})
    if(LANG STREQUAL "C")
      include(CheckCSourceCompiles)
      check_c_source_compiles("int main(int argc,char*argv[]){(void)argc;(void)argv;return 0;}" ${VAR} FAIL_REGEX "(unsupported|syntax error)")
    elseif(LANG STREQUAL "CXX")
      include(CheckCXXSourceCompiles)
      check_cxx_source_compiles("int main(int argc,char*argv[]){(void)argc;(void)argv;return 0;}" ${VAR} FAIL_REGEX "(unsupported|syntax error)")
    else()
      message(FATAL_ERROR "Unsupported language: ${LANG}")
    endif()
    cmake_pop_check_state()
  endfunction()
else()
  cmake_policy(SET CMP0057 NEW)  # Support new if() IN_LIST operator. (used inside check_linker_flag, used in CMake 3.18)
  include(CheckLinkerFlag)
endif()

if(APPLE)
  check_language(OBJC)
  if(NOT CMAKE_OBJC_COMPILER)
    message(WARNING "Cannot find working OBJC compiler.")
  endif()
endif()

function(SDL_PrintSummary)
  ##### Info output #####
  message(STATUS "")
  message(STATUS "SDL3 was configured with the following options:")
  message(STATUS "")
  message(STATUS "Platform: ${CMAKE_SYSTEM}")
  message(STATUS "64-bit:   ${ARCH_64}")
  message(STATUS "Compiler: ${CMAKE_C_COMPILER}")
  message(STATUS "Revision: ${SDL_REVISION}")
  message(STATUS "Vendor:   ${SDL_VENDOR_INFO}")
  message(STATUS "")
  message(STATUS "Subsystems:")

  find_stringlength_longest_item(SDL_SUBSYSTEMS maxLength)
  foreach(_SUB IN LISTS SDL_SUBSYSTEMS)
    string(LENGTH ${_SUB} _SUBLEN)
    math(EXPR _PADLEN "(${maxLength} + 1) - ${_SUBLEN}")
    string(RANDOM LENGTH ${_PADLEN} ALPHABET " " _PADDING)
    string(TOUPPER ${_SUB} _OPT)
    message_bool_option(${_SUB} SDL_${_OPT} ${_PADDING})
  endforeach()
  message(STATUS "")
  message(STATUS "Options:")
  list(SORT ALLOPTIONS)
  message_dictlist(ALLOPTIONS)
  message(STATUS "")
  message(STATUS " Build Shared Library: ${SDL_SHARED}")
  message(STATUS " Build Static Library: ${SDL_STATIC}")
  if(SDL_STATIC)
    message(STATUS " Build Static Library with Position Independent Code: ${SDL_STATIC_PIC}")
  endif()
  if(APPLE)
    message(STATUS " Build libraries as Apple Framework: ${SDL_FRAMEWORK}")
  endif()
  message(STATUS "")
  if(UNIX)
    message(STATUS "If something was not detected, although the libraries")
    message(STATUS "were installed, then make sure you have set the")
    message(STATUS "CMAKE_C_FLAGS and CMAKE_PREFIX_PATH CMake variables correctly.")
    message(STATUS "")
  endif()

  if(WARN_ABOUT_ARM_SIMD_ASM_MIT)
    message(STATUS "SDL is being built with ARM SIMD optimizations, which")
    message(STATUS "uses code licensed under the MIT license. If this is a")
    message(STATUS "problem, please disable that code by rerunning CMake with:")
    message(STATUS "")
    message(STATUS "    -DSDL_ARMSIMD=OFF")
    message(STATUS "")
  endif()

  if(WARN_ABOUT_ARM_NEON_ASM_MIT)
    message(STATUS "SDL is being built with ARM NEON optimizations, which")
    message(STATUS "uses code licensed under the MIT license. If this is a")
    message(STATUS "problem, please disable that code by rerunning CMake with:")
    message(STATUS "")
    message(STATUS "    -DSDL_ARMNEON=OFF")
    message(STATUS "")
  endif()
endfunction()

function(SDL_install_pdb TARGET DIRECTORY)
  get_property(type TARGET ${TARGET} PROPERTY TYPE)
  if(type MATCHES "^(SHARED_LIBRARY|EXECUTABLE)$")
    install(FILES $<TARGET_PDB_FILE:${TARGET}> DESTINATION "${DIRECTORY}" OPTIONAL)
  elseif(type STREQUAL "STATIC_LIBRARY")
    # FIXME: Use $<TARGET_COMPILE_PDB_FILE:${TARGET} once it becomes available (https://gitlab.kitware.com/cmake/cmake/-/issues/25244)
    if(CMAKE_GENERATOR MATCHES "^Visual Studio.*")
      install(CODE "file(INSTALL DESTINATION \"\${CMAKE_INSTALL_PREFIX}/${DIRECTORY}\" TYPE FILE OPTIONAL FILES \"${CMAKE_CURRENT_BINARY_DIR}/\${CMAKE_INSTALL_CONFIG_NAME}/${TARGET}.pdb\")")
    else()
      install(CODE "file(INSTALL DESTINATION \"\${CMAKE_INSTALL_PREFIX}/${DIRECTORY}\" TYPE FILE OPTIONAL FILES \"${CMAKE_CURRENT_BINARY_DIR}/CMakeFiles/${TARGET}.dir/${TARGET}.pdb\")")
    endif()
  endif()
endfunction()
