// Licensed to Elasticsearch B.V. under one or more contributor
// license agreements. See the NOTICE file distributed with
// this work for additional information regarding copyright
// ownership. Elasticsearch B.V. licenses this file to you under
// the Apache License, Version 2.0 (the "License"); you may
// not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package add_cloud_metadata

import (
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/assert"

	"github.com/elastic/beats/libbeat/beat"
	"github.com/elastic/beats/libbeat/common"
	"github.com/elastic/beats/libbeat/logp"
)

func initOpenstackNovaTestServer() *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.RequestURI == osMetadataInstanceIDURI {
			w.Write([]byte("i-0000ffac"))
			return
		}
		if r.RequestURI == osMetadataInstanceTypeURI {
			w.Write([]byte("m1.xlarge"))
			return
		}
		if r.RequestURI == osMetadataHostnameURI {
			w.Write([]byte("testvm01.stack.cloud"))
			return
		}
		if r.RequestURI == osMetadataZoneURI {
			w.Write([]byte("az-test-2"))
			return
		}

		http.Error(w, "not found", http.StatusNotFound)
	}))
}

func TestRetrieveOpenstackNovaMetadata(t *testing.T) {
	logp.TestingSetup()

	server := initOpenstackNovaTestServer()
	defer server.Close()

	config, err := common.NewConfigFrom(map[string]interface{}{
		"host": server.Listener.Addr().String(),
	})

	if err != nil {
		t.Fatal(err)
	}

	p, err := newCloudMetadata(config)
	if err != nil {
		t.Fatal(err)
	}

	actual, err := p.Run(&beat.Event{Fields: common.MapStr{}})
	if err != nil {
		t.Fatal(err)
	}

	expected := common.MapStr{
		"meta": common.MapStr{
			"cloud": common.MapStr{
				"provider":          "openstack",
				"instance_id":       "i-0000ffac",
				"instance_name":     "testvm01.stack.cloud",
				"machine_type":      "m1.xlarge",
				"availability_zone": "az-test-2",
			},
		},
	}
	assert.Equal(t, expected, actual.Fields)
}
