﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace ImageResizer.Plugins.LicenseVerifier.Tests
{
    static class LicenseStrings
    {
        public class LicenseSet
        {
            public string Name { get; set; }
            public string Placeholder { get; set; }
            public string Remote { get; set;  }

        }

        public static IEnumerable<LicenseSet> GetSets(params string[] names) => names.Select(name => new LicenseSet {
            Name = name,
            Placeholder = Licenses[name + "Placeholder"],
            Remote = Licenses[name + "Remote"]
        });
    

        public const string Offlinev4DomainAcmeComCreative =
                "acme.com(R4Creative includes R4Creative R4Performance):S2luZDogdjQtZG9tYWluLW9mZmxpbmUKU2t1OiBSNENyZWF0aXZlCkRvbWFpbjogYWNtZS5jb20KT3duZXI6IEFjbWUgQ29ycApJc3N1ZWQ6IDIwMTctMDQtMjFUMDA6MDA6MDArMDA6MDAKRmVhdHVyZXM6IFI0Q3JlYXRpdmUgUjRQZXJmb3JtYW5jZQ==:eFLsTwUCEdQiEt34zdnzzxKFeEoAOrZoheE85LLYB9Pgx5wypsYpcG+58GlXUtgldbPyq+9e+m/ZeDhyqXPUkd6wk43EqUu07//20RE3XEWeKEGK1LBTNUJ6gfL9iPsA9qnSLpJNV7QLp9JxWI2VztvPUol9W5dORtUWtfzna+hujSQ5lym9vjVBaxsbsyRBS9x27lzGKUL+RoonHDYpeIolAnNu28WuBmFGQ3S3ALcNZ4dSjoapyAXQyEH07A5pQ/p18Vv5FqD24p7dh45BGMqJXLVuZli13kvdh812UQvKwyL223k9cEYiyV7F+YN6YHPL5/Ebrh1nYDC00/1b7A=="
            ;

        public static readonly string EliteSubscriptionPlaceholder = SiteWidePlaceholder;
        public static readonly string EliteSubscriptionRemote = SiteWideRemote;
        public static readonly string V4ElitePerpetualPlaceholder = V4EliteRemotePlaceholder;
        public static readonly string V4ElitePerpetualRemote = V4EliteRemoteRemote;


        //Autogenerated below this point - do not edit - use imazen/licensing tests/support/licenses/license_strings.cs
        public static IReadOnlyDictionary<string, string> Licenses { get; } = new Dictionary<string, string> {
            {"V4EliteRemotePlaceholder", V4EliteRemotePlaceholder},
            {"V4EliteRemoteRemote", V4EliteRemoteRemote},
            {"V4DomainRemoteSubscriptionPlaceholder", V4DomainRemoteSubscriptionPlaceholder},
            {"V4DomainRemoteSubscriptionRemote", V4DomainRemoteSubscriptionRemote},
            {"HardRevocationPlaceholder", HardRevocationPlaceholder},
            {"HardRevocationRemote", HardRevocationRemote},
            {"SoftRevocationPlaceholder", SoftRevocationPlaceholder},
            {"SoftRevocationRemote", SoftRevocationRemote},
            {"PerpetualPlaceholder", PerpetualPlaceholder},
            {"PerpetualRemote", PerpetualRemote},
            {"CancelledPlaceholder", CancelledPlaceholder},
            {"CancelledRemote", CancelledRemote},
            {"SiteWidePlaceholder", SiteWidePlaceholder},
            {"SiteWideRemote", SiteWideRemote},
            {"OemPlaceholder", OemPlaceholder},
            {"OemRemote", OemRemote},
            {"CoresPlaceholder", CoresPlaceholder},
            {"CoresRemote", CoresRemote},
            {"PerCore2DomainsPlaceholder", PerCore2DomainsPlaceholder},
            {"PerCore2DomainsRemote", PerCore2DomainsRemote},
            {"XmlPlaceholder", XmlPlaceholder},
            {"XmlRemote", XmlRemote},
        };

        /// Kind: id
        /// Id: 4271246357
        /// Secret: test150c96fee33ba1cc659e9d6170801ed127bd114f5694e78c31483b374a036542
        /// Owner: Acme Corp
        /// Issued: 2017-04-21T00:00:00+00:00
        /// NetworkGraceMinutes: 480
        /// IsPublic: false
        public const string V4EliteRemotePlaceholder =
                "License 4271246357 for Acme Corp :S2luZDogaWQKSWQ6IDQyNzEyNDYzNTcKU2VjcmV0OiB0ZXN0MTUwYzk2ZmVlMzNiYTFjYzY1OWU5ZDYxNzA4MDFlZDEyN2JkMTE0ZjU2OTRlNzhjMzE0ODNiMzc0YTAzNjU0MgpPd25lcjogQWNtZSBDb3JwCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApOZXR3b3JrR3JhY2VNaW51dGVzOiA0ODAKSXNQdWJsaWM6IGZhbHNl:IfPITv4m0XcvBBrqdgzkEhohvlt/yeblofnu8G9vDOuNjKKhndrCn2/DKDLb8/kJ7+4Ckjt6fPdMOMZVXhIP+UFichdIaBxPruvf2V+wmn7yQtJ9TgUDjeSquz3rZdd1MkQqTo971bYkMMLfvuOcTcj5e4NLzvBH/pfKdMewLf0XO/nEYJWdMvrjxyjOToLVubyZ3KkzT/28coPR2902iUIsj8pLHsYNAPopceYpKRQ1QhTFeVzF5ed2ErFofZxuPwOiQirKrl89N3qMcbY1Ie0N0P5sBgNXF5hxckdLjYaajfvRZFunUyJVZh7ckr/GPZE6jjHWnA09bwy3ZHen4w=="
            ;

        /// Id: 4271246357
        /// Owner: Acme Corp
        /// Features: R4Elite R4Creative R4Performance
        /// Kind: v4-elite
        /// Issued: 2017-04-21T00:00:00+00:00
        /// IsPublic: true
        /// MustBeFetched: true
        public const string V4EliteRemoteRemote =
                "License 4271246357 for Acme Corp :SWQ6IDQyNzEyNDYzNTcKT3duZXI6IEFjbWUgQ29ycApGZWF0dXJlczogUjRFbGl0ZSBSNENyZWF0aXZlIFI0UGVyZm9ybWFuY2UKS2luZDogdjQtZWxpdGUKSXNzdWVkOiAyMDE3LTA0LTIxVDAwOjAwOjAwKzAwOjAwCklzUHVibGljOiB0cnVlCk11c3RCZUZldGNoZWQ6IHRydWU=:GwFW2I/5zJTSNPO/doHFZAeHFu0alPznCZs6s/tlYQArH5RsGrTNbz19NwJaBgQUCEBetZkuPJpIvZQQxmTadXwKO9/TOSR/ntLxJ39axP6MAH0560jXfXXhp4bMhMMxcZeOgSZe2iIaJ/BkbiVxCJC1MMktLDqY6LV9lhvxiaqxBQnBK5eHJ4YjWCL+gX1nVBp+5fAPb8h7GurOzkd7ZOFaoCIJNxwZY766R9EGY4N3bgQY1gDkFTF0gu1AmVB3CWvl2gF/tIGjqXw5qXhIwChnQgoboEnCegQ3KMsT+uIUlKcBBsCdeazT0QjYzrfiHOObMEJoRWNSrEP2WPOKZA=="
            ;

        /// Kind: id
        /// Id: 3168226392
        /// Secret: test584cd7bcd3980539d2ef8c43960aacf906733ce66d44896852b0f813e52e5353
        /// Owner: Acme Corp
        /// Issued: 2017-04-21T00:00:00+00:00
        /// NetworkGraceMinutes: 480
        /// IsPublic: false
        public const string V4DomainRemoteSubscriptionPlaceholder =
                "License 3168226392 for Acme Corp :S2luZDogaWQKSWQ6IDMxNjgyMjYzOTIKU2VjcmV0OiB0ZXN0NTg0Y2Q3YmNkMzk4MDUzOWQyZWY4YzQzOTYwYWFjZjkwNjczM2NlNjZkNDQ4OTY4NTJiMGY4MTNlNTJlNTM1MwpPd25lcjogQWNtZSBDb3JwCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApOZXR3b3JrR3JhY2VNaW51dGVzOiA0ODAKSXNQdWJsaWM6IGZhbHNl:h/867OFFncop2mz0ydKhqWa/sM6ROpiEzprcQWga/fgiS52WrAsarAYbOZePgAO95Fry2upkdrus3DCnx8f4gc5TyZfgcIHVx4rLdZ13WVwo6X0WgAxxAYAetkiLiMOvSO44tXZiR8fa9KVosvzdM7WIytPwk5HbcdhIkg9Igp5KAegV9HgIRj4gtWkYFNrRlw77csQWejcb0cBt1ryOwekQZavS05vrKTeMw8pktyL7/TWGqxgFYkCcgxB+DUXVLG9qYB1mw0W0AGqJgHg9QpDsF11TtczByNbM14VN9KQPCNcbeuliJU/cAOlYPbFA6XjYF01VLe0MWEysdWRaKA=="
            ;

        /// Id: 3168226392
        /// Owner: Acme Corp
        /// Domains: box.com stamps.com
        /// Features: R4Performance
        /// Kind: v4-domain
        /// Issued: 2017-04-21T00:00:00+00:00
        /// Expires: 2017-06-05T00:00:00+00:00
        /// IsPublic: true
        /// MustBeFetched: true
        public const string V4DomainRemoteSubscriptionRemote =
                "License 3168226392 for Acme Corp :SWQ6IDMxNjgyMjYzOTIKT3duZXI6IEFjbWUgQ29ycApEb21haW5zOiBib3guY29tIHN0YW1wcy5jb20KRmVhdHVyZXM6IFI0UGVyZm9ybWFuY2UKS2luZDogdjQtZG9tYWluCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApFeHBpcmVzOiAyMDE3LTA2LTA1VDAwOjAwOjAwKzAwOjAwCklzUHVibGljOiB0cnVlCk11c3RCZUZldGNoZWQ6IHRydWU=:HzSUkDjYxXLcDl6ZSZ3gZZXlGHHSa2BOZjlC3u3r//jvw9I4SVAjZCAMGkxU1IaeFMLJr+4LpWnUPFk/sSxjE9FQc6ZMM8LaQXAUlt7nKe6nA7MivOb6O0iyWXhbLHh4SKt4pbTMAvskTZ4q93suxIxxXbFiDy+DJR2dpFfH49Wj5I6UaUcxnfb4CyOVYn5yMLIv0oWsUEggU1HdtkAva/uMI6Ou4AIU6+tul1JeLJg634PlLRYgjdpg7DwhoPKQKkZQ1a/Jk9RQ/cHTg3PmhM4yJ3iyglSo3zBm8QZqxt94mXKVcwfBJQVG39wdEw0bVpb+xE2IIBzUHv38XFWdtQ=="
            ;

        /// Kind: id
        /// Id: 1812344071
        /// Secret: test072d066cb21a8f3a732a274af7ad525acf845f24d8bad1c6c3bc83659ea1a219
        /// Owner: Acme Corp
        /// Issued: 2017-04-21T00:00:00+00:00
        /// NetworkGraceMinutes: 480
        /// IsPublic: false
        public const string HardRevocationPlaceholder =
                "License 1812344071 for Acme Corp :S2luZDogaWQKSWQ6IDE4MTIzNDQwNzEKU2VjcmV0OiB0ZXN0MDcyZDA2NmNiMjFhOGYzYTczMmEyNzRhZjdhZDUyNWFjZjg0NWYyNGQ4YmFkMWM2YzNiYzgzNjU5ZWExYTIxOQpPd25lcjogQWNtZSBDb3JwCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApOZXR3b3JrR3JhY2VNaW51dGVzOiA0ODAKSXNQdWJsaWM6IGZhbHNl:TDaZw8DKc0slNaw346gBa5buswsQH7grE4j4OyNlxWQl4WIvGVRin2wVLHEeMXI0FYX3lGxXG+TlSbtD17RuYzUCed6IqdjYZXRWA0UPLM149BIOu97zGu0zslHNMpPJ1EkVVWCg9kL+9whlcI3HJ5CHDUo8/Ck6ixwpBRkM4DhfmvuH7IzlU/Wv8D+TW5zw3Xr67OhJSZU4hUegXh0j+OPNi9BmvOYf2Y3iYvpeqquuAhlntdi01zSjs8t55otHHAAz6OY2/nPsET/B9AsgxTSVYoyLuLFwadm0Y+1ImL4vWQAIDUsorG63Dc9GlcMYd1W3HaFzhwBg6Ss5i9F99w=="
            ;

        /// Id: 1812344071
        /// Owner: Acme Corp
        /// Features: R_Elite R_Creative R_Performance
        /// Product: Enterprise-wide license
        /// Kind: site-wide
        /// Issued: 2017-04-21T00:00:00+00:00
        /// IsPublic: true
        /// Valid: false
        /// Message: Please contact support; the license was shared with an unauthorized party and has been revoked.
        public const string HardRevocationRemote =
                "License 1812344071 for Acme Corp (Enterprise-wide license) :SWQ6IDE4MTIzNDQwNzEKT3duZXI6IEFjbWUgQ29ycApGZWF0dXJlczogUl9FbGl0ZSBSX0NyZWF0aXZlIFJfUGVyZm9ybWFuY2UKUHJvZHVjdDogRW50ZXJwcmlzZS13aWRlIGxpY2Vuc2UKS2luZDogc2l0ZS13aWRlCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApJc1B1YmxpYzogdHJ1ZQpWYWxpZDogZmFsc2UKTWVzc2FnZTogUGxlYXNlIGNvbnRhY3Qgc3VwcG9ydDsgdGhlIGxpY2Vuc2Ugd2FzIHNoYXJlZCB3aXRoIGFuIHVuYXV0aG9yaXplZCBwYXJ0eSBhbmQgaGFzIGJlZW4gcmV2b2tlZC4=:kVRcjlOZyRvoUUZ3F0ecVoAia2z0tOraoSPhcrX+dsKEkYERFCIayPY1Wov/gfeaSKOh4opHwqu2iEt/xz5SS63H8MYRbSoL4mnN5zHVVYfiRiYBEnBjNWyKmkqxfGb0ATqwPV4Wgz0Ggh7hXaPVCkpNUq/59fQrzPmXbQQPKlHeX4Myh713SRXmSHJIcc8S1Pk6FFxRwk5I8PhS2CiQ9XkWsU4uFFAIlNrcstF27XZY4188EWjvcks15kOPVh9hjrJD2noX5xTUb0AHf0BOiviptomYd0LAmMSIW4/gvTkANSmslw5PyGIY2W9TZ7maQnG6LeIbv/ZQotj++OeVjg=="
            ;

        /// Kind: id
        /// Id: 2037987837
        /// Secret: testfd397979e7652bd334a729947c3b651d0ef079535b7100dc866085ed93c7c766
        /// Owner: Acme Corp
        /// Issued: 2017-04-21T00:00:00+00:00
        /// NetworkGraceMinutes: 480
        /// IsPublic: false
        public const string SoftRevocationPlaceholder =
                "License 2037987837 for Acme Corp :S2luZDogaWQKSWQ6IDIwMzc5ODc4MzcKU2VjcmV0OiB0ZXN0ZmQzOTc5NzllNzY1MmJkMzM0YTcyOTk0N2MzYjY1MWQwZWYwNzk1MzViNzEwMGRjODY2MDg1ZWQ5M2M3Yzc2NgpPd25lcjogQWNtZSBDb3JwCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApOZXR3b3JrR3JhY2VNaW51dGVzOiA0ODAKSXNQdWJsaWM6IGZhbHNl:P0camMLGAgfvI26ItziGTdNZJYpkmEXNW1G2B5Vv2XBpvWTal5oLVDA0/H5LKtk4PAigTvx+waKuFefcksyuWYjVZDkF/vdzvwy4q6sws4SClV6YlrQgTWmgUlIlrA6BqYsl53zuyUnMJglCzt2o/G03h0cLPetYiGfUELInPBD47S0gz4uCbZsePQ77Gm8+Ul/k4WSDPs5yLacyFdQwFOB8BHZjP4Y7NvuoHwxNOHUYFd4n+0TU3xAaq1iK67MSqIUqeTftFO/8BGReOlVQkKQENRb6TNaBfVvn/yvwIIcC7zX/BrFHCdfRLm08/6ruFZMCeoMLo0i6wiAzyXe2pQ=="
            ;

        /// Id: 2037987837
        /// Owner: Acme Corp
        /// Features: R_Elite R_Creative R_Performance
        /// Product: OEM redistribution license
        /// Kind: oem
        /// Issued: 2017-04-21T00:00:00+00:00
        /// Expires: 2017-06-05T00:00:00+00:00
        /// IsPublic: true
        /// MustBeFetched: true
        /// SubscriptionExpirationDate: 2017-04-21T00:00:00+00:00
        /// Message: This license has been compromised; please contact Vendor Gamma for an updated license
        /// Restrictions: Only for use within Vendor Gamma Product
        public const string SoftRevocationRemote =
                "License 2037987837 for Acme Corp (OEM redistribution license) :SWQ6IDIwMzc5ODc4MzcKT3duZXI6IEFjbWUgQ29ycApGZWF0dXJlczogUl9FbGl0ZSBSX0NyZWF0aXZlIFJfUGVyZm9ybWFuY2UKUHJvZHVjdDogT0VNIHJlZGlzdHJpYnR1aW9uIGxpY2Vuc2UKS2luZDogb2VtCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApFeHBpcmVzOiAyMDE3LTA2LTA1VDAwOjAwOjAwKzAwOjAwCklzUHVibGljOiB0cnVlCk11c3RCZUZldGNoZWQ6IHRydWUKU3Vic2NyaXB0aW9uRXhwaXJhdGlvbkRhdGU6IDIwMTctMDQtMjFUMDA6MDA6MDArMDA6MDAKTWVzc2FnZTogVGhpcyBsaWNlbnNlIGhhcyBiZWVuIGNvbXByb21pc2VkOyBwbGVhc2UgY29udGFjdCBWZW5kb3IgR2FtbWEgZm9yIGFuIHVwZGF0ZWQgbGljZW5zZQpSZXN0cmljdGlvbnM6IE9ubHkgZm9yIHVzZSB3aXRoaW4gVmVuZG9yIEdhbW1hIFByZHVjdA==:i/OYk3IhtPc23zMrBU/J793bfmc9bhEXwHRwq7uV9Ec3EO0fAIPtwlCa0ovn7rx1O1hKo941hkY4CiJCIMEzPhbGYXszdUXrOScpxXZjs22iFLt2pL8I8YriZrveAqlwTa+Q/dU+hOyNEPyI0fTtO+Bw2vctyzS/HlprsEPM/vAF9k3DZCW8HETysVmO1itrKGKBiUZsT0r4a0TFlEO6EzZcidRKWmuR1+sengAKUV30driF0D4uhhPqt+Cyc4K4rb00uVDRdfq1cVZq07pVmbtM6RntHvtOYUAM+fIQBnnaWseofNBv2SV/4O0SkwEoKGAHgET51dLaIuFT5F3Fsg=="
            ;

        /// Kind: id
        /// Id: 590782408
        /// Secret: testc89f3623c7a2359a643c985bfde219b9bed09d48afc54931143331955b98315f
        /// Owner: Acme Corp
        /// Issued: 2017-04-21T00:00:00+00:00
        /// NetworkGraceMinutes: 480
        /// IsPublic: false
        public const string PerpetualPlaceholder =
                "License 590782408 for Acme Corp :S2luZDogaWQKSWQ6IDU5MDc4MjQwOApTZWNyZXQ6IHRlc3RjODlmMzYyM2M3YTIzNTlhNjQzYzk4NWJmZGUyMTliOWJlZDA5ZDQ4YWZjNTQ5MzExNDMzMzE5NTViOTgzMTVmCk93bmVyOiBBY21lIENvcnAKSXNzdWVkOiAyMDE3LTA0LTIxVDAwOjAwOjAwKzAwOjAwCk5ldHdvcmtHcmFjZU1pbnV0ZXM6IDQ4MApJc1B1YmxpYzogZmFsc2U=:J7lmHd2rL8qiaD6yjh0UOmCN2hrQWManpU1HD1UEDxc+i6jAfqaPSTf1shgK8M9Lv89FRbdwVd5OUFX4ahUlHMsxjBixEhUt50gCZ+bXdi8e+q5LrnJkAqosnU3XrBurOz6V28PN6S+RzQdcuxR+tfMDtCP0yiSm0SCG09rUjXF5DvYXzNNjQ42ncdpB4IA5PxxVyNi1uwq25eVVIPmno8GhHf2F/RPeLlZPCSL664osH0qXeYWcMoy1fa0gYnGH8qSWzi2+VW/zCZoFpzEzyH+QUInstQ8fKTIy6x0WKMnQrwvIIaxYc2buZBFXF4MS9kU236z1jlFs1vxsFc0/PA=="
            ;

        /// Id: 590782408
        /// Owner: Acme Corp
        /// Features: R_Elite R_Creative R_Performance
        /// Product: site-wide
        /// Kind: site-wide
        /// Issued: 2017-04-21T00:00:00+00:00
        /// IsPublic: true
        /// MustBeFetched: true
        /// SubscriptionExpirationDate: 2017-04-21T00:00:00+00:00
        /// Message: Your subscription has expired; please renew to access newer product releases.
        public const string PerpetualRemote =
                "License 590782408 for Acme Corp (site-wide) :SWQ6IDU5MDc4MjQwOApPd25lcjogQWNtZSBDb3JwCkZlYXR1cmVzOiBSX0VsaXRlIFJfQ3JlYXRpdmUgUl9QZXJmb3JtYW5jZQpQcm9kdWN0OiBzaXRlLXdpZGUKS2luZDogc2l0ZS13aWRlCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApJc1B1YmxpYzogdHJ1ZQpNdXN0QmVGZXRjaGVkOiB0cnVlClN1YnNjcmlwdGlvbkV4cGlyYXRpb25EYXRlOiAyMDE3LTA0LTIxVDAwOjAwOjAwKzAwOjAwCk1lc3NhZ2U6IFlvdXIgc3Vic2NyaXB0aW9uIGhhcyBleHBpcmVkOyBwbGVhc2UgcmVuZXcgdG8gYWNjZXNzIG5ld2VyIHByb2R1Y3QgcmVsZWFzZXMu:jYhhm49uitV0Yo0VBbD8HROkZOLpyCjaOc1Bl1MkX6rCHWMP7/Tb1PlacmN9nPx1uu4jtUDjTOrk9OO4bu7JrcCs539tPCLeUsiFXtTL1cNdV5y6VeGeZqYWf7S8QjlYr0uoyfGfCS9qHxm4p9g6+iTCeCtNY9PPdy1iAMffoqAJo4yWwapQu66LQf0sm6ZbD70voK9wbMdye6Utt4t+JO5s4BdoxB2PGoiSTurr9+PMz1XiB/zT7fruTs9IlmZGp2ua4QcgzHc7oVIyfpmUP05zUHe+7T4Zr/1ru3L7M1+PEny+noGweMY+uHkMVwlOf4TM31Q+wP7WlVr5rCecLQ=="
            ;

        /// Kind: id
        /// Id: 1577338763
        /// Secret: test8b47045eb7b8ca42aa967f33ee1d014ba89f8d1ac207426b482d34b5c0d90935
        /// Owner: Acme Corp
        /// Issued: 2017-04-21T00:00:00+00:00
        /// NetworkGraceMinutes: 480
        /// IsPublic: false
        public const string CancelledPlaceholder =
                "License 1577338763 for Acme Corp :S2luZDogaWQKSWQ6IDE1NzczMzg3NjMKU2VjcmV0OiB0ZXN0OGI0NzA0NWViN2I4Y2E0MmFhOTY3ZjMzZWUxZDAxNGJhODlmOGQxYWMyMDc0MjZiNDgyZDM0YjVjMGQ5MDkzNQpPd25lcjogQWNtZSBDb3JwCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApOZXR3b3JrR3JhY2VNaW51dGVzOiA0ODAKSXNQdWJsaWM6IGZhbHNl:dXygA7cbBgeamNz/ar0spKGU5TOYgMbbUGhZgSsWHCgFQZ1XhJ0Dy4zmfUB/YDNb4ZIL289kxTm0h36K7mPzeNbkGFB/8NtjisBeOZQJ0uKa9D5vb/HhtZfONuyJ4oZsyMEKKIUxAqVIBURrHFVUHUtFyIje9SygaoD4SLRHqfHUbyxLikyut3gxojBA+9PitS0oHCXLhzzwxMDJLrl0XAseZ+gRLbXsWqiPEyqf0cUeYTuEURuqj5wH5ZLtBgpTMTxCJ5p0c0cZsWtGNGmj3pobSLuoHAYL3EG+IYGpjbcckpmsDCP7esa1ViI9uYTZO0mUyM5PW5p1HGluH9dpYw=="
            ;

        /// Id: 1577338763
        /// Owner: Acme Corp
        /// Features: R_Elite R_Creative R_Performance
        /// Product: site-wide
        /// Kind: site-wide
        /// Issued: 2017-04-21T00:00:00+00:00
        /// IsPublic: true
        /// MustBeFetched: true
        /// Valid: false
        /// Message: Your subscription has lapsed; please renew to continue using product.
        public const string CancelledRemote =
                "License 1577338763 for Acme Corp (site-wide) :SWQ6IDE1NzczMzg3NjMKT3duZXI6IEFjbWUgQ29ycApGZWF0dXJlczogUl9FbGl0ZSBSX0NyZWF0aXZlIFJfUGVyZm9ybWFuY2UKUHJvZHVjdDogc2l0ZS13aWRlCktpbmQ6IHNpdGUtd2lkZQpJc3N1ZWQ6IDIwMTctMDQtMjFUMDA6MDA6MDArMDA6MDAKSXNQdWJsaWM6IHRydWUKTXVzdEJlRmV0Y2hlZDogdHJ1ZQpWYWxpZDogZmFsc2UKTWVzc2FnZTogWW91ciBzdWJzY3JpcHRpb24gaGFzIGxhcHNlZDsgcGxlYXNlIHJlbmV3IHRvIGNvbnRpbnVlIHVzaW5nIHByb2R1Y3Qu:eRwku3vZCLCe2YtOtmS/hHgW8UnUvRPD7VAwJfqvG5gDxZ49RD/hLbpKiKDI9Dw/L5qEaUSqkpHdMrLsXr6MdYmg6uiKX1c+zq6ywaUotkQibgyMKBZgHfqkn6zjB/62tRWjHkqQVl+E2fhQFgDVjp6Yep94RV7h8G7QOg/WRjKFcR9dSeLRbz4EpJJ1uWWrKZEU93161u+GqLpgd85e/rHEbTTpvUXpRKRzaDMuQ6DWqlZmwsWpkOpf10Kn+cVyBNg8uYAU7W5zIKNLioGo4E15daVXb48zhX5UYNQByjqe81Lu73p9ggOoZJwRuSbzQb9daarMAClJldBGlTHtcg=="
            ;

        /// Kind: id
        /// Id: 2766684890
        /// Secret: testda42e8a40db14c091dea84efd572933fdfe31ba9620e5fee79edb823a448b6e8
        /// Owner: Acme Corp
        /// Issued: 2017-04-21T00:00:00+00:00
        /// NetworkGraceMinutes: 480
        /// IsPublic: false
        public const string SiteWidePlaceholder =
                "License 2766684890 for Acme Corp :S2luZDogaWQKSWQ6IDI3NjY2ODQ4OTAKU2VjcmV0OiB0ZXN0ZGE0MmU4YTQwZGIxNGMwOTFkZWE4NGVmZDU3MjkzM2ZkZmUzMWJhOTYyMGU1ZmVlNzllZGI4MjNhNDQ4YjZlOApPd25lcjogQWNtZSBDb3JwCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApOZXR3b3JrR3JhY2VNaW51dGVzOiA0ODAKSXNQdWJsaWM6IGZhbHNl:Z3dUtjcuWaNOasGk4g/6BUSKjrxDZBTHGHE7vn0OUQsuw0AEnVS1XI/MogXmA5C3yF2BQfbDHktZbuh8aDihy6MXJnS/2FXLenwjZ4AXqcAzM9D7KvDiimUsypjywWclc95yB6LOpO784ekKsVWxsGdAj2P3+y2ZjaJIoxs3qTUzal1lsbEVPtKMfVpkUwj4gxnpJe9lTx6CVgzbSkphKFEfvTfuvzgTahIVHcJcv+Sk35zIqWJPpnUDAj4NxOxpsbI17HV84ivMgknqj2t1LY2P4g16+h7+1XzNIm3j6i2wAIBvM7kVW1XX7wFYD8kS+gqxK4HCs1Ge5UQIGDnlyg=="
            ;

        /// Id: 2766684890
        /// Owner: Acme Corp
        /// Features: R_Elite R_Creative R_Performance
        /// Product: Enterprise-wide license
        /// Kind: site-wide
        /// Issued: 2017-04-21T00:00:00+00:00
        /// Expires: 2017-06-05T00:00:00+00:00
        /// IsPublic: true
        /// MustBeFetched: true
        /// ManageYourSubscription: https://account.imazen.io
        /// Restrictions: No resale of usage. Only for organizations with less than 500 employees.
        public const string SiteWideRemote =
                "License 2766684890 for Acme Corp (Enterprise-wide license) :SWQ6IDI3NjY2ODQ4OTAKT3duZXI6IEFjbWUgQ29ycApGZWF0dXJlczogUl9FbGl0ZSBSX0NyZWF0aXZlIFJfUGVyZm9ybWFuY2UKUHJvZHVjdDogRW50ZXJwcmlzZS13aWRlIGxpY2Vuc2UKS2luZDogc2l0ZS13aWRlCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApFeHBpcmVzOiAyMDE3LTA2LTA1VDAwOjAwOjAwKzAwOjAwCklzUHVibGljOiB0cnVlCk11c3RCZUZldGNoZWQ6IHRydWUKTWFuYWdlWW91clN1YnNjcmlwdGlvbjogaHR0cHM6Ly9hY2NvdW50LmltYXplbi5pbwpSZXN0cmljdGlvbnM6IE5vIHJlc2FsZSBvZiB1c2FnZS4gT25seSBmb3Igb3JnYW5pemF0aW9ucyB3aXRoIGxlc3MgdGhhbiA1MDAgZW1wbG95ZWVzLg==:nXfjvTFO6JOAxbdSa5/BWYzD0fRcezVURSqaCdX+EMe5Lc+KAgicmt8shwYX2r/DTZkhkiQJ+yqNMs3+OYvOmMpK5qCwlLcPq9zykLbxL4SIth7C6UPx8HAs0TbpWnInz6N/n4xCh75Cw+n80bhqG/4p55xy1ltXVeLJFngS7xS4p7rplGlwuD4Jmh+KzOzrqvhxBmXI9ecPGtG080x3tMMiCkSjm6LQM9GWcEwgxEsFd02MCNHZcERoJ024tzgSndTBnPjxSTLBq5FWKo1mHIqOnvvc+Pbe6Z1EeOkW6MxFRK4myFl7ZwusRvHj0RN8UgPDCwHAy2VJ0SwD/PKiKg=="
            ;

        /// Kind: id
        /// Id: 2613256515
        /// Secret: test4321c39bec40289e14ab42503a533efb43dbe0e355be787bb3c89f4519f503b7
        /// Owner: Acme Corp
        /// Issued: 2017-04-21T00:00:00+00:00
        /// NetworkGraceMinutes: 480
        /// IsPublic: false
        public const string OemPlaceholder =
                "License 2613256515 for Acme Corp :S2luZDogaWQKSWQ6IDI2MTMyNTY1MTUKU2VjcmV0OiB0ZXN0NDMyMWMzOWJlYzQwMjg5ZTE0YWI0MjUwM2E1MzNlZmI0M2RiZTBlMzU1YmU3ODdiYjNjODlmNDUxOWY1MDNiNwpPd25lcjogQWNtZSBDb3JwCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApOZXR3b3JrR3JhY2VNaW51dGVzOiA0ODAKSXNQdWJsaWM6IGZhbHNl:kmfYe02MlS5NMZK6j7ae/eDvVjX7d4hFITopORMtRGtIY6gZ4tp2AcXYPi61WsRQQcbuK1scxsx+hFSs0tCmXxx6AHQScoTosJequT2dlY9GqDdt4CZ6qeFNol7+YqU/6SK0W6MsOR/lw5nZHg3bshTkvg8LkoXBJyyGX4MX5vtucLwRTZtzj7RATuBghoKFhSnKPL8Pc7mcWwArSSo8KU4Un0Eu8Z+8I+R4kiMy1+liI7ZiF66YFW5pxCDxx/SoQhrV8pwC9d8kA1TGDRHbBzzRcg0DS5PDynPX8GyjceluBcrSJIQ/VVZiRz45gT75cj7wrahVlId1T9UtEC2ytA=="
            ;

        /// Id: 2613256515
        /// Owner: Acme Corp
        /// Features: R_Elite R_Creative R_Performance
        /// Product: OEM redistribution license
        /// Kind: oem
        /// Issued: 2017-04-21T00:00:00+00:00
        /// Expires: 2017-06-10T00:00:00+00:00
        /// IsPublic: true
        /// MustBeFetched: true
        /// ManageYourSubscription: https://account.imazen.io
        /// Restrictions: Only for use within Vendor Gamma Product
        public const string OemRemote =
                "License 2613256515 for Acme Corp (OEM redistribution license) :SWQ6IDI2MTMyNTY1MTUKT3duZXI6IEFjbWUgQ29ycApGZWF0dXJlczogUl9FbGl0ZSBSX0NyZWF0aXZlIFJfUGVyZm9ybWFuY2UKUHJvZHVjdDogT0VNIHJlZGlzdHJpYnR1aW9uIGxpY2Vuc2UKS2luZDogb2VtCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApFeHBpcmVzOiAyMDE3LTA2LTEwVDAwOjAwOjAwKzAwOjAwCklzUHVibGljOiB0cnVlCk11c3RCZUZldGNoZWQ6IHRydWUKTWFuYWdlWW91clN1YnNjcmlwdGlvbjogaHR0cHM6Ly9hY2NvdW50LmltYXplbi5pbwpSZXN0cmljdGlvbnM6IE9ubHkgZm9yIHVzZSB3aXRoaW4gVmVuZG9yIEdhbW1hIFByZHVjdA==:Czr3ds3Dn5/Q9pcKFmv4PLi4BcDL3C4zdK40hQNw/mO224nWM+LAYFF82pv5GnA6ocFqZ7knxrF5tt4l9JaXVzbVkV0SSNnPuzsQQd94C+O7TF9YJnGSMEjrJEm0NeKwH0N8RMqtbcIyFiJf0BDM3amQJ5gQY+sNS56shtzCtqh1P3KYXzM7wgfJVFP9fCSANGfJxthF0qntp2RccUBMEUIqLEimUkxNLE5kNLPnJHe8x1Pf7FfnhgATeXZj7azQy6Q9UH4dSUU3GdOIGAjEMt1L3x7iFSPyd9oYKrXJV3J6w5zIFLh7tC8+l/enIxzTYp21ixjWjmkVbDyCPmmfTg=="
            ;

        /// Kind: id
        /// Id: 3550635254
        /// Secret: testf664a2d346d8cfef90e3f4819a5c2454f03a68993f28f2bb4773039f6f08762e
        /// Owner: Acme Corp
        /// Issued: 2017-04-21T00:00:00+00:00
        /// NetworkGraceMinutes: 480
        /// IsPublic: false
        public const string CoresPlaceholder =
                "License 3550635254 for Acme Corp :S2luZDogaWQKSWQ6IDM1NTA2MzUyNTQKU2VjcmV0OiB0ZXN0ZjY2NGEyZDM0NmQ4Y2ZlZjkwZTNmNDgxOWE1YzI0NTRmMDNhNjg5OTNmMjhmMmJiNDc3MzAzOWY2ZjA4NzYyZQpPd25lcjogQWNtZSBDb3JwCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApOZXR3b3JrR3JhY2VNaW51dGVzOiA0ODAKSXNQdWJsaWM6IGZhbHNl:l3dZRrQoJiHer3t4Moxuai9ZEARhMCK71IfsnvmfWhmnmLMYNgH8wyrN/txyoM5ANtx/J1QqMEy2Y9mqLiWBf4wtgcg9X/5ZDx1l/tTivJad/05L91ojEiG1NYi2Co/wrRfXw+ubA5/DM98vNAqWLFuqbEklj+D251dokBEvCGCNRcs/c/APvvtyL2n3Rhwf24SIc85w8iUjyN3XgYq6m6n/Ev57mYrqYjrgjLYClbJYQuhPDeETquYZj3es8MTuLNtXTiIp59iQHpw+IO2CuHVSZYmsGkNLKydtAuM4++6qiVfFn0iY7fpwB5CTOl6hw9cer5Xx89H/4Rr5Q73qBA=="
            ;

        /// Id: 3550635254
        /// Owner: Acme Corp
        /// Features: R_Elite R_Creative R_Performance
        /// Product: Per-core license
        /// Kind: per-core
        /// Issued: 2017-04-21T00:00:00+00:00
        /// Expires: 2017-06-10T00:00:00+00:00
        /// IsPublic: true
        /// MustBeFetched: true
        /// MaxServers: 4
        /// TotalCores: 16
        /// ManageYourSubscription: https://account.imazen.io
        public const string CoresRemote =
                "License 3550635254 for Acme Corp (Per-core license) :SWQ6IDM1NTA2MzUyNTQKT3duZXI6IEFjbWUgQ29ycApGZWF0dXJlczogUl9FbGl0ZSBSX0NyZWF0aXZlIFJfUGVyZm9ybWFuY2UKUHJvZHVjdDogUGVyLWNvcmUgbGljZW5zZQpLaW5kOiBwZXItY29yZQpJc3N1ZWQ6IDIwMTctMDQtMjFUMDA6MDA6MDArMDA6MDAKRXhwaXJlczogMjAxNy0wNi0xMFQwMDowMDowMCswMDowMApJc1B1YmxpYzogdHJ1ZQpNdXN0QmVGZXRjaGVkOiB0cnVlCk1heFNlcnZlcnM6IDQKVG90YWxDb3JlczogMTYKTWFuYWdlWW91clN1YnNjcmlwdGlvbjogaHR0cHM6Ly9hY2NvdW50LmltYXplbi5pbw==:bKWvg/9wa7NYWzmb5l59A7ZD8lfWwo6hJEBl9FEIomYCBGiEQF3RDmoYgPOmjnQTG4mNj8Hh0XuLbxOS5Vi03wVOY7SGW0TwMh52XY0TIX+a4f9fQl4mofSrAe1G8WtPSdwC7ReW4IBDQg/zydzcMJJGpp1mWsARlFW2T3I7RYONVHt54L/X6+gP5UGoAbHC6pFI4q6HbOh679bfI7Ts8aEmYCsxeRHFRkk0ONOfuzXlrUghLkx05RFA3XCzcLaJ4DxGyJENNaYyAohbX2nWLsIv6ePVsJdNRm2aeIHWNDMJ31O0I4V2kllMkEG4YnHwvR6BPWOzlkjgeZgIcnu2qA=="
            ;

        /// Kind: id
        /// Id: 3140539178
        /// Secret: test2ad330bb24212a764310654319b42e2658774a87e4da58dc5eca01ac193577f1
        /// Owner: Acme Corp
        /// Issued: 2017-04-21T00:00:00+00:00
        /// NetworkGraceMinutes: 480
        /// IsPublic: false
        public const string PerCore2DomainsPlaceholder =
                "License 3140539178 for Acme Corp :S2luZDogaWQKSWQ6IDMxNDA1MzkxNzgKU2VjcmV0OiB0ZXN0MmFkMzMwYmIyNDIxMmE3NjQzMTA2NTQzMTliNDJlMjY1ODc3NGE4N2U0ZGE1OGRjNWVjYTAxYWMxOTM1NzdmMQpPd25lcjogQWNtZSBDb3JwCklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApOZXR3b3JrR3JhY2VNaW51dGVzOiA0ODAKSXNQdWJsaWM6IGZhbHNl:YNJgKkYTgdZs5a8zNpRq41E5bDrKzHXiGeCtzAxJCdy01d/utzApX8eBRPENfaJ/V2b/cZI98HuUF2Btdf6W9V+1VYQvkzoB9k73Suk7BgSQubf0dbBidH2pgRWATm526xw5i3YMafzL+584vaMVnvSwdBMKca4VS/mTD/zSfGiUFNbPlUAj5B696OMczB8qrczvdfD1gBdaaTDRJDYf/7/fCKM9nLqBtb5S/aEgjjpqXP8PW51dcDtvfCqWH/82tx+mAoB62D1jL9fm7aqQPtq8JGvv8pjH/R5RfPmVA8UR28mwUCbfJTKlvJ8M7Uc3Ym3Gzv2w1LV/8he1WXNWHQ=="
            ;

        /// Id: 3140539178
        /// Owner: Acme Corp
        /// Domains: acme.com acmestaging.com
        /// Features: R_Performance
        /// Product: Per Server | 2 Domains | Performance
        /// Kind: per-core-domain
        /// Issued: 2017-04-21T00:00:00+00:00
        /// Expires: 2017-06-10T00:00:00+00:00
        /// IsPublic: true
        /// MustBeFetched: true
        /// MaxServers: 4
        /// TotalCores: 16
        /// ManageYourSubscription: https://account.imazen.io
        public const string PerCore2DomainsRemote =
                "License 3140539178 for Acme Corp (Per Server | 2 Domains | Performance) :SWQ6IDMxNDA1MzkxNzgKT3duZXI6IEFjbWUgQ29ycApEb21haW5zOiBhY21lLmNvbSBhY21lc3RhZ2luZy5jb20KRmVhdHVyZXM6IFJfUGVyZm9ybWFuY2UKUHJvZHVjdDogUGVyIFNlcnZlciB8IDIgRG9tYWlucyB8IFBlcmZvcm1hbmNlCktpbmQ6IHBlci1jb3JlLWRvbWFpbgpJc3N1ZWQ6IDIwMTctMDQtMjFUMDA6MDA6MDArMDA6MDAKRXhwaXJlczogMjAxNy0wNi0xMFQwMDowMDowMCswMDowMApJc1B1YmxpYzogdHJ1ZQpNdXN0QmVGZXRjaGVkOiB0cnVlCk1heFNlcnZlcnM6IDQKVG90YWxDb3JlczogMTYKTWFuYWdlWW91clN1YnNjcmlwdGlvbjogaHR0cHM6Ly9hY2NvdW50LmltYXplbi5pbw==:bkPk/CKi++X1Nq8+px7wRpIdPkuUVe0s6o3sLoAtclcBmSt4loCpEku/6zix+EqAtlY98bD+WHWzD3rxTP5kPjtWURnte0aSoYzPuD6gf2aOJdWMgxPy3xLlWyQr+dVT7Ip85PdxBYCDIV7hfjXAeAH+cniNK8PFVYhukapl8prEuuAXnQJhYjU38zKDgwUfrWkwkkUPHbpFgsdqs0vyhV+YUpPSUObL7NdGky9RZABmVfVQhacPkh3nOkm3p5m1b29ehgyr8rIR2JvOBC9DLtzRTgzXb7bXq0msLzrLvZuH7DsZ1oziz9of7p+c9msji19PcOZqUbVhujbvyt36vQ=="
            ;

        /// Kind: id
        /// Id: 1319146175
        /// Secret: testbf92a04e60b4b2362d45490e5142e47b687c2ebb5898da6bc71b697602016e6e
        /// Owner: Acme Corp &lt;&gt;
        /// Issued: 2017-04-21T00:00:00+00:00
        /// NetworkGraceMinutes: 480
        /// IsPublic: false
        public const string XmlPlaceholder =
                "License 1319146175 for Acme Corp &lt;&gt; :S2luZDogaWQKSWQ6IDEzMTkxNDYxNzUKU2VjcmV0OiB0ZXN0YmY5MmEwNGU2MGI0YjIzNjJkNDU0OTBlNTE0MmU0N2I2ODdjMmViYjU4OThkYTZiYzcxYjY5NzYwMjAxNmU2ZQpPd25lcjogQWNtZSBDb3JwICZsdDsmZ3Q7Cklzc3VlZDogMjAxNy0wNC0yMVQwMDowMDowMCswMDowMApOZXR3b3JrR3JhY2VNaW51dGVzOiA0ODAKSXNQdWJsaWM6IGZhbHNl:L4IFQqsiinXwOEUuVgd42X3yxBBjvph8g51vwZ16RXWTDr/MCwJ3cdXwuArrGCq9x/yr9TLsUwzLcjn8DcxdHjpNmHK3qN0WZl1fyjMD50kH4kZlXlCVV71zH55HsAsQ1oMcUyuLTugbJy6cY2OPKQWayFyDxm8i+xqdUaaXof7tT/OX9YVTviwNgL2Mj2FEaXtN2ufAB6uN+iGpTKAmdvOaYKuVDMp9FT1G9jByGIXfzsyG3TKrKs25rnKp3ZCjtorzg5J1k2/O+W9xFKnEU6+vonlbzGa6tcycPFPkzS0lnj8fGJOOC6AeOfKMSeX/plcfnSLt4ykjnIUKr0S7rQ=="
            ;

        /// Id: 1319146175
        /// Owner: Acme Corp &lt;&gt;
        /// Domains: acme.com acmestaging.com
        /// Features: R_Performance
        /// Product: Per Server | 2 Domains | Performance
        /// Kind: per-core-domain
        /// Issued: 2017-04-21T00:00:00+00:00
        /// Expires: 2017-06-10T00:00:00+00:00
        /// IsPublic: true
        /// MustBeFetched: true
        /// MaxServers: 4
        /// TotalCores: 16
        /// ManageYourSubscription: https://account.imazen.io
        public const string XmlRemote =
                "License 1319146175 for Acme Corp &lt;&gt; (Per Server | 2 Domains | Performance) :SWQ6IDEzMTkxNDYxNzUKT3duZXI6IEFjbWUgQ29ycCAmbHQ7Jmd0OwpEb21haW5zOiBhY21lLmNvbSBhY21lc3RhZ2luZy5jb20KRmVhdHVyZXM6IFJfUGVyZm9ybWFuY2UKUHJvZHVjdDogUGVyIFNlcnZlciB8IDIgRG9tYWlucyB8IFBlcmZvcm1hbmNlCktpbmQ6IHBlci1jb3JlLWRvbWFpbgpJc3N1ZWQ6IDIwMTctMDQtMjFUMDA6MDA6MDArMDA6MDAKRXhwaXJlczogMjAxNy0wNi0xMFQwMDowMDowMCswMDowMApJc1B1YmxpYzogdHJ1ZQpNdXN0QmVGZXRjaGVkOiB0cnVlCk1heFNlcnZlcnM6IDQKVG90YWxDb3JlczogMTYKTWFuYWdlWW91clN1YnNjcmlwdGlvbjogaHR0cHM6Ly9hY2NvdW50LmltYXplbi5pbw==:k+3Zjk/2D+o21Y3bVlMiifThcobnxzdwimLkFutS/y8CgzHvYcbtvlHLaEVYAvXSvAqXbF8NlCbcIrA/liW+9fNsLd78dYICfry20tyH9Z8Sf46DIsq+V7YDN7NvlOG8DAJeLBbSSX21yi9851g4uthbNeSK5QpZ6XG0b5uUe/jPSk/+ZCmB/Gco0tCFtwiJkli4j5GBZOFcvKP9PrwmTFJtELNyfx0/WiAo1y+XXGsdeAQUbnHOWAOZ/nbzzNE0XOoE87uUXcUh6BwQFeII7KGGxsFU0T2R9O/q6M90K8Mdph/Kb6Mv6IVm8cUMXaqIJJN6U8SRz4QW5IePWlI9ww=="
            ;
    }

}
