package aggregategenerator_test

import (
	"bytes"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"github.com/inklabs/rangedb/pkg/aggregategenerator"
)

func TestWrite(t *testing.T) {
	const (
		pkg           = "mypkg"
		aggregateName = "thing"
	)
	aggregategenerator.NowFunc = func() time.Time {
		return time.Unix(1611080667, 25600800).UTC()
	}

	t.Run("no commands found", func(t *testing.T) {
		// Given
		var commands []string
		var out bytes.Buffer

		// When
		err := aggregategenerator.Write(&out, commands, pkg, aggregateName)

		// Then
		require.EqualError(t, err, "no commands found")
	})

	t.Run("single command", func(t *testing.T) {
		// Given
		commands := []string{
			"DoThing",
		}
		var out bytes.Buffer

		// When
		err := aggregategenerator.Write(&out, commands, pkg, aggregateName)

		// Then
		require.NoError(t, err)
		expectedOut := `// Code generated by go generate; DO NOT EDIT.
// This file was generated at
// 2021-01-19 18:24:27.0256008 +0000 UTC
package mypkg

import (
	"github.com/inklabs/rangedb"
	"github.com/inklabs/rangedb/pkg/cqrs"
)

func (a *thing) Load(recordIterator rangedb.RecordIterator) {
	for recordIterator.Next() {
		if recordIterator.Err() == nil {
			if event, ok := recordIterator.Record().Data.(rangedb.Event); ok {
				a.apply(event)
			}
		}
	}
}

func (a *thing) Handle(command cqrs.Command) []rangedb.Event {
	switch c := command.(type) {
	case DoThing:
		a.DoThing(c)

	}

	defer a.resetPendingEvents()
	return a.pendingEvents
}

func (a *thing) resetPendingEvents() {
	a.pendingEvents = nil
}

func (a *thing) CommandTypes() []string {
	return []string{
		DoThing{}.CommandType(),
	}
}

func (a *thing) raise(events ...rangedb.Event) {
	a.pendingEvents = append(a.pendingEvents, events...)
}
`
		assert.Equal(t, expectedOut, out.String())
	})

	t.Run("two commands", func(t *testing.T) {
		// Given
		commands := []string{
			"DoThing",
			"DoAnother",
		}
		var out bytes.Buffer

		// When
		err := aggregategenerator.Write(&out, commands, pkg, aggregateName)

		// Then
		require.NoError(t, err)
		expectedOut := `// Code generated by go generate; DO NOT EDIT.
// This file was generated at
// 2021-01-19 18:24:27.0256008 +0000 UTC
package mypkg

import (
	"github.com/inklabs/rangedb"
	"github.com/inklabs/rangedb/pkg/cqrs"
)

func (a *thing) Load(recordIterator rangedb.RecordIterator) {
	for recordIterator.Next() {
		if recordIterator.Err() == nil {
			if event, ok := recordIterator.Record().Data.(rangedb.Event); ok {
				a.apply(event)
			}
		}
	}
}

func (a *thing) Handle(command cqrs.Command) []rangedb.Event {
	switch c := command.(type) {
	case DoThing:
		a.DoThing(c)

	case DoAnother:
		a.DoAnother(c)

	}

	defer a.resetPendingEvents()
	return a.pendingEvents
}

func (a *thing) resetPendingEvents() {
	a.pendingEvents = nil
}

func (a *thing) CommandTypes() []string {
	return []string{
		DoThing{}.CommandType(),
		DoAnother{}.CommandType(),
	}
}

func (a *thing) raise(events ...rangedb.Event) {
	a.pendingEvents = append(a.pendingEvents, events...)
}
`
		assert.Equal(t, expectedOut, out.String())
	})
}
