package commandgenerator_test

import (
	"bytes"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"github.com/inklabs/rangedb/pkg/commandgenerator"
)

func TestWrite(t *testing.T) {
	const (
		pkg           = "mypkg"
		id            = "ID"
		aggregateType = "thing"
	)
	commandgenerator.NowFunc = func() time.Time {
		return time.Unix(1611080667, 25600800).UTC()
	}

	t.Run("no commands found", func(t *testing.T) {
		// Given
		var commands []string
		var out bytes.Buffer

		// When
		err := commandgenerator.Write(&out, commands, pkg, id, aggregateType)

		// Then
		require.EqualError(t, err, "no commands found")
	})

	t.Run("single command", func(t *testing.T) {
		// Given
		commands := []string{
			"DoThing",
		}
		var out bytes.Buffer

		// When
		err := commandgenerator.Write(&out, commands, pkg, id, aggregateType)

		// Then
		require.NoError(t, err)
		expectedOut := `// Code generated by go generate; DO NOT EDIT.
// This file was generated at
// 2021-01-19 18:24:27.0256008 +0000 UTC
package mypkg

func (c DoThing) AggregateID() string   { return c.ID }
func (c DoThing) AggregateType() string { return "thing" }
func (c DoThing) CommandType() string   { return "DoThing" }
`
		assert.Equal(t, expectedOut, out.String())
	})

	t.Run("two commands", func(t *testing.T) {
		// Given
		commands := []string{
			"DoThing",
			"DoAnother",
		}
		var out bytes.Buffer

		// When
		err := commandgenerator.Write(&out, commands, pkg, id, aggregateType)

		// Then
		require.NoError(t, err)
		expectedOut := `// Code generated by go generate; DO NOT EDIT.
// This file was generated at
// 2021-01-19 18:24:27.0256008 +0000 UTC
package mypkg

func (c DoThing) AggregateID() string   { return c.ID }
func (c DoThing) AggregateType() string { return "thing" }
func (c DoThing) CommandType() string   { return "DoThing" }

func (c DoAnother) AggregateID() string   { return c.ID }
func (c DoAnother) AggregateType() string { return "thing" }
func (c DoAnother) CommandType() string   { return "DoAnother" }
`
		assert.Equal(t, expectedOut, out.String())
	})
}
