package eventparser_test

import (
	"bytes"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"github.com/inklabs/rangedb/pkg/eventparser"
)

func TestEventWriter(t *testing.T) {
	const (
		pkg           = "mypkg"
		id            = "ID"
		aggregateType = "thing"
	)
	eventparser.NowFunc = func() time.Time {
		return time.Unix(1611080667, 25600800).UTC()
	}

	t.Run("no events found", func(t *testing.T) {
		// Given
		var events []eventparser.ParsedEvent
		var out bytes.Buffer

		// When
		err := eventparser.WriteEvents(&out, events, pkg, id, aggregateType)

		// Then
		require.EqualError(t, err, "no events found")
	})

	t.Run("single event", func(t *testing.T) {
		// Given
		events := []eventparser.ParsedEvent{
			{
				Name:         "ThingWasDone",
				PersonalData: nil,
			},
		}
		var out bytes.Buffer

		// When
		err := eventparser.WriteEvents(&out, events, pkg, id, aggregateType)

		// Then
		require.NoError(t, err)
		expectedOut := `// Code generated by go generate; DO NOT EDIT.
// This file was generated at
// 2021-01-19 18:24:27.0256008 +0000 UTC
package mypkg

func (e ThingWasDone) AggregateID() string   { return e.ID }
func (e ThingWasDone) AggregateType() string { return "thing" }
func (e ThingWasDone) EventType() string     { return "ThingWasDone" }
`
		assert.Equal(t, expectedOut, out.String())
	})

	t.Run("two events", func(t *testing.T) {
		// Given
		events := []eventparser.ParsedEvent{
			{
				Name:         "ThingWasDone",
				PersonalData: nil,
			},
			{
				Name:         "AnotherThingWasDone",
				PersonalData: nil,
			},
		}
		var out bytes.Buffer

		// When
		err := eventparser.WriteEvents(&out, events, pkg, id, aggregateType)

		// Then
		require.NoError(t, err)
		expectedOut := `// Code generated by go generate; DO NOT EDIT.
// This file was generated at
// 2021-01-19 18:24:27.0256008 +0000 UTC
package mypkg

func (e ThingWasDone) AggregateID() string   { return e.ID }
func (e ThingWasDone) AggregateType() string { return "thing" }
func (e ThingWasDone) EventType() string     { return "ThingWasDone" }

func (e AnotherThingWasDone) AggregateID() string   { return e.ID }
func (e AnotherThingWasDone) AggregateType() string { return "thing" }
func (e AnotherThingWasDone) EventType() string     { return "AnotherThingWasDone" }
`
		assert.Equal(t, expectedOut, out.String())
	})

	t.Run("single event with single string personal data field", func(t *testing.T) {
		// Given
		events := []eventparser.ParsedEvent{
			{
				Name: "CustomerSignedUp",
				PersonalData: &eventparser.PersonalData{
					SubjectID:        "ID",
					Fields:           []string{"Email"},
					SerializedFields: nil,
				},
			},
		}
		aggregateType := "customer"
		var out bytes.Buffer

		// When
		err := eventparser.WriteEvents(&out, events, pkg, id, aggregateType)

		// Then
		require.NoError(t, err)
		expectedOut := `// Code generated by go generate; DO NOT EDIT.
// This file was generated at
// 2021-01-19 18:24:27.0256008 +0000 UTC
package mypkg

import (
	"github.com/inklabs/rangedb/pkg/crypto"
)

func (e CustomerSignedUp) AggregateID() string   { return e.ID }
func (e CustomerSignedUp) AggregateType() string { return "customer" }
func (e CustomerSignedUp) EventType() string     { return "CustomerSignedUp" }
func (e *CustomerSignedUp) Encrypt(encryptor crypto.Encryptor) error {
	var err error
	e.Email, err = encryptor.Encrypt(e.ID, e.Email)
	if err != nil {
		e.Email = ""
		return err
	}

	return nil
}
func (e *CustomerSignedUp) Decrypt(encryptor crypto.Encryptor) error {
	var err error
	e.Email, err = encryptor.Decrypt(e.ID, e.Email)
	if err != nil {
		e.Email = ""
		return err
	}

	return nil
}
`
		assert.Equal(t, expectedOut, out.String())
	})

	t.Run("single event with two string personal data fields", func(t *testing.T) {
		// Given
		events := []eventparser.ParsedEvent{
			{
				Name: "CustomerSignedUp",
				PersonalData: &eventparser.PersonalData{
					SubjectID:        "ID",
					Fields:           []string{"Name", "Email"},
					SerializedFields: nil,
				},
			},
		}
		aggregateType := "customer"
		var out bytes.Buffer

		// When
		err := eventparser.WriteEvents(&out, events, pkg, id, aggregateType)

		// Then
		require.NoError(t, err)
		expectedOut := `// Code generated by go generate; DO NOT EDIT.
// This file was generated at
// 2021-01-19 18:24:27.0256008 +0000 UTC
package mypkg

import (
	"github.com/inklabs/rangedb/pkg/crypto"
)

func (e CustomerSignedUp) AggregateID() string   { return e.ID }
func (e CustomerSignedUp) AggregateType() string { return "customer" }
func (e CustomerSignedUp) EventType() string     { return "CustomerSignedUp" }
func (e *CustomerSignedUp) Encrypt(encryptor crypto.Encryptor) error {
	var err error
	e.Name, err = encryptor.Encrypt(e.ID, e.Name)
	if err != nil {
		e.Name = ""
		e.Email = ""
		return err
	}

	e.Email, err = encryptor.Encrypt(e.ID, e.Email)
	if err != nil {
		e.Name = ""
		e.Email = ""
		return err
	}

	return nil
}
func (e *CustomerSignedUp) Decrypt(encryptor crypto.Encryptor) error {
	var err error
	e.Name, err = encryptor.Decrypt(e.ID, e.Name)
	if err != nil {
		e.Name = ""
		e.Email = ""
		return err
	}

	e.Email, err = encryptor.Decrypt(e.ID, e.Email)
	if err != nil {
		e.Name = ""
		e.Email = ""
		return err
	}

	return nil
}
`
		assert.Equal(t, expectedOut, out.String())
	})

	t.Run("single event with single int personal data field", func(t *testing.T) {
		// Given
		events := []eventparser.ParsedEvent{
			{
				Name: "CustomerAddedBirthYear",
				PersonalData: &eventparser.PersonalData{
					SubjectID: "ID",
					Fields:    nil,
					SerializedFields: map[string]string{
						"BirthYearEncrypted": "BirthYear",
					},
				},
			},
		}
		aggregateType := "customer"
		var out bytes.Buffer

		// When
		err := eventparser.WriteEvents(&out, events, pkg, id, aggregateType)

		// Then
		require.NoError(t, err)
		expectedOut := `// Code generated by go generate; DO NOT EDIT.
// This file was generated at
// 2021-01-19 18:24:27.0256008 +0000 UTC
package mypkg

import (
	"strconv"

	"github.com/inklabs/rangedb/pkg/crypto"
)

func (e CustomerAddedBirthYear) AggregateID() string   { return e.ID }
func (e CustomerAddedBirthYear) AggregateType() string { return "customer" }
func (e CustomerAddedBirthYear) EventType() string     { return "CustomerAddedBirthYear" }
func (e *CustomerAddedBirthYear) Encrypt(encryptor crypto.Encryptor) error {
	var err error
	stringBirthYear := strconv.Itoa(e.BirthYear)
	e.BirthYearEncrypted, err = encryptor.Encrypt(e.ID, stringBirthYear)
	if err != nil {
		e.BirthYear = 0
		return err
	}
	e.BirthYear = 0

	return nil
}
func (e *CustomerAddedBirthYear) Decrypt(encryptor crypto.Encryptor) error {
	var err error
	decryptedBirthYear, err := encryptor.Decrypt(e.ID, e.BirthYearEncrypted)
	if err != nil {
		e.BirthYear = 0
		return err
	}
	e.BirthYear, err = strconv.Atoi(decryptedBirthYear)
	if err != nil {
		e.BirthYear = 0
		return err
	}
	e.BirthYearEncrypted = ""

	return nil
}
`
		assert.Equal(t, expectedOut, out.String())
	})

	t.Run("single event with two int personal data fields", func(t *testing.T) {
		// Given
		events := []eventparser.ParsedEvent{
			{
				Name: "CustomerAddedBirth",
				PersonalData: &eventparser.PersonalData{
					SubjectID: "ID",
					Fields:    nil,
					SerializedFields: map[string]string{
						"BirthMonthEncrypted": "BirthMonth",
						"BirthYearEncrypted":  "BirthYear",
					},
				},
			},
		}
		aggregateType := "customer"
		var out bytes.Buffer

		// When
		err := eventparser.WriteEvents(&out, events, pkg, id, aggregateType)

		// Then
		require.NoError(t, err)
		expectedOut := `// Code generated by go generate; DO NOT EDIT.
// This file was generated at
// 2021-01-19 18:24:27.0256008 +0000 UTC
package mypkg

import (
	"strconv"

	"github.com/inklabs/rangedb/pkg/crypto"
)

func (e CustomerAddedBirth) AggregateID() string   { return e.ID }
func (e CustomerAddedBirth) AggregateType() string { return "customer" }
func (e CustomerAddedBirth) EventType() string     { return "CustomerAddedBirth" }
func (e *CustomerAddedBirth) Encrypt(encryptor crypto.Encryptor) error {
	var err error
	stringBirthMonth := strconv.Itoa(e.BirthMonth)
	e.BirthMonthEncrypted, err = encryptor.Encrypt(e.ID, stringBirthMonth)
	if err != nil {
		e.BirthMonth = 0
		e.BirthYear = 0
		return err
	}
	e.BirthMonth = 0

	stringBirthYear := strconv.Itoa(e.BirthYear)
	e.BirthYearEncrypted, err = encryptor.Encrypt(e.ID, stringBirthYear)
	if err != nil {
		e.BirthMonth = 0
		e.BirthYear = 0
		return err
	}
	e.BirthYear = 0

	return nil
}
func (e *CustomerAddedBirth) Decrypt(encryptor crypto.Encryptor) error {
	var err error
	decryptedBirthMonth, err := encryptor.Decrypt(e.ID, e.BirthMonthEncrypted)
	if err != nil {
		e.BirthMonth = 0
		e.BirthYear = 0
		return err
	}
	e.BirthMonth, err = strconv.Atoi(decryptedBirthMonth)
	if err != nil {
		e.BirthMonth = 0
		e.BirthYear = 0
		return err
	}
	e.BirthMonthEncrypted = ""

	decryptedBirthYear, err := encryptor.Decrypt(e.ID, e.BirthYearEncrypted)
	if err != nil {
		e.BirthMonth = 0
		e.BirthYear = 0
		return err
	}
	e.BirthYear, err = strconv.Atoi(decryptedBirthYear)
	if err != nil {
		e.BirthMonth = 0
		e.BirthYear = 0
		return err
	}
	e.BirthYearEncrypted = ""

	return nil
}
`
		assert.Equal(t, expectedOut, out.String())
	})

	t.Run("single event with two string and int personal data fields", func(t *testing.T) {
		// Given
		events := []eventparser.ParsedEvent{
			{
				Name: "CustomerAddedBirth",
				PersonalData: &eventparser.PersonalData{
					SubjectID: "ID",
					Fields:    []string{"Name", "Email"},
					SerializedFields: map[string]string{
						"BirthMonthEncrypted": "BirthMonth",
						"BirthYearEncrypted":  "BirthYear",
					},
				},
			},
		}
		aggregateType := "customer"
		var out bytes.Buffer

		// When
		err := eventparser.WriteEvents(&out, events, pkg, id, aggregateType)

		// Then
		require.NoError(t, err)
		expectedOut := `// Code generated by go generate; DO NOT EDIT.
// This file was generated at
// 2021-01-19 18:24:27.0256008 +0000 UTC
package mypkg

import (
	"strconv"

	"github.com/inklabs/rangedb/pkg/crypto"
)

func (e CustomerAddedBirth) AggregateID() string   { return e.ID }
func (e CustomerAddedBirth) AggregateType() string { return "customer" }
func (e CustomerAddedBirth) EventType() string     { return "CustomerAddedBirth" }
func (e *CustomerAddedBirth) Encrypt(encryptor crypto.Encryptor) error {
	var err error
	e.Name, err = encryptor.Encrypt(e.ID, e.Name)
	if err != nil {
		e.Name = ""
		e.Email = ""
		return err
	}

	e.Email, err = encryptor.Encrypt(e.ID, e.Email)
	if err != nil {
		e.Name = ""
		e.Email = ""
		return err
	}

	stringBirthMonth := strconv.Itoa(e.BirthMonth)
	e.BirthMonthEncrypted, err = encryptor.Encrypt(e.ID, stringBirthMonth)
	if err != nil {
		e.BirthMonth = 0
		e.BirthYear = 0
		return err
	}
	e.BirthMonth = 0

	stringBirthYear := strconv.Itoa(e.BirthYear)
	e.BirthYearEncrypted, err = encryptor.Encrypt(e.ID, stringBirthYear)
	if err != nil {
		e.BirthMonth = 0
		e.BirthYear = 0
		return err
	}
	e.BirthYear = 0

	return nil
}
func (e *CustomerAddedBirth) Decrypt(encryptor crypto.Encryptor) error {
	var err error
	e.Name, err = encryptor.Decrypt(e.ID, e.Name)
	if err != nil {
		e.Name = ""
		e.Email = ""
		return err
	}

	e.Email, err = encryptor.Decrypt(e.ID, e.Email)
	if err != nil {
		e.Name = ""
		e.Email = ""
		return err
	}

	decryptedBirthMonth, err := encryptor.Decrypt(e.ID, e.BirthMonthEncrypted)
	if err != nil {
		e.BirthMonth = 0
		e.BirthYear = 0
		return err
	}
	e.BirthMonth, err = strconv.Atoi(decryptedBirthMonth)
	if err != nil {
		e.BirthMonth = 0
		e.BirthYear = 0
		return err
	}
	e.BirthMonthEncrypted = ""

	decryptedBirthYear, err := encryptor.Decrypt(e.ID, e.BirthYearEncrypted)
	if err != nil {
		e.BirthMonth = 0
		e.BirthYear = 0
		return err
	}
	e.BirthYear, err = strconv.Atoi(decryptedBirthYear)
	if err != nil {
		e.BirthMonth = 0
		e.BirthYear = 0
		return err
	}
	e.BirthYearEncrypted = ""

	return nil
}
`
		assert.Equal(t, expectedOut, out.String())
	})
}
