package main

import (
	"bytes"
	"compress/zlib"
	"encoding/base64"
	"flag"
	"io"
	"log"
	"os"
	"path/filepath"
	"strings"
	"text/template"
)

func main() {
	templatesPath := flag.String("path", ".", "path to templates")
	outPath := flag.String("out", ".", "destination path")
	packageName := flag.String("package", os.Getenv("GOPACKAGE"), "package name")
	flag.Parse()

	templates := make(map[string]string)

	err := filepath.Walk(*templatesPath,
		func(path string, info os.FileInfo, err error) error {
			if err != nil {
				return err
			}

			if info.IsDir() {
				return nil
			}

			filePath := strings.TrimPrefix(path, "templates/")
			templates[filePath] = readFileIntoBase64String(path)
			return nil
		})
	if err != nil {
		log.Println(err)
	}

	outputFilePath := *outPath + "/templates_gen.go"
	file, err := os.Create(outputFilePath)
	if err != nil {
		log.Fatalf("unable to create templates file (%s): %v", outputFilePath, err)
	}

	err = fileTemplate.Execute(file, templateData{
		PackageName: *packageName,
		Templates:   templates,
	})
	if err != nil {
		log.Fatalf("unable to write to events file: %v", err)
	}

	closeOrLog(file)
}

func readFileIntoBase64String(filePath string) string {
	file, err := os.Open(filePath)
	if err != nil {
		log.Fatal(err)
	}
	defer closeOrLog(file)

	var buf bytes.Buffer
	base64Encoder := base64.NewEncoder(base64.StdEncoding, &buf)
	zlibWriter := zlib.NewWriter(base64Encoder)

	_, err = io.Copy(zlibWriter, file)
	if err != nil {
		log.Fatal(err)
	}
	_ = zlibWriter.Flush()
	_ = zlibWriter.Close()
	_ = base64Encoder.Close()

	return buf.String()
}

func closeOrLog(c io.Closer) {
	err := c.Close()
	if err != nil {
		log.Printf("failed closing: %v", err)
	}
}

type templateData struct {
	PackageName string
	Templates   map[string]string
}

var fileTemplate = template.Must(template.New("").Parse(`// Code generated by go generate
package {{ .PackageName }}

func GetTemplates() map[string]string {
	return map[string]string{ {{- range $key, $value := .Templates }}
		"{{ $key }}": "{{ $value }}",{{ end }}
	}
}
`))
