import fs from "node:fs";
import path from "node:path";
import yaml from "yaml";
import type { Plugin } from "vite";

interface Options {
    /** root translation file (yaml) */
    source: string;
    /** path to the .d.ts file that should be generated */
    output?: string;
}

/**
 * vite plugin to get automatic type completions for vue-i18n translation functions based on the structure
 * of a yaml translation file
 */
export default function yamlDtsPlugin(options: Options): Plugin {
    const absSource = path.resolve(options.source);
    const absOutput = path.resolve(options.output ?? "typed-i18n.d.ts");

    function yamlToTs(obj: unknown, indent: number = 0): string {
        const space = "  ".repeat(indent);
        if (typeof obj === "object" && obj !== null) {
            return (
                "{\n" +
                Object.entries(obj)
                    .map(([key, value]) => `${space}  ${JSON.stringify(key)}: ${yamlToTs(value, indent+1)}`)
                    .join("\n") +
                `\n${space}}`
            )
        }
        return "string";
    }

    function generateOutput(yamlObj: unknown): string {
        const tsBody = yamlToTs(yamlObj, 1);
        return `/* eslint-disable */
/* prettier-ignore */
// @ts-nocheck
// noinspection ES6UnusedImports
// Generated by vite-plugin-yaml-dts. ‼️ DO NOT MODIFY THIS FILE ‼️
// It's recommended to commit this file.
// Make sure to add this file to your tsconfig.json file as an "includes" or "files" entry.

export {}

declare module "vue-i18n" {
  export interface DefineLocaleMessage ${tsBody}
}
`;
    }

    function updateDts() {
        const yamlContent = fs.readFileSync(absSource, "utf8");
        const yamlObj = yaml.parse(yamlContent);
        const newDts = generateOutput(yamlObj);
        fs.writeFileSync(absOutput, newDts);
        console.log(`[vite-plugin-yaml-dts] Updated ${absOutput}`);
    }

    return {
        name: "vite-plugin-yaml-dts",
        buildStart() {
            updateDts();
        },
        configureServer(server) {
            server.watcher.add(absSource);
            server.watcher.on("change", (changedPath) => {
               if (path.resolve(changedPath) === absSource) updateDts();
            });
        },
    }
}
