﻿using System.Diagnostics;

namespace Gum.InnerThoughts
{
    public enum FactKind
    {
        Invalid,
        Int,
        Bool,
        String,

        /// <summary>
        /// Used when the fact is only a weight which will be applied when picking
        /// the most suitable dialog.
        /// </summary>
        Weight,

        /// <summary>
        /// Used when checking for required components.
        /// </summary>
        Component,
        Float
    }

    [DebuggerDisplay("{Name}")]
    public readonly struct Fact
    {
        /// <summary>
        /// If null, the user did not especify any blackboard and can be assumed the default one.
        /// </summary>
        public readonly string? Blackboard = null;

        public readonly string Name = string.Empty;

        public readonly FactKind Kind = FactKind.Invalid;

        /// <summary>
        /// Set when the fact is of type <see cref="FactKind.Component"/>
        /// </summary>
        public readonly string? ComponentType = null;

        public Fact() { }

        private Fact(FactKind kind) => Kind = kind;

        public Fact(string componentType) => (Kind, ComponentType) = (FactKind.Component, componentType);

        public Fact(string? blackboard, string name, FactKind kind) =>
            (Blackboard, Name, Kind) = (blackboard, name, kind);

        /// <summary>
        /// Creates a fact of type <see cref="FactKind.Weight"/>.
        /// </summary>
        internal static Fact Weight => new(FactKind.Weight);

        /// <summary>
        /// Creates a fact of type <see cref="FactKind.Component"/>.
        /// </summary>
        internal static Fact Component => new(FactKind.Component);
    }
}