import { ICarrierCreateObject } from './ICarrier';
import IOrderProduct, { IOrderProductCreateObject } from './IOrderProduct';
import IUser, { IUserCreateObject } from './IUser';
import IWarehouse, { IWarehouseCreateObject } from './IWarehouse';
import OrderCarrierStatus from '../enums/OrderCarrierStatus';
import OrderWarehouseStatus from '../enums/OrderWarehouseStatus';
import { DBCreateObject, DBRawObject, PyroObjectId } from '../@pyro/db';
import Carrier from '../entities/Carrier';
import DeliveryType from '../enums/DeliveryType';

export interface IOrderCreateObject extends DBCreateObject {
	/**
	 * User which makes an order.
	 * Note: this is not a reference, but value we put here in the moment when creating an Order.
	 * This is needed because we don't want to allow user to
	 * change this data in his profile later so it effects user orders
	 *
	 * @type {IUserCreateObject}
	 * @memberof IOrderCreateObject
	 */
	user: IUserCreateObject;

	/**
	 * Every order go to single warehouse only.
	 * Currently, it's not possible to include items in the order from different warehouses by design!
	 *
	 * @type {(IWarehouseCreateObject | string)}
	 * @memberof IOrderCreateObject
	 */
	warehouse: IWarehouseCreateObject | string;

	/**
	 * Same as for user, it's not a reference but copy of data about products
	 *
	 * @type {IOrderProductCreateObject[]}
	 * @memberof IOrderCreateObject
	 */
	products: IOrderProductCreateObject[];

	/**
	 * Client can confirm order or order can be auto-confirmed
	 *
	 * @type {boolean}
	 * @memberof IOrderCreateObject
	 */
	isConfirmed?: boolean;

	/**
	 * Order can be cancelled by user or by merchant
	 *
	 * @type {boolean}
	 * @memberof IOrderCreateObject
	 */
	isCancelled?: boolean;

	/**
	 * Indicate if Order was fully paid
	 *
	 * @type {boolean}
	 * @memberof IOrderCreateObject
	 */
	isPaid?: boolean;

	/**
	 * Deliver time for order (DateTime when order was actually delivered to customer)
	 *
	 * @type {Date}
	 * @memberof IOrderCreateObject
	 */
	deliveryTime?: Date;

	/**
	 * Time when order processing is finished
	 * i.e order become canceled from customer or get failed during Store preparing etc.
	 *
	 * @type {Date}
	 * @memberof IOrderCreateObject
	 */
	finishedProcessingTime?: Date;

	/**
	 * The time when some carrier start delivery the order
	 *
	 * @type {Date}
	 * @memberof IOrderCreateObject
	 */
	startDeliveryTime?: Date;

	/**
	 * How many seconds more it should take to delivery order to customer
	 * (reset to 0 when order is delivered or when carrier is not deliver any order to the customer)
	 *
	 * @type {number}
	 * @memberof IOrderCreateObject
	 */
	deliveryTimeEstimate?: number;

	/**
	 * Check out OrderWarehouseStatus
	 *
	 * @type {OrderWarehouseStatus}
	 * @memberof IOrderCreateObject
	 */
	warehouseStatus?: OrderWarehouseStatus;

	/**
	 * Check out OrderCarrierStatus
	 *
	 * @type {OrderCarrierStatus}
	 * @memberof IOrderCreateObject
	 */
	carrierStatus?: OrderCarrierStatus;

	/**
	 * Some carrier which responsible to deliver order to the client
	 * (can be empty if order is not planed for delivery yet or not require delivery, e.g. pickup)
	 *
	 * @type {(ICarrierCreateObject | string | null)}
	 * @memberof IOrderCreateObject
	 */
	carrier?: ICarrierCreateObject | string | null;

	stripeChargeId?: string;

	/**
	 * Used for quick destinction between orders made in the same day (for packaging and etc)
	 * This value is autogenerated every day, so next day order numbers can repeat!
	 * To make unique order Id, add order date and make format like: 'orderDate-orderNumber', e.g. '04062018-5'
	 *
	 * @type {number}
	 * @memberof IOrderCreateObject
	 */
	orderNumber?: number;

	/**
	 * Check out DeliveryType
	 *
	 * @type {DeliveryType}
	 * @memberof IOrderCreateObject
	 */
	orderType?: DeliveryType;
}

interface IOrder extends IOrderCreateObject, DBRawObject {
	_id: PyroObjectId;
	user: IUser;
	warehouse: IWarehouse | string;
	products: IOrderProduct[];
	carrier?: Carrier | string | null;
	isConfirmed: boolean;
	isCancelled: boolean;
	isPaid: boolean;
	warehouseStatus: OrderWarehouseStatus;
	carrierStatus: OrderCarrierStatus;
	orderNumber: number;
}

export default IOrder;
