// Copyright 2014 Google Inc. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.devtools.build.lib.actions;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

import java.io.PrintStream;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

/**
 * Class for debugging, that dumps the Action graph to a pseudo-Makefile.
 */
public class MakefileDumper implements Dumper {

  private final Executor executor;
  private final Collection<Artifact> roots;
  private final ActionGraph actionGraph;

  /**
   * @param executor Used for estimating the resource consumption.
   * @param roots The Artifacts from which the traversal of the action graph should start.
   */
  public MakefileDumper(Executor executor, Collection<Artifact> roots,
      ActionGraph actionGraph) {
    this.executor = Preconditions.checkNotNull(executor);
    this.roots = Preconditions.checkNotNull(roots);
    this.actionGraph = Preconditions.checkNotNull(actionGraph);
  }

  @Override
  public void dump(PrintStream out) {
    Preconditions.checkNotNull(out);
    new GraphVisitor(out).visitRoots();
  }

  @Override
  public String getFileName() {
    return "Makefile.blaze";
  }

  @Override
  public String getName() {
    return "Pseudo Makefile";
  }

  // Print the graph as a Makefile.
  private class GraphVisitor extends BipartiteVisitor<Artifact, Action> {

    private final PrintStream out;

    public GraphVisitor(PrintStream out) {
      this.out = out;
    }

    public void visitRoots() {
      printHeader();

      // Emit an "all" rule which matches the root artifacts of the build.
      Collection<Artifact> sortedRoots = sort(roots);
      out.println();
      out.println("all : \\\n  " + prettyArtifacts(sortedRoots, " \\\n  "));
      out.println();
      out.println();

      for (Artifact root : sortedRoots) {
        visitBlackNode(root);
      }
    }

    @Override
    protected void black(Artifact artifact) {
      if (!artifact.isSourceArtifact()) {
        visitWhiteNode(actionGraph.getGeneratingAction(artifact));
      }
    }

    @Override
    protected void white(Action action) {
      printMakefileRule(action);
      for (Artifact input : sort(action.getInputs())) {
        visitBlackNode(input);
      }
    }

    private void printHeader() {
      out.println("# Automatically generated by Blaze.");
      out.println("# This file is for debugging purposes only.");
      out.println("# The format uses Make style syntax, but");
      out.println("# cannot be used to perform builds with Make.");
    }

    private void printMakefileRule(Action action) {
      out.println(prettyArtifacts(sort(action.getOutputs()), " \\\n") + " : \\");
      if (!action.getOutputs().isEmpty()) {
        // TODO(bazel-team): (2011) We should skip Artifacts that are marked having constant
        // metadata.
        // We skip checking dependencies for the volatile build changelist file
        // in the dependency checker.
        out.println("  " + prettyArtifacts(action.getInputs(), " \\\n  "));
      }
      out.println(prettyAction(action));
      out.println("");
    }
  }

  /**
   * Sort the specified set into an undefined but stable order, so that
   * (pseudo-)Makefiles can be meaningfully compared with diff.
   */
  private static Collection<Artifact> sort(Iterable<Artifact> in) {
    List<Artifact> list = Lists.newArrayList(in);
    Collections.sort(list, new Comparator<Artifact>() {
        @Override
        public int compare(Artifact x, Artifact y) {
          return x.getExecPath().toString().compareTo(y.getExecPath().toString());
        }
      });
    return list;
  }

  // Helpers

  private String prettyAction(Action action) {
    ActionOwner owner = action.getOwner();
    ResourceSet estimatedResources = action.estimateResourceConsumption(executor);
    return "\t# Pretty name: " + action.prettyPrint() + "\n"
           + "\t# Progress message: " + action.getProgressMessage() + "\n"
           + "\t# Mnemonic: " + action.getMnemonic() + "\n"
           + "\t# Owner: " + (owner == null ? "none" : owner.getLabel()) + "\n"
           + "\t# Inputs known: " + action.inputsKnown() + "\n"
           + "\t# Key: " + action.getKey() + "\n"
           + "\t# Estimated memory use: " + (estimatedResources == null ? "dynamic\n" :
                                             estimatedResources.getMemoryMb() + " MB\n")
           + "\t# Estimated cpu usage: " + (estimatedResources == null ? "dynamic\n" :
                                            estimatedResources.getCpuUsage() + " processors\n")
           + "\t# Estimated workload: " + action.estimateWorkload() + " units\n";
  }

  private static String prettyArtifacts(Iterable<Artifact> artifacts, String separator) {
    String sep = "";
    StringBuilder buf = new StringBuilder();
    for (Artifact artifact : artifacts) {
      buf.append(sep).append(artifact.getExecPath());
      sep = separator;
    }
    return buf.toString();
  }
}
