# db400topsql.sh
# Creates a table on Postgres as a replica of a DB400 table and copies over the data.
# Author: Jack J. Woehr 2008-08-15
function usage () {
echo
echo "usage: $0 [-h | --help] tablename src_host src_schema src_userid src_password dest_host dest_schema dest_userid dest_password \"primary_key_field primary_key_field ... \" | - "
echo
echo "Creates a new table on postgres dest_host:schema.tablename replicating the"
echo "structure of a db400 source_host:schema.tablename with the list of primary"
echo " key fields supplied on the command line"
echo
echo "The primary key list can optionally be just a dash ( - ) which indicates"
echo "that the primary key should be autogenerated."
echo
echo "If provided, this primary keys list must be in shell quotes as shown,"
echo "each key space-separated from the previous."
echo
echo "$0 then inserts all records from the source into the destination and"
echo "closes the connection."
echo
echo "Note that the command will fail if the table already exists."
echo "--------"
echo "Returns one of the following as a result code:"
echo "   0 on success of all ublu commands (or for -h or --help)"
echo "   1 on failure of any ublu command, aborting execution"
echo "   2 on too few args"
echo
return
}
###
# Main
if [ "$1" = "-h" ]
    then
    usage
    exit 0
fi
if [ "$1" = "--help" ]
    then
    usage
    exit 0
fi
if [ "$#" -ne 10 ]
    then
    usage
    exit 2
fi
# Instance vars
tablename=$1
src_host=$2
src_schema=$3
src_userid=$4
src_password=$5
dest_host=$6
dest_schema=$7
dest_userid=$8
dest_password=$9
primary_keys=${10}
if [ "${primary_keys}" != $"-" ]
then
primary_key_phrase="-pklist \${"
for key in ${primary_keys}
do
primary_key_phrase="${primary_key_phrase} ${key}"
done
primary_key_phrase="${primary_key_phrase} }$"
else
primary_key_phrase= # no primary key list was provided by user
fi
echo
echo "========"
echo "Replicating ${src_host}:${src_schema}.${tablename} for ${src_userid} with ${src_password}"
echo "as ${dest_host}:${dest_schema}.${tablename} with primary keys ${primary_keys} for ${dest_userid} with ${dest_password}"
echo "========"
echo
#cat << ENDENDENDENDEND
# Setting block size has to be thought about more. See com/ibm/as400/access/doc-files/JDBCProperties.html, e.g. :
# db -db as400 -property \${ block size }$ 512 -star ${tablename} -to @_src_AutoReplicate ${src_host} ${src_schema} ${src_userid} ${src_password}
java -Xms4g -Xmx4g -jar /opt/ublu/ublu.jar << ENDENDENDENDEND
put \${ Replicating table structure as specified ... }$
db -db as400 -replicate ${tablename} ${dest_host} postgres ${dest_schema} ${dest_userid} ${dest_password} ${primary_key_phrase} ${src_host} ${src_schema} ${src_userid} ${src_password}
put \${ Querying source database ... }$
db -db as400 -star ${tablename} -to @_src_AutoReplicate ${src_host} ${src_schema} ${src_userid} ${src_password}
put \${ Querying destination database ... }$
db -db postgres -star ${tablename} -to @_dest_AutoReplicate ${dest_host} ${dest_schema} ${dest_userid} ${dest_password}
put \${ Copying records ... }$
rs -insert -from @_src_AutoReplicate -to @_dest_AutoReplicate
put \${ Closing destination database ... }$
rs -closedb @_dest_AutoReplicate
put \${ Closing source database ... }$
rs -closedb @_src_AutoReplicate
put \${ Replication complete. }$
ENDENDENDENDEND
exit $? # return the exit status of Ublu
#End
