/***************************************************************************
* Copyright (c) Johan Mabille, Sylvain Corlay and Wolf Vollprecht          *
* Copyright (c) QuantStack                                                 *
*                                                                          *
* Distributed under the terms of the BSD 3-Clause License.                 *
*                                                                          *
* The full license is in the file LICENSE, distributed with this software. *
****************************************************************************/

#include <algorithm>

#include "gtest/gtest.h"
#include "xtensor/xarray.hpp"
#include "xtensor/xtensor.hpp"
#include "xtensor/xmath.hpp"

// This file is generated from test/files/cppy_source/test_extended_xmath_reducers.cppy by preprocess.py!

namespace xt
{
    using namespace xt::placeholders;

// Tmporary disabling for TBB and OPENMP due to a potential
// race conditoin that must be identified

    /*py
    a  = np.random.random(4*5*6*7).reshape(4,5,6,7)
    w  = np.random.random(4*5*6*7).reshape(4,5,6,7)
    av = np.average(a, weights=w, axis=(0,1))
    */
    TEST(xtest_extended_xmath, average)
    {
        // py_a
        xarray<double> py_a = {{{{0.3745401188473625,0.9507143064099162,0.7319939418114051,
                                  0.5986584841970366,0.1560186404424365,0.1559945203362026,
                                  0.0580836121681995},
                                 {0.8661761457749352,0.6011150117432088,0.7080725777960455,
                                  0.0205844942958024,0.9699098521619943,0.8324426408004217,
                                  0.2123391106782762},
                                 {0.1818249672071006,0.1834045098534338,0.3042422429595377,
                                  0.5247564316322378,0.4319450186421158,0.2912291401980419,
                                  0.6118528947223795},
                                 {0.1394938606520418,0.2921446485352182,0.3663618432936917,
                                  0.4560699842170359,0.7851759613930136,0.1996737821583597,
                                  0.5142344384136116},
                                 {0.5924145688620425,0.0464504127199977,0.6075448519014384,
                                  0.1705241236872915,0.0650515929852795,0.9488855372533332,
                                  0.9656320330745594},
                                 {0.8083973481164611,0.3046137691733707,0.0976721140063839,
                                  0.6842330265121569,0.4401524937396013,0.1220382348447788,
                                  0.4951769101112702}},
                              
                                {{0.0343885211152184,0.9093204020787821,0.2587799816000169,
                                  0.662522284353982 ,0.311711076089411 ,0.5200680211778108,
                                  0.5467102793432796},
                                 {0.184854455525527 ,0.9695846277645586,0.7751328233611146,
                                  0.9394989415641891,0.8948273504276488,0.5978999788110851,
                                  0.9218742350231168},
                                 {0.0884925020519195,0.1959828624191452,0.0452272889105381,
                                  0.3253303307632643,0.388677289689482 ,0.2713490317738959,
                                  0.8287375091519293},
                                 {0.3567533266935893,0.2809345096873808,0.5426960831582485,
                                  0.1409242249747626,0.8021969807540397,0.0745506436797708,
                                  0.9868869366005173},
                                 {0.7722447692966574,0.1987156815341724,0.0055221171236024,
                                  0.8154614284548342,0.7068573438476171,0.7290071680409873,
                                  0.7712703466859457},
                                 {0.0740446517340904,0.3584657285442726,0.1158690595251297,
                                  0.8631034258755935,0.6232981268275579,0.3308980248526492,
                                  0.0635583502860236}},
                              
                                {{0.3109823217156622,0.325183322026747 ,0.7296061783380641,
                                  0.6375574713552131,0.8872127425763265,0.4722149251619493,
                                  0.1195942459383017},
                                 {0.713244787222995 ,0.7607850486168974,0.5612771975694962,
                                  0.770967179954561 ,0.4937955963643907,0.5227328293819941,
                                  0.4275410183585496},
                                 {0.0254191267440952,0.1078914269933045,0.0314291856867343,
                                  0.6364104112637804,0.3143559810763267,0.5085706911647028,
                                  0.907566473926093 },
                                 {0.2492922291488749,0.4103829230356297,0.7555511385430487,
                                  0.2287981654916225,0.076979909828793 ,0.289751452913768 ,
                                  0.1612212872540044},
                                 {0.9296976523425731,0.808120379564417 ,0.6334037565104235,
                                  0.8714605901877177,0.8036720768991145,0.1865700588860358,
                                  0.8925589984899778},
                                 {0.5393422419156507,0.8074401551640625,0.8960912999234932,
                                  0.3180034749718639,0.1100519245276768,0.2279351625419417,
                                  0.4271077886262563}},
                              
                                {{0.8180147659224931,0.8607305832563434,0.0069521305311907,
                                  0.5107473025775657,0.417411003148779 ,0.2221078104707302,
                                  0.1198653673336828},
                                 {0.337615171403628 ,0.9429097039125192,0.3232029320207552,
                                  0.5187906217433661,0.7030189588951778,0.363629602379294 ,
                                  0.9717820827209607},
                                 {0.9624472949421112,0.2517822958253642,0.4972485058923855,
                                  0.3008783098167697,0.2848404943774676,0.0368869473545328,
                                  0.6095643339798968},
                                 {0.5026790232288615,0.0514787512499894,0.2786464642366114,
                                  0.9082658859666537,0.2395618906669724,0.1448948720912231,
                                  0.489452760277563 },
                                 {0.9856504541106007,0.2420552715115004,0.6721355474058786,
                                  0.7616196153287176,0.2376375439923997,0.7282163486118596,
                                  0.3677831327192532},
                                 {0.6323058305935795,0.6335297107608947,0.5357746840747585,
                                  0.0902897700544083,0.835302495589238 ,0.3207800649717358,
                                  0.1865185103998542}},
                              
                                {{0.0407751415547639,0.5908929431882418,0.6775643618422824,
                                  0.0165878289278562,0.512093058299281 ,0.226495775197938 ,
                                  0.6451727904094499},
                                 {0.1743664290049914,0.690937738102466 ,0.3867353463005374,
                                  0.9367299887367345,0.1375209441459933,0.3410663510502585,
                                  0.1134735212405891},
                                 {0.9246936182785628,0.877339353380981 ,0.2579416277151556,
                                  0.659984046034179 ,0.8172222002012158,0.5552008115994623,
                                  0.5296505783560065},
                                 {0.2418522909004517,0.0931027678058992,0.8972157579533268,
                                  0.9004180571633305,0.6331014572732679,0.3390297910487007,
                                  0.3492095746126609},
                                 {0.7259556788702394,0.8971102599525771,0.8870864242651173,
                                  0.7798755458576239,0.6420316461542878,0.0841399649950488,
                                  0.1616287140946138},
                                 {0.8985541885270792,0.6064290596595899,0.0091970516166296,
                                  0.1014715428660321,0.6635017691080558,0.0050615838462187,
                                  0.1608080514174987}}},
                              
                              
                               {{{0.5487337893665861,0.6918951976926933,0.6519612595026005,
                                  0.2242693094605598,0.7121792213475359,0.2372490874968001,
                                  0.3253996981592677},
                                 {0.7464914051180241,0.6496328990472147,0.8492234104941779,
                                  0.6576128923003434,0.5683086033354716,0.0936747678280925,
                                  0.3677158030594335},
                                 {0.2652023676817254,0.2439896433790836,0.9730105547524456,
                                  0.3930977246667604,0.8920465551771133,0.6311386259972629,
                                  0.7948113035416484},
                                 {0.5026370931051921,0.5769038846263591,0.4925176938188639,
                                  0.1952429877980445,0.7224521152615053,0.2807723624408558,
                                  0.0243159664314538},
                                 {0.6454722959071678,0.1771106794070489,0.9404585843529143,
                                  0.9539285770025874,0.9148643902204485,0.3701587002554444,
                                  0.0154566165288674},
                                 {0.9283185625877254,0.4281841483173143,0.9666548190436696,
                                  0.9636199770892528,0.8530094554673601,0.2944488920695857,
                                  0.3850977286019253}},
                              
                                {{0.8511366715168569,0.3169220051562777,0.1694927466860925,
                                  0.5568012624583502,0.936154774160781 ,0.696029796674973 ,
                                  0.570061170089365 },
                                 {0.0971764937707685,0.6150072266991697,0.9900538501042633,
                                  0.140084015236524 ,0.5183296523637367,0.8773730719279554,
                                  0.7407686177542044},
                                 {0.697015740995268 ,0.7024840839871093,0.3594911512197552,
                                  0.2935918442644934,0.8093611554785136,0.8101133946791808,
                                  0.8670723185801037},
                                 {0.9132405525564713,0.5113423988609378,0.5015162946871996,
                                  0.7982951789667752,0.6499639307777652,0.7019668772577033,
                                  0.795792669436101 },
                                 {0.8900053418175663,0.3379951568515358,0.375582952639944 ,
                                  0.093981939840869 ,0.578280140996174 ,0.0359422737967421,
                                  0.4655980181324602},
                                 {0.5426446347075766,0.2865412521282844,0.5908332605690108,
                                  0.0305002499390494,0.0373481887492144,0.8226005606596583,
                                  0.3601906414112629}},
                              
                                {{0.1270605126518848,0.5222432600548044,0.7699935530986108,
                                  0.2158210274968432,0.6228904758190003,0.085347464993768 ,
                                  0.0516817211686077},
                                 {0.531354631568148 ,0.5406351216101065,0.6374299014982066,
                                  0.7260913337226615,0.9758520794625346,0.5163003483011953,
                                  0.322956472941246 },
                                 {0.7951861947687037,0.2708322512620742,0.4389714207056361,
                                  0.078456381342266 ,0.0253507434154575,0.9626484146779251,
                                  0.8359801205122058},
                                 {0.695974206093698 ,0.4089529444142699,0.1732943200708458,
                                  0.156437042671086 ,0.2502428981645953,0.5492266647061205,
                                  0.7145959227000623},
                                 {0.6601973767177313,0.2799338969459428,0.9548652806631941,
                                  0.7378969166957685,0.5543540525114007,0.6117207462343522,
                                  0.4196000624277899},
                                 {0.2477309895011575,0.3559726786512616,0.7578461104643691,
                                  0.0143934886297559,0.1160726405069162,0.0460026420217527,
                                  0.0407288023189701}},
                              
                                {{0.8554605840110072,0.7036578593800237,0.4741738290873252,
                                  0.0978341606510015,0.4916158751168324,0.4734717707805657,
                                  0.1732018699100152},
                                 {0.433851649237973 ,0.3985047343973734,0.6158500980522165,
                                  0.6350936508676438,0.0453040097720445,0.3746126146264712,
                                  0.6258599157142364},
                                 {0.5031362585800877,0.8564898411883223,0.658693631618945 ,
                                  0.1629344270814297,0.0705687474004298,0.6424192782063156,
                                  0.0265113105416218},
                                 {0.5857755812734633,0.9402302414249576,0.575474177875879 ,
                                  0.3881699262065219,0.6432882184423532,0.4582528904915166,
                                  0.5456167893159349},
                                 {0.9414648087765252,0.3861026378007743,0.9611905638239142,
                                  0.9053506419560637,0.1957911347892964,0.0693613008751655,
                                  0.1007780013774267},
                                 {0.0182218256515497,0.0944429607559284,0.6830067734163568,
                                  0.071188648460229 ,0.3189756302937613,0.8448753109694546,
                                  0.0232719357358259}},
                              
                                {{0.8144684825889358,0.2818547747733999,0.1181648276216563,
                                  0.6967371653641506,0.628942846779884 ,0.877472013527053 ,
                                  0.7350710438038858},
                                 {0.8034809303848486,0.2820345725713065,0.1774395437797228,
                                  0.7506147516408583,0.806834739267264 ,0.9905051420006733,
                                  0.4126176769114265},
                                 {0.3720180857927832,0.7764129607419968,0.3408035402530178,
                                  0.9307573256035647,0.8584127518430118,0.4289940273750183,
                                  0.7508710677914974},
                                 {0.7545428740846823,0.1031238688359326,0.9025529066795667,
                                  0.5052523724478571,0.8264574661077416,0.3200496010306118,
                                  0.8955232284962005},
                                 {0.3892016787341631,0.0108376514802984,0.9053819764192637,
                                  0.0912866767861336,0.3193136375904149,0.9500619670508049,
                                  0.9506071469375561},
                                 {0.5734378881232861,0.6318372121697993,0.4484455219783198,
                                  0.2932107716980645,0.328664545369916 ,0.6725184560770384,
                                  0.75237452943768  }}},
                              
                              
                               {{{0.7915790437258485,0.7896181427945539,0.0912061030486904,
                                  0.4944203047025815,0.0575587600166443,0.5495288823237355,
                                  0.441530501373377 },
                                 {0.8877041827582998,0.3509150125520787,0.1170670164276059,
                                  0.1429916820528359,0.7615106317174722,0.6182180633162611,
                                  0.1011226761227902},
                                 {0.0841068061149974,0.70096913145912  ,0.0727630063641935,
                                  0.8218600592903562,0.7062422271564962,0.0813487806418998,
                                  0.0848377140851919},
                                 {0.9866395785011755,0.3742707957561203,0.3706421470668909,
                                  0.8127995672575026,0.9472485773838587,0.9860010638228709,
                                  0.7533781852589416},
                                 {0.3762595855309158,0.0835007166986688,0.7771469159274368,
                                  0.558404249735805 ,0.4242220092469763,0.906354385094736 ,
                                  0.1111974823061513},
                                 {0.4926251042908592,0.0113536447674191,0.4686606419941263,
                                  0.0563032756818374,0.1188179162680719,0.1175262467771049,
                                  0.6492103021160636}},
                              
                                {{0.7460448792654233,0.5833687650971596,0.9621725484745419,
                                  0.3748705795237041,0.2857120862818607,0.8685991281894603,
                                  0.2235958385194526},
                                 {0.9632225394406113,0.0121544746898163,0.969878826707639 ,
                                  0.0431599119505761,0.8911431136980711,0.5277011090862999,
                                  0.9929647961193003},
                                 {0.0737965647353989,0.5538542844013208,0.969302535619099 ,
                                  0.5230978441701488,0.6293986381352625,0.6957486889846172,
                                  0.4545410647677732},
                                 {0.6275580800840634,0.5843143119231002,0.901158010490989 ,
                                  0.0454463803414579,0.2809631895922303,0.9504114840765587,
                                  0.8902637838909163},
                                 {0.4556567527857129,0.6201325978015367,0.2773811829811327,
                                  0.1881211597237613,0.4636984049399822,0.3533522280260528,
                                  0.5836561118508721},
                                 {0.0777346369649848,0.9743948076661665,0.9862107444796029,
                                  0.6981617140197451,0.5360963663441204,0.3095276162863277,
                                  0.8137950197069487}},
                              
                                {{0.6847311725538793,0.1626169393448913,0.9109271844938425,
                                  0.822537242923169 ,0.949799913291924 ,0.72571950838836  ,
                                  0.6134151959357899},
                                 {0.4182430362906189,0.9327284833540133,0.8660638895004084,
                                  0.0452186701061894,0.026366974497252 ,0.3764633668780496,
                                  0.8105533307818329},
                                 {0.9872761293149445,0.1504168911035282,0.5941307153521351,
                                  0.3808908566310215,0.9699143978146032,0.8421189231357087,
                                  0.8383287047111379},
                                 {0.4686931597949703,0.4148195023376652,0.2734070719307062,
                                  0.0563754966509271,0.8647223762550532,0.8129010091300776,
                                  0.9997176732861306},
                                 {0.9966368370739054,0.5554317056026274,0.7689874151805105,
                                  0.944765729882428 ,0.8496473906774115,0.2473481017431977,
                                  0.4505441353100935},
                                 {0.129159415151495 ,0.9540510272587224,0.60617463445088  ,
                                  0.2286428055034627,0.6717006844058567,0.6181282404578958,
                                  0.3581627180328405}},
                              
                                {{0.113557592199629 ,0.6715731955927996,0.5203077009037933,
                                  0.7723183917356393,0.5201635011119934,0.8521815003185401,
                                  0.5519068387744855},
                                 {0.5609379715353863,0.876653602658345 ,0.403482866212397 ,
                                  0.1340152284506407,0.028782676313339 ,0.755137255673619 ,
                                  0.6203095513534647},
                                 {0.7040797680992236,0.2129641615089107,0.1363714755867698,
                                  0.0145446656678819,0.350587558806597 ,0.5899176868546331,
                                  0.3922440450997323},
                                 {0.4374749220237291,0.9041586944937484,0.3482554670233003,
                                  0.5139894891598108,0.7836530127411431,0.3965427823212702,
                                  0.6220867002278735},
                                 {0.8623637087467452,0.9495206236576421,0.1470734809290379,
                                  0.9265876251614944,0.4921162930795382,0.2582443882989583,
                                  0.4591357562382613},
                                 {0.980032575285477 ,0.4926180939928696,0.3287516102875082,
                                  0.6334008543167258,0.2401456187781931,0.0758633281086639,
                                  0.1288797219106492}},
                              
                                {{0.1280458389577724,0.1519026935122944,0.1388271726494101,
                                  0.6408747448032146,0.1818800843991448,0.3456672833238632,
                                  0.8967884099060118},
                                 {0.4739616402628724,0.6675577385210272,0.1723198712016298,
                                  0.1922890188086708,0.0408686162664789,0.1689350630721645,
                                  0.2785903390319586},
                                 {0.1770104842767468,0.0887025337570556,0.1206358711006008,
                                  0.4607787680327258,0.2063337184057925,0.3642698610480755,
                                  0.5034172708548569},
                                 {0.6903948286293653,0.0393121398410989,0.7994103989090426,
                                  0.6279003894909078,0.0817590319488719,0.8735786241067772,
                                  0.9208724005318132},
                                 {0.0610779598548638,0.2768776481472037,0.8062012797930613,
                                  0.7482596903836584,0.1845210193563773,0.2093493233367103,
                                  0.370472102791382 },
                                 {0.4845229851910213,0.618254771530296 ,0.3689136395697724,
                                  0.4625347161331479,0.7474709381337565,0.0366832028905979,
                                  0.2524369443440208}}},
                              
                              
                               {{{0.7133495858845524,0.8952068376871993,0.5116774421156661,
                                  0.5321134852653157,0.1071720113397761,0.4474123668234546,
                                  0.5326172664550232},
                                 {0.2424705036347297,0.269243230949381 ,0.3772841631046226,
                                  0.0200711977777264,0.3220791655831783,0.2114480069965446,
                                  0.3274973521779146},
                                 {0.1197621318192512,0.890527280739895 ,0.5935924535540487,
                                  0.6791023191444896,0.7891712386073383,0.4984421989290573,
                                  0.0869202880874237},
                                 {0.5371065418185478,0.5868411180208791,0.74543947418433  ,
                                  0.4316595462296794,0.1275803027955638,0.2837759057987245,
                                  0.3630822963986351},
                                 {0.6459172413316012,0.5707783046689119,0.3560967258978462,
                                  0.9865152487929797,0.6057748193568872,0.2372267917359945,
                                  0.1017824726204037},
                                 {0.152859139184332 ,0.2459577283845081,0.1606813732595557,
                                  0.1865670240513058,0.2850951686938471,0.1733735952947548,
                                  0.8967654246264252}},
                              
                                {{0.0802337456616422,0.5245113895702547,0.4103968269896615,
                                  0.9823786169086064,0.1120389021680523,0.3978555990457416,
                                  0.9694704332753689},
                                 {0.8655071258939803,0.8170720709492799,0.2579028270449398,
                                  0.1708875873900658,0.668643219924431 ,0.9293759891275858,
                                  0.5567628930139298},
                                 {0.5716126894698998,0.2799790936602842,0.7694929331919369,
                                  0.1870437485575234,0.3236792364042437,0.4254364386164168,
                                  0.507610378684455 },
                                 {0.2424097324150802,0.1148368247392035,0.6106200424416326,
                                  0.2886305532402558,0.5812382214226123,0.1543627152742023,
                                  0.4811401018548175},
                                 {0.5325894325515859,0.0518235368224269,0.3366042781939206,
                                  0.1344146769389742,0.0633749704727677,0.9899602323899452,
                                  0.322353844974723 },
                                 {0.8098744458546349,0.2546406547637639,0.6815027222239293,
                                  0.7602278598896866,0.5956387406078443,0.4715761885501584,
                                  0.4118409141472685}},
                              
                                {{0.3488682665429953,0.9295291442478258,0.8306194077877291,
                                  0.9650269106665126,0.1242972234855447,0.7308674752036443,
                                  0.9383404568210378},
                                 {0.1812330661656602,0.0664962673667775,0.741120649290059 ,
                                  0.5744731131799119,0.8418287767582721,0.1397723766262895,
                                  0.7952673118598902},
                                 {0.2016273200477445,0.1636559428657045,0.1642657979309929,
                                  0.8145747202313821,0.6651972206962001,0.5230654247691193,
                                  0.3588304841235025},
                                 {0.8772005408131083,0.3924451074226354,0.8165994394715771,
                                  0.4391349085702184,0.3769444294249076,0.4626797856696064,
                                  0.3013778741641421},
                                 {0.7476093801762511,0.5027203900924792,0.2322126951468173,
                                  0.8995745732745685,0.3838912213732114,0.5435528611139886,
                                  0.9064721109645469},
                                 {0.6242379959139921,0.1168980407083641,0.9398321236134751,
                                  0.6277080530714179,0.3349056146570861,0.1392720726633873,
                                  0.794025189270296 }},
                              
                                {{0.6200727559285135,0.5334610919763215,0.8938925830509576,
                                  0.7885972112245307,0.1516748797327512,0.3117220677955482,
                                  0.2484891398144657},
                                 {0.743946292572677 ,0.0335324347357794,0.5698896848713165,
                                  0.7624586857406905,0.8767656367617495,0.3420817487159075,
                                  0.8212573046720129},
                                 {0.1106317369552072,0.8464522917345182,0.1274886623319824,
                                  0.3972872905603673,0.7972953657795536,0.1499174273487738,
                                  0.2292513952326415},
                                 {0.7222525683930663,0.7200365365460744,0.6411476328852973,
                                  0.6939484444671   ,0.5427244433475962,0.2517990589069528,
                                  0.3456959935039194},
                                 {0.1815977168014257,0.9084505613336283,0.5833917947661205,
                                  0.4008514167636399,0.4620058036441327,0.9472833396118153,
                                  0.1533514031160802},
                                 {0.5862298320167972,0.505888678884466 ,0.6114542354346478,
                                  0.0181101838208405,0.8721239089441515,0.9321182824836124,
                                  0.5651331835892089}},
                              
                                {{0.6966508238768923,0.9224993811772957,0.7072386343133986,
                                  0.1525390429142613,0.5762883601668132,0.6067150463828559,
                                  0.424130671302386 },
                                 {0.7364442356247228,0.9343670147690148,0.9255685129067763,
                                  0.4508393714041321,0.1132380458407553,0.9848411989623346,
                                  0.8388980864459341},
                                 {0.1246626812032668,0.9208418826173723,0.8698963620621283,
                                  0.5188380571260721,0.5912754357449292,0.3990027038701303,
                                  0.0547616388220313},
                                 {0.3351972416459009,0.8028534485980114,0.0046320230046029,
                                  0.3334991716911442,0.3981686935909433,0.5373956029379229,
                                  0.9198556164127605},
                                 {0.3463459943659613,0.3469532018962277,0.7375012481097484,
                                  0.4522179408898072,0.224604822939982 ,0.4524395161326934,
                                  0.1408570203797999},
                                 {0.1763869865062233,0.4983677727394797,0.4189254495045479,
                                  0.9148459010681   ,0.3623938991166331,0.5805883502780435,
                                  0.6322642879195304}}}};
        // py_w
        xarray<double> py_w = {{{{0.0130944565883336,0.6635373720167106,0.1780359668697514,
                                  0.9610703174694551,0.148662727753113 ,0.4146241237270237,
                                  0.0853496680786439},
                                 {0.9968742518459474,0.5021950103312426,0.5953850173200438,
                                  0.0670764773884275,0.7499604703991778,0.2099055930955858,
                                  0.8980542894407137},
                                 {0.2051396404820072,0.1906877206636666,0.0365496678480949,
                                  0.4720669451099992,0.5648411332626164,0.0657086394283524,
                                  0.7755276166950106},
                                 {0.4532888347480276,0.52439026932758  ,0.4407627469382281,
                                  0.400763060875326 ,0.5596403313082179,0.1552402459307125,
                                  0.1819281304952711},
                                 {0.8617856210135173,0.9461154621336327,0.3733093162797529,
                                  0.2707446731435538,0.6439995432390158,0.4087341710980964,
                                  0.0253863556603449},
                                 {0.1561525973661904,0.7159722288473975,0.6589239419101514,
                                  0.0270959925034835,0.2219721619329494,0.2310747965880714,
                                  0.6718927435987284}},
                              
                                {{0.0197105377543642,0.1041085819845738,0.7999160853731894,
                                  0.1785446620543336,0.6527461078518747,0.2381827810467265,
                                  0.0994413927593452},
                                 {0.2431721909994541,0.7222669318556592,0.8556964681062857,
                                  0.8302198645669916,0.3971835296184548,0.6680851365706462,
                                  0.204984295415821 },
                                 {0.2931477302610133,0.8963358185211198,0.013001923510736 ,
                                  0.0855085308544681,0.2078862551460273,0.0265322038738197,
                                  0.1814354350897973},
                                 {0.5830415609696922,0.4214245505924979,0.8926717110769748,
                                  0.8174435617384409,0.341817351697876 ,0.2594234334312925,
                                  0.3796924081672669},
                                 {0.5902949425148077,0.2680636408228763,0.624148907849134 ,
                                  0.4094116521912404,0.5520471808519802,0.4361265291353167,
                                  0.2944657595419177},
                                 {0.9484533069621567,0.7636057941597608,0.1401131757664525,
                                  0.8684679758979128,0.4874311982495136,0.8945522268940914,
                                  0.7998552559473152}},
                              
                                {{0.4252135044692334,0.0224693083201174,0.2686773593849461,
                                  0.5416342146608669,0.6334782198261473,0.2578876854332023,
                                  0.1393560740728241},
                                 {0.8349302367992991,0.9844021807035521,0.5256901823026858,
                                  0.1716792858483035,0.272307326519369 ,0.0183906765474667,
                                  0.914298806560499 },
                                 {0.1177510828901411,0.5765164755142536,0.274055220687206 ,
                                  0.5541780025157934,0.6514203883518642,0.8297418037072016,
                                  0.2064212717606032},
                                 {0.0109958286584805,0.1368856300688073,0.9000186418481051,
                                  0.8738900775625152,0.5974131021703083,0.6005168604336534,
                                  0.6650366745462555},
                                 {0.1753712786234496,0.9144119459249598,0.4187705248920731,
                                  0.3831385282494975,0.5189177052828375,0.046965966775055 ,
                                  0.1662833687560794},
                                 {0.7380336164263704,0.0827986679251265,0.6031521094663882,
                                  0.245349109681321 ,0.3892956140419765,0.2886937367706999,
                                  0.3556727164649491}},
                              
                                {{0.7190459051842455,0.297121715623175 ,0.5664046402968972,
                                  0.4760504021990997,0.6636711653626481,0.936829739324758 ,
                                  0.7325720972102503},
                                 {0.2149403785907568,0.0311831350612847,0.2622640442998251,
                                  0.5950779307002548,0.0514258134842503,0.4963662472012363,
                                  0.5968428489168892},
                                 {0.3342438908169608,0.7709122037458219,0.1065982531337718,
                                  0.0751377817358088,0.7281887562036033,0.4954913162061985,
                                  0.688402396427736 },
                                 {0.4348273386037462,0.2464020332391068,0.8191023176741997,
                                  0.7994158789689793,0.6946964708544268,0.2721451372299627,
                                  0.5902306668690871},
                                 {0.3609738969400268,0.0915820733266342,0.9173135754622428,
                                  0.1368186309189614,0.9502373538208024,0.4460057729579557,
                                  0.1851329288386196},
                                 {0.5419009473783581,0.8729458358764083,0.7322248864095612,
                                  0.8065611478614497,0.6587833667107174,0.6922765645178525,
                                  0.8491956515653193}},
                              
                                {{0.2496680088591859,0.4894249636431405,0.2212094418196022,
                                  0.987668007996647 ,0.9440593396866133,0.0394268113685059,
                                  0.7055751725156885},
                                 {0.9252483174156658,0.1805753451273335,0.5679452305526295,
                                  0.9154882975880418,0.0339459785857988,0.6974202672468399,
                                  0.2973490073725508},
                                 {0.9243961953765304,0.9710582451653678,0.9442664891134339,
                                  0.4742142166574638,0.8620426509893134,0.8445493985350704,
                                  0.3191004732432556},
                                 {0.8289154741506773,0.0370076347154926,0.596269878482053 ,
                                  0.230008837287703 ,0.1205668857772788,0.0769532016292092,
                                  0.6962887758781398},
                                 {0.3398749637680661,0.7247667715287615,0.0653563407989424,
                                  0.3152903378306103,0.5394912923753372,0.7907231648389635,
                                  0.3187525029320699},
                                 {0.6258913764370091,0.8859777482361871,0.6158631881823046,
                                  0.2329594747536338,0.024400781556538 ,0.8700988739009299,
                                  0.0212694108503871}}},
                              
                              
                               {{{0.8747016726841994,0.528937134027212 ,0.9390676985128962,
                                  0.7987832357736654,0.9979341105333375,0.3507118154517102,
                                  0.7671882889311269},
                                 {0.4019309136092422,0.4798756203039088,0.6275054632183701,
                                  0.8736771141863388,0.9840834691992951,0.7682734138645182,
                                  0.4177667821673338},
                                 {0.4213570022770702,0.7375823015888915,0.2387771457683023,
                                  0.1104741131313947,0.3546221576407765,0.2872389916540817,
                                  0.2963081204559901},
                                 {0.2336077510499099,0.0420931896363619,0.0178739347334138,
                                  0.9877223897360315,0.4277731337358623,0.3843266471596817,
                                  0.6796472826930698},
                                 {0.2182538878650642,0.9499611839502253,0.786345014415552 ,
                                  0.0894110023122595,0.4175807757849284,0.8791183075621647,
                                  0.9447320222914103},
                                 {0.4674015112498697,0.6134113892107077,0.1670339460920752,
                                  0.9911686261369765,0.2316717013834434,0.9427317741351257,
                                  0.6496466489923685}},
                              
                                {{0.6077367948788591,0.5126885110165086,0.2306698117177587,
                                  0.1765280320055083,0.2204862090701754,0.1864382621442545,
                                  0.7795844735667533},
                                 {0.3501252591667301,0.0578426765633964,0.9691026301408113,
                                  0.8837858849634257,0.927752283195213 ,0.9949078226464353,
                                  0.1738952492194873},
                                 {0.3962420189026588,0.7582384757040913,0.6960206180537922,
                                  0.1538959063398551,0.8158331249906178,0.2244405718366607,
                                  0.2238176148226231},
                                 {0.5369744228934136,0.5929399348417032,0.5800862078378112,
                                  0.0914868373977549,0.8774608626303804,0.2656000425887061,
                                  0.1295149212828042},
                                 {0.8887480798689772,0.9556514982297534,0.8621276172654506,
                                  0.8095160747248781,0.6552419806390217,0.5508573706091393,
                                  0.0869867599114101},
                                 {0.4084532130706987,0.372688517012314 ,0.2597537837603262,
                                  0.7234201136885849,0.4958757350787708,0.0810462159076479,
                                  0.2201832019498113}},
                              
                                {{0.6832587636595959,0.0761308594903002,0.8512069140487687,
                                  0.4951465270139743,0.480586577326648 ,0.5924077846595176,
                                  0.8246809659251492},
                                 {0.3478092079021931,0.6780161525590636,0.5657319639957911,
                                  0.2670282701694213,0.878629986355158 ,0.7974260216069281,
                                  0.6584518346584255},
                                 {0.8505817290942419,0.8672942009598024,0.7083629767150348,
                                  0.837013328363672 ,0.6974714616692836,0.6801407717603004,
                                  0.6186113782151843},
                                 {0.7527166395576412,0.1586051052930737,0.8808707591989284,
                                  0.871843527745232 ,0.0292472830345591,0.8258167505647627,
                                  0.1288698674734452},
                                 {0.3351188542591419,0.7435082562916078,0.1607598960483082,
                                  0.8179670241190622,0.8321341779577419,0.5074677337608362,
                                  0.0063858717168336},
                                 {0.2870381331749128,0.6169269183757422,0.9811861780274235,
                                  0.6318135270166684,0.2598035810641598,0.6340057030996114,
                                  0.5399853797158555}},
                              
                                {{0.7798453951511436,0.1069806388269584,0.761027902502028 ,
                                  0.5412665786761103,0.9629920038589946,0.3418721660386861,
                                  0.6326218931339481},
                                 {0.9320281055100175,0.1025097279906793,0.9372284872364511,
                                  0.6878857223008134,0.0678370591051719,0.3009635669468163,
                                  0.7081720886452816},
                                 {0.0673506014687717,0.5821704601762733,0.3458830569529567,
                                  0.6209155177674771,0.0457420338125129,0.8715368061523763,
                                  0.9734889691773969},
                                 {0.9688778552856915,0.7496518317429247,0.130086240130632 ,
                                  0.7582631959290226,0.0245869164588015,0.0221235515289973,
                                  0.3236102191495412},
                                 {0.4886431904046658,0.7704074178077932,0.6832953766065052,
                                  0.4459027063767483,0.2736266662816659,0.9971245001577111,
                                  0.4261813022359729},
                                 {0.4513870243296755,0.1636238211916692,0.7948095487499294,
                                  0.6936822257814886,0.2207696127887604,0.0823810456158427,
                                  0.6804993020747131}},
                              
                                {{0.6545112142811352,0.2732595269982093,0.9508635622504104,
                                  0.1510578917809002,0.4323348010426019,0.9436159201675989,
                                  0.4197273169261223},
                                 {0.6385259476640855,0.3975943979627943,0.27421520234867  ,
                                  0.9839776479598282,0.409334006315043 ,0.8940992036791344,
                                  0.2299546058910862},
                                 {0.2131047040250833,0.0311340828845114,0.651666825375885 ,
                                  0.3685263437237611,0.864358249839637 ,0.4732099066915572,
                                  0.9681934279147217},
                                 {0.1855255157031211,0.8686231679556108,0.7765968527917417,
                                  0.770921844644052 ,0.8447832281168877,0.7610239909429989,
                                  0.6262203216314154},
                                 {0.1312448776823876,0.0325261794912519,0.9208478478156868,
                                  0.616650314520738 ,0.7965372909761763,0.4815223515125501,
                                  0.1173081889623945},
                                 {0.1251857922025504,0.6855652872289714,0.4303058948994627,
                                  0.2005247267003367,0.4915945467414369,0.0642089370751341,
                                  0.5819714019143988}}},
                              
                              
                               {{{0.2689934044350977,0.7975591006371882,0.3103619589235738,
                                  0.4552201490818012,0.0116205399081006,0.0724468877945589,
                                  0.3924935564066848},
                                 {0.4799388347171235,0.6000205481193037,0.2916625787093032,
                                  0.6949818861265425,0.8601223971892046,0.7798509888933498,
                                  0.0396188253483571},
                                 {0.4805069472578289,0.1049301784181749,0.2420450158625275,
                                  0.9866625932671464,0.1424955429015132,0.4988881534513561,
                                  0.6181557343181191},
                                 {0.702464970544104 ,0.5596486834869141,0.0097708474191837,
                                  0.3264613082439919,0.5177116433847114,0.0878664991448334,
                                  0.3506269312091808},
                                 {0.0332031087913666,0.0785784971550207,0.3969232762015961,
                                  0.1327157540430604,0.5675408482615582,0.6894649691372657,
                                  0.8005866991090829},
                                 {0.2001502442448101,0.1674825822590698,0.1045678403344003,
                                  0.6364302495436367,0.7064757264869012,0.031586144825642 ,
                                  0.9362122462436898}},
                              
                                {{0.0519712836514724,0.5412963353010705,0.7090605194509164,
                                  0.8709691237460856,0.7140869321324278,0.8017280830697918,
                                  0.3394501925428061},
                                 {0.8148251137465122,0.0801148463846751,0.8948166560605276,
                                  0.5475923761537362,0.8172977699624937,0.4523182845183001,
                                  0.643577695196525 },
                                 {0.5264026609361132,0.7315895217553319,0.0816299820305896,
                                  0.0603520839905638,0.2471032340101464,0.1595446801131883,
                                  0.8717835665922017},
                                 {0.2192139873580443,0.9758652558191313,0.3368957917711068,
                                  0.1821179156886993,0.7896985071424791,0.6587077755008761,
                                  0.498195716453139 },
                                 {0.5553635509376312,0.7192017782722638,0.2284547413312987,
                                  0.9963339160567419,0.9747931621467831,0.6503256863469369,
                                  0.1995424509291451},
                                 {0.6802282424312913,0.0721984089791758,0.0306525022058061,
                                  0.2576828885112137,0.4626229567393163,0.8682725054083805,
                                  0.727169069766308 }},
                              
                                {{0.742706521199981 ,0.4254933344480754,0.3459349925469629,
                                  0.3710387629846027,0.9876495637360578,0.0401091914124825,
                                  0.8670314961224486},
                                 {0.5786754085723934,0.4386154191895905,0.7252576604151365,
                                  0.4866689414247029,0.8734232380816606,0.9007018640112585,
                                  0.4217209268734554},
                                 {0.2768277972350516,0.5923503285933622,0.912363345616691 ,
                                  0.2106621890056666,0.6229665835634828,0.6315602200925206,
                                  0.733113022415281 },
                                 {0.1315676851272598,0.7158249646820886,0.909032520665641 ,
                                  0.179683108870242 ,0.2375433249238776,0.9713950940416396,
                                  0.1809769527094898},
                                 {0.8543850933695791,0.4922778564480348,0.247231074403177 ,
                                  0.8707499012725114,0.4453052550026655,0.5148173539296902,
                                  0.3592333693997657},
                                 {0.5929508514349274,0.1635238725850228,0.3910815366517607,
                                  0.9694123223352875,0.2581334327011275,0.6567366645412922,
                                  0.3251900642246949}},
                              
                                {{0.7734731256866006,0.1308736607183498,0.9698210450785447,
                                  0.4537895413836392,0.236050463346464 ,0.0734967473300127,
                                  0.1697579050875407},
                                 {0.5197739485560177,0.3370031764312863,0.8288833658826095,
                                  0.4308875236618059,0.2487142725876299,0.61714498660403  ,
                                  0.7067772168854458},
                                 {0.1670419079009499,0.1676192162883177,0.036671426933543 ,
                                  0.7364020150656405,0.6638045276218051,0.4746308757498139,
                                  0.8441704489691971},
                                 {0.8056701529500532,0.5853543643967521,0.8682712805132586,
                                  0.2058412100367683,0.1119196193977279,0.2697496115169896,
                                  0.0570868560893149},
                                 {0.531169528001032 ,0.9366056922949615,0.0393435406685096,
                                  0.1221099140100268,0.4521990282834353,0.9338750175279049,
                                  0.3161561049767505},
                                 {0.5072348086883924,0.0415728590503921,0.1483432009620889,
                                  0.9866301229588469,0.9651186964360455,0.0049399809344096,
                                  0.951811785423239 }},
                              
                                {{0.6391199378155004,0.8679182945200221,0.4547398556338591,
                                  0.5155960285792502,0.4888465802569202,0.6668642575461043,
                                  0.1396512547563129},
                                 {0.029973589872678 ,0.3079299415911909,0.7046807627366467,
                                  0.2018534521229484,0.6734324333249173,0.9699120461072704,
                                  0.0939007157894184},
                                 {0.6726021182251263,0.4437502193045711,0.8681422543775054,
                                  0.1771497894473859,0.6926259522261645,0.8381152896481102,
                                  0.9446142194740232},
                                 {0.6832480282896458,0.4971747640084273,0.6178472402012138,
                                  0.8689049844787182,0.5706097466388297,0.0303870596968369,
                                  0.9309486955069778},
                                 {0.6895267510315611,0.6765133857772465,0.2156751523977172,
                                  0.6588854702326572,0.3938644056541585,0.6512329770414921,
                                  0.1065930303079907},
                                 {0.657845304603036 ,0.9994137257706666,0.048212038860556 ,
                                  0.9771741842213173,0.4069079607228706,0.8707534503372591,
                                  0.7823854840582735}}},
                              
                              
                               {{{0.5670162609866678,0.7384492092513265,0.8785155613013479,
                                  0.404140321651033 ,0.3270331615632387,0.6675933855635847,
                                  0.8078459419412586},
                                 {0.7622851347766157,0.7978136488586345,0.435583314812933 ,
                                  0.8178342161750216,0.1202090553706084,0.5444890979608495,
                                  0.0057586604981216},
                                 {0.3245858299002703,0.3664615348713979,0.3961726916092554,
                                  0.6954672066898622,0.3885581012703457,0.4486936226225307,
                                  0.2375441308743275},
                                 {0.3732517916293401,0.2272696272631515,0.0731959237684436,
                                  0.6034485933813779,0.6682127985550163,0.6194903460096145,
                                  0.4634940438005724},
                                 {0.3797857801792298,0.8633336495718252,0.519081785136218 ,
                                  0.479181877629731 ,0.0256420658064357,0.3412478276261564,
                                  0.3801956187858571},
                                 {0.3988227808969005,0.5801723692273323,0.5336025467268785,
                                  0.6079050927934446,0.7648832615421283,0.8129857387160935,
                                  0.7181230762264944}},
                              
                                {{0.9555236959549347,0.0182325826763616,0.1957779856835461,
                                  0.0075628749781421,0.647474714154603 ,0.8980305489499869,
                                  0.2434822965762993},
                                 {0.9270345447963066,0.0602673902895609,0.9344360268055143,
                                  0.3516226872396344,0.1014208284492165,0.4858717591863256,
                                  0.2567765677205079},
                                 {0.2848729019619228,0.3072899649262866,0.8030258978326634,
                                  0.5391612766752545,0.3113076991560036,0.6103337833109184,
                                  0.7161506742231936},
                                 {0.2726240007335338,0.4135491015253515,0.1218860933369476,
                                  0.1811493495346803,0.6811178539649828,0.1814383476956269,
                                  0.5251633836670602},
                                 {0.7090462617224327,0.1068769230947438,0.5673122196289035,
                                  0.2565627837013638,0.9629268752375798,0.4835456467199247,
                                  0.8059925497751581},
                                 {0.5502265421562618,0.0434125328548192,0.6331513755940223,
                                  0.951403342206657 ,0.6016118201463312,0.8191888594267902,
                                  0.8842064633484078}},
                              
                                {{0.2280797719489325,0.2120448399082266,0.6109809888509917,
                                  0.4110284698067047,0.8398613028667203,0.9000231233980112,
                                  0.3534213793027181},
                                 {0.2368705589744821,0.7805255152438618,0.2748060355481361,
                                  0.8226143185136784,0.4237382536487628,0.6675498992914705,
                                  0.0955353138346682},
                                 {0.623859324676573 ,0.4517676787372291,0.5866084631754741,
                                  0.1680142077448634,0.7368737449441821,0.8627970775432285,
                                  0.2167398000907756},
                                 {0.0957145555201617,0.0236385860463364,0.641971500443266 ,
                                  0.6070940360887808,0.54669741271407  ,0.2319470961159754,
                                  0.3909059959055147},
                                 {0.594476335186173 ,0.4967668588591694,0.9877855202148873,
                                  0.1364397525567331,0.6951445540475938,0.4043187681042921,
                                  0.4281996149256148},
                                 {0.7175977656581762,0.6924361513243696,0.9912559937021883,
                                  0.1283942894541252,0.1041096493841663,0.7243388171287412,
                                  0.5783869170064091}},
                              
                                {{0.2741606665598059,0.0794193689889866,0.0856582495964305,
                                  0.8941908746641183,0.1918673247187429,0.3233715621155208,
                                  0.226656402226988 },
                                 {0.3549963058319373,0.0694238427460934,0.5190597909082704,
                                  0.0676125635922149,0.8003565078223616,0.233712081910433 ,
                                  0.5400119146045106},
                                 {0.8800790875614583,0.6508773690471942,0.5329577865513353,
                                  0.3243337531153756,0.3330019130581979,0.6694869551981901,
                                  0.9941393612211675},
                                 {0.6618391898493786,0.5577834173700249,0.7306505102392461,
                                  0.4652056122837182,0.0601423426007852,0.5622968158779098,
                                  0.9576252845053802},
                                 {0.1753029423576596,0.6900048908206657,0.2009336886590565,
                                  0.5358276844730813,0.0966764496467196,0.4503709362581851,
                                  0.7561633288952461},
                                 {0.3475715242389821,0.664911724487218 ,0.7954499607897292,
                                  0.9271778194794527,0.2346420818554535,0.3993159158093081,
                                  0.1524160133304174}},
                              
                                {{0.9924835021583752,0.9270009679242953,0.5399571302805596,
                                  0.8420332943764213,0.5209579772720422,0.6235856880027597,
                                  0.0891244321160166},
                                 {0.7552704204968079,0.1277134836352348,0.8260676302525284,
                                  0.7820280876703859,0.7087446977429764,0.0361603805247621,
                                  0.3031283559453286},
                                 {0.2631125699108429,0.3601364074715191,0.0876427467638385,
                                  0.9369578230310628,0.5538022407123763,0.3055243106580235,
                                  0.3969815170742834},
                                 {0.4472025405048357,0.6005943333606173,0.5156794276964978,
                                  0.9193919732144173,0.4969634829058303,0.9921580148233127,
                                  0.8514249577450957},
                                 {0.2085105143573358,0.9305952146439163,0.1163663978854761,
                                  0.8174497084911916,0.3806232934765801,0.8779743204450695,
                                  0.8680566901074503},
                                 {0.8059254001176211,0.7900304363070614,0.3046791390980408,
                                  0.0809192830512522,0.4029801787651615,0.1735245149865087,
                                  0.6949510889024617}}}};
        // py_av
        xarray<double> py_av = {{0.5106795969797712,0.6397044109091891,0.5112635235731234,
                                 0.4586765467957137,0.4869877873873078,0.4943932998356453,
                                 0.4385520508602098},
                                {0.5978927985524186,0.559905994727937 ,0.619729942105118 ,
                                 0.4653893064895739,0.5937783013353551,0.5048245639269587,
                                 0.5831376677923145},
                                {0.4165551320778472,0.4546935552796757,0.3973719402163453,
                                 0.4544365450264588,0.5634293281574402,0.4992966676969995,
                                 0.4805121186481612},
                                {0.5571737102656098,0.4734357748888989,0.5334877219467363,
                                 0.4183775498265881,0.4971210216796947,0.475560939392655 ,
                                 0.5715317612540888},
                                {0.6780835926478442,0.4848427790974752,0.5816075391773083,
                                 0.5583959492918773,0.4287789529418257,0.4549499764850471,
                                 0.2909222329544836},
                                {0.455560911393571 ,0.4523219422883247,0.5616099216331595,
                                 0.3817307280172807,0.4162942894756477,0.2873504573593249,
                                 0.423361164166236 }};

        auto av = xt::average(py_a, py_w, {0,1});
        EXPECT_TRUE(xt::allclose(av, py_av));
    }

    /*py
    a  = np.random.random(4*5*6*7).reshape(4,5,6,7)
    vr_all = np.var(a)
    st_all = np.std(a)
    vr = np.var(a, axis=(0,2))
    st = np.std(a, axis=(0,2))
    */
    TEST(xtest_extended_xmath, var_std)
    {
        // py_a
        xarray<double> py_a = {{{{0.3460997265377855,0.9756102008599149,0.640972077345332 ,
                                  0.822480563634361 ,0.1325246727418484,0.8620144822726226,
                                  0.9227571905133943},
                                 {0.4870619188189359,0.6062529377509774,0.7648098009029954,
                                  0.1748386272604125,0.5025660764634424,0.3986630277723131,
                                  0.1463739921622491},
                                 {0.3675344222915314,0.0681723093854936,0.025811906949487 ,
                                  0.1351662891548228,0.9631151115891738,0.549529535895017 ,
                                  0.96582216115322  },
                                 {0.432497878330132 ,0.3118161330852596,0.5061418628402441,
                                  0.4395116886329101,0.1056646842617439,0.6408263146902062,
                                  0.2160381825095996},
                                 {0.6195879548978296,0.6502010923981534,0.1520248531286481,
                                  0.0613496271106682,0.7807615860684398,0.4598004237693202,
                                  0.0581637955054758},
                                 {0.9948663171697671,0.0577805609963666,0.6950352288585532,
                                  0.9836789252634136,0.2391801042159256,0.1422493686871962,
                                  0.1213849394224289}},
                              
                                {{0.3032751475675955,0.1010458110106467,0.6921613416941319,
                                  0.0622917981300627,0.5094221281497897,0.9966968538529768,
                                  0.8139702695060188},
                                 {0.6152194382310109,0.3062536207866117,0.6238958362916385,
                                  0.5270414639116516,0.4260833838903715,0.1307103767401995,
                                  0.8866042152404667},
                                 {0.4497846488213225,0.1946225118207651,0.3677593508772896,
                                  0.414129752528927 ,0.8275378955668743,0.733614380561402 ,
                                  0.7693048936413398},
                                 {0.0110312644286472,0.4161539981797615,0.4813441516353236,
                                  0.0191922768230567,0.2598132087450342,0.760289823122135 ,
                                  0.1371097057289991},
                                 {0.5353101314525917,0.2152018720172375,0.0121207746438933,
                                  0.2412014575995415,0.9758737549100538,0.8015371118829896,
                                  0.9595766443210334},
                                 {0.4878540439263624,0.1097361986775035,0.547959488973693 ,
                                  0.4543773345599609,0.844357082739616 ,0.0980825807842446,
                                  0.4882411370217782}},
                              
                                {{0.1500486646939843,0.32467592689006  ,0.7373570773877997,
                                  0.4760181293687665,0.3758882851457693,0.3944762785700353,
                                  0.4594467684059866},
                                 {0.7850165438370569,0.8920846889402344,0.9553346865039283,
                                  0.7869033862552951,0.3154068957480127,0.6881347060865421,
                                  0.4376031240553919},
                                 {0.2546706220460512,0.8408715794259226,0.038426348962325 ,
                                  0.9017619921354235,0.4614774646247342,0.6372014768082804,
                                  0.6593539222057827},
                                 {0.8951177407065364,0.6366696821672073,0.6139335847600618,
                                  0.06665204056069  ,0.5184080219452041,0.1501690008170417,
                                  0.7374337687214856},
                                 {0.5122219208126745,0.6802277792083533,0.0416729006023843,
                                  0.0847920160458032,0.7163233880626823,0.0720843359986734,
                                  0.0712567277652586},
                                 {0.0121084752348875,0.9565013982994801,0.737508359459007 ,
                                  0.3532514085020642,0.2965355812876552,0.3497032297458496,
                                  0.7746535311811819}},
                              
                                {{0.6613706110704144,0.1851955678674679,0.1741093353340843,
                                  0.0983956497133973,0.6603027197013381,0.7643726622217359,
                                  0.265046428325004 },
                                 {0.0209449610309128,0.0821716674764273,0.9678600348450305,
                                  0.2954447774774203,0.7692231478381336,0.6246635688542638,
                                  0.3819396396528959},
                                 {0.2056872621955551,0.1213864181594064,0.6150129680388666,
                                  0.774633780637788 ,0.6439042535411307,0.5303021332491583,
                                  0.0419512234316363},
                                 {0.9684887775943838,0.7987141916545746,0.2928220367009647,
                                  0.9799703293588506,0.6018815896363221,0.5824226582072465,
                                  0.7480731794295321},
                                 {0.8117697880901376,0.6564786073315027,0.1280957463662   ,
                                  0.3382675059944984,0.9280836403260497,0.2246156118438519,
                                  0.3721670238375133},
                                 {0.432076882944196 ,0.4394049896944522,0.6129395824945255,
                                  0.9430758377447076,0.240692711832505 ,0.1215013763009181,
                                  0.1974704898599373}},
                              
                                {{0.8869249002092695,0.645810812655206 ,0.2859067892391601,
                                  0.8159469389686173,0.861369996363436 ,0.846514338690278 ,
                                  0.9189265336395545},
                                 {0.2522410201733616,0.7550419287462783,0.46053949727414  ,
                                  0.8419985528900615,0.7284906771616132,0.7764474464997246,
                                  0.6561618376810715},
                                 {0.1774287694697069,0.5450269188969383,0.9846697395245082,
                                  0.9373880664971189,0.0431737357970117,0.1648148157294371,
                                  0.1317287734545619},
                                 {0.7259799099852955,0.8177853298569451,0.2135113663106719,
                                  0.505852692951829 ,0.8407030280112   ,0.7328015447662506,
                                  0.5422372054918483},
                                 {0.5903476904587729,0.50836054691628  ,0.2975484513188206,
                                  0.5650219928590161,0.6888853020500907,0.8733229158608077,
                                  0.6362913539792481},
                                 {0.761121537043158 ,0.160071634954573 ,0.4615574744197899,
                                  0.0093316198270933,0.246678864521623 ,0.7264617151130048,
                                  0.9918099513493635}}},
                              
                              
                               {{{0.0991780995872159,0.4014943158612883,0.8000709686991595,
                                  0.2040356346288988,0.5550849493022862,0.7330712960296748,
                                  0.615985450260179 },
                                 {0.1880247346154161,0.3553845686857394,0.7837917844853972,
                                  0.5542265141320166,0.0052296135429127,0.7609907600280938,
                                  0.0353113549402391},
                                 {0.7457337827312321,0.2024805601652374,0.9580734801193802,
                                  0.3679407513020845,0.3269316163380791,0.1488880503324447,
                                  0.3056042154902149},
                                 {0.8766507618282006,0.9963343376272473,0.3683095305068483,
                                  0.4486106309545426,0.722070938202737 ,0.8861957804650745,
                                  0.5930443341867842},
                                 {0.3915256981778293,0.4126218408486237,0.6956181464968069,
                                  0.0032182636042787,0.6195893374598407,0.3554930104413574,
                                  0.7941973312770843},
                                 {0.0929906398231322,0.5882022669483905,0.4809728898861237,
                                  0.6423255259052274,0.0648535944829683,0.5799837874281313,
                                  0.5614845879595237}},
                              
                                {{0.5606600944161709,0.6034876665964548,0.6764679372263622,
                                  0.8049889996807698,0.2698207201282143,0.8250494101800621,
                                  0.4982556822634536},
                                 {0.0770582791343312,0.0585509292439792,0.3342383186960608,
                                  0.7848969771114676,0.7076809347409158,0.7886149646724064,
                                  0.5172690564416094},
                                 {0.4401989979554906,0.1474525267062839,0.3281927536093115,
                                  0.4340193661738753,0.0886004296796172,0.2206119527112113,
                                  0.5982252940637388},
                                 {0.7356631142391898,0.9983475113929496,0.9331133313435577,
                                  0.6425651995351915,0.421248053294072 ,0.6361773647082327,
                                  0.785651618080232 },
                                 {0.1183361909233873,0.4099048896962882,0.8398022856356172,
                                  0.383832951933044 ,0.5718722712271865,0.5877693608621982,
                                  0.1844762531336394},
                                 {0.3622354416522884,0.3345112887419321,0.0261967085320648,
                                  0.0241917638913552,0.8316970436757735,0.2730708099728124,
                                  0.5180787662388154}},
                              
                                {{0.2987255735887731,0.9406792451153405,0.2592967588657737,
                                  0.4296568127389351,0.8727302504846018,0.8419335667918086,
                                  0.1861014178462773},
                                 {0.8026433097957104,0.4581868867378927,0.4829688720853552,
                                  0.1334799724159438,0.0806015137571247,0.7279393069737652,
                                  0.4964611523552998},
                                 {0.4368507026151796,0.7295082286523967,0.7655128989911097,
                                  0.1589081676693536,0.6102251494776413,0.1353540822776855,
                                  0.7513750860290457},
                                 {0.6569551562671395,0.956614621083458 ,0.0689580163564212,
                                  0.0570547211512543,0.2821870746932001,0.261705683735906 ,
                                  0.2469787990719998},
                                 {0.9062545805210367,0.2495461998495061,0.271949726128631 ,
                                  0.7593982624179637,0.44973984245054  ,0.7767105569551812,
                                  0.0653661575643852},
                                 {0.4875711936733843,0.0336136001832826,0.0626532034553545,
                                  0.906437453344411 ,0.1392453711175952,0.5324206822752171,
                                  0.4110956026007065}},
                              
                                {{0.3473433262588427,0.8998333456872725,0.0218233967754895,
                                  0.6637896861755891,0.9633944342135504,0.5601681834618419,
                                  0.9368224620330918},
                                 {0.0522578792860238,0.4187933190741614,0.2601577909620512,
                                  0.7308209649807298,0.9812970904972467,0.2565300624442478,
                                  0.6541746014740745},
                                 {0.1980976327597401,0.5653302545716099,0.4639324866883362,
                                  0.9720053296315371,0.6085272726964666,0.3495063720868188,
                                  0.1140957848879334},
                                 {0.1512468287349346,0.2253169299512351,0.250966661340298 ,
                                  0.8506160578411378,0.5612227882825249,0.5233909113743664,
                                  0.1147688718692175},
                                 {0.860139693349435 ,0.7228143032084837,0.0676683619000016,
                                  0.7078350972297687,0.5435382173426461,0.0817253457467783,
                                  0.4583006415286565},
                                 {0.484696287130047 ,0.1657745401632758,0.9456981470002823,
                                  0.8499753714806619,0.6690223373291083,0.4622955610006072,
                                  0.4117655415845085}},
                              
                                {{0.6509734671509542,0.5454318694069055,0.0622731058818856,
                                  0.5125026487703317,0.8064036172328395,0.4592398775969581,
                                  0.0519565775542684},
                                 {0.7862781993840331,0.2013637821359744,0.2586208348880388,
                                  0.1647063534322959,0.3302150648714975,0.756751505688635 ,
                                  0.5193858595039373},
                                 {0.2048812741507126,0.8778300781520408,0.8795818549515292,
                                  0.8705784250460069,0.2387962113815393,0.4512393441809881,
                                  0.9849896561408751},
                                 {0.7720124763099904,0.0271674212255598,0.0652045912604921,
                                  0.4639313768123328,0.9092202058615602,0.5387017979527142,
                                  0.4978125080342902},
                                 {0.1054736999566306,0.656780104724294 ,0.8221031606002255,
                                  0.3804200075075623,0.7756118509253275,0.9644766539008116,
                                  0.2037664591276891},
                                 {0.5233296311947413,0.2871379580845755,0.7928544190338394,
                                  0.5775933658811575,0.6345824175503038,0.7979141596158941,
                                  0.3959704726624549}}},
                              
                              
                               {{{0.9150900684070432,0.53302886724614  ,0.157954822835348 ,
                                  0.6958991181688025,0.7932613504382544,0.3167616770675267,
                                  0.857179256998753 },
                                 {0.9061432547657664,0.2769044873778089,0.9835214727152221,
                                  0.1407115280487584,0.20201566840249  ,0.1842248392380261,
                                  0.8939897100078704},
                                 {0.6542925523249619,0.1521042832222729,0.440323418368349 ,
                                  0.615298025069802 ,0.0834640849893833,0.8824163684073387,
                                  0.8036035354415352},
                                 {0.5052067735766382,0.9671266152591704,0.4177609792750997,
                                  0.9841103190364978,0.6679199988036271,0.6346712732539415,
                                  0.165954895295785 },
                                 {0.881927758304644 ,0.4274896879206151,0.1622334057824794,
                                  0.012607519975759 ,0.5597556836383826,0.5273996221226763,
                                  0.7193536223691586},
                                 {0.8902580530178021,0.0794065846430295,0.7314964903403623,
                                  0.1874119440975822,0.858177053157164 ,0.8190635138616561,
                                  0.5407937549855444}},
                              
                                {{0.7102427986233115,0.3143501131837484,0.4711678989328146,
                                  0.8216368977104717,0.4592651971732139,0.3577982653254418,
                                  0.4942124472502999},
                                 {0.8282492524372184,0.3352080473708727,0.1737615931357438,
                                  0.7120140203307808,0.8259782127080818,0.1006374785235333,
                                  0.2398741127559447},
                                 {0.1419719897609939,0.3479414115821666,0.4503509789054126,
                                  0.7488263369633065,0.6511475848816741,0.6209285674784312,
                                  0.3523749466096544},
                                 {0.8414475530847599,0.4712871673981646,0.9791049380151545,
                                  0.6341421382957368,0.1262647415645027,0.676177562075434 ,
                                  0.325104015431699 },
                                 {0.6863271966860974,0.0696411481624057,0.1748814371759594,
                                  0.8557372589057863,0.2271795570235149,0.83704112524294  ,
                                  0.2792760884544104},
                                 {0.6428819100587077,0.694150573509831 ,0.5126547531450529,
                                  0.3053106766967313,0.2126443763710961,0.0331893545624262,
                                  0.3039464012407238}},
                              
                                {{0.653162790531283 ,0.9383048184484827,0.871204482263052 ,
                                  0.7660647765353026,0.7884473432532987,0.6649849034759953,
                                  0.260286857566003 },
                                 {0.9071950307058299,0.670732311905834 ,0.56044063098044  ,
                                  0.1109894692134811,0.4470555438064254,0.4603582712019263,
                                  0.86456360952267  },
                                 {0.5466482201877885,0.3804005489704096,0.9768003285106593,
                                  0.1107205933263157,0.4225484283119446,0.0420246751714053,
                                  0.7399041207669304},
                                 {0.9180765521991506,0.2800373295504894,0.8583416767991209,
                                  0.2922182304618978,0.9107701616439152,0.7539619035967839,
                                  0.8049067768414386},
                                 {0.0180065556004545,0.962810357074026 ,0.7266709920703233,
                                  0.3047510953861333,0.8293946648588469,0.28152119841493  ,
                                  0.872753839566128 },
                                 {0.112580390411937 ,0.7036859774607958,0.5406984766449103,
                                  0.0965344701954408,0.241893828258839 ,0.0124040362773148,
                                  0.4687679654645058}},
                              
                                {{0.3012652607600004,0.5983574842977588,0.2972378077086172,
                                  0.2999191115069847,0.7431931987316034,0.0481428359783534,
                                  0.9028949951919872},
                                 {0.8522638743911815,0.6678046832264065,0.5932215701073843,
                                  0.8923024981396522,0.1853301795287144,0.0789691621908286,
                                  0.2395101699095381},
                                 {0.7945782787001903,0.0346702667264833,0.5828068350834896,
                                  0.9954375161913099,0.8556960938375394,0.5214456890435895,
                                  0.0636409092866423},
                                 {0.8313735143857098,0.5989785081772105,0.1149329970595668,
                                  0.0938572848233562,0.9096268194365924,0.6692002629380784,
                                  0.8292867993167555},
                                 {0.8789789002822673,0.5717723520483222,0.5174463517550413,
                                  0.4304274103209902,0.3169465877911193,0.4345959608587118,
                                  0.7738796557301564},
                                 {0.6019234310151316,0.8925232899548461,0.4433800143791877,
                                  0.607089578718754 ,0.6313075530919887,0.5916971623063255,
                                  0.7026337730855163}},
                              
                                {{0.2374334683748063,0.5123637775457907,0.1042248028083477,
                                  0.3845113276448126,0.4876670686212644,0.6522242595454725,
                                  0.9505310525862035},
                                 {0.6006510665483559,0.7435939161765598,0.5062660594952028,
                                  0.6341040313763313,0.0709322161337734,0.254391580841847 ,
                                  0.3618530932051387},
                                 {0.4724933550065136,0.0456487455023837,0.1400240975533177,
                                  0.2768143077913058,0.9715326909010507,0.3313470071030508,
                                  0.4820410581120825},
                                 {0.1960977127383929,0.6107800710757683,0.2806832155159743,
                                  0.2069925784384268,0.5165726284459791,0.0055057991604208,
                                  0.0076640326492253},
                                 {0.2190688091018578,0.0367213625008211,0.1080257541306914,
                                  0.3388606493387516,0.8025856795798129,0.5720486433693008,
                                  0.5126677259564164},
                                 {0.2934888232081216,0.9317537194328164,0.3970150412371917,
                                  0.0870927667957715,0.6170665161178013,0.1138383934766014,
                                  0.3452231428016921}}},
                              
                              
                               {{{0.5074119387178089,0.8742225246851758,0.4935465803401187,
                                  0.7022587660316643,0.9928168389990704,0.1314891198979816,
                                  0.2747315255195195},
                                 {0.3945760755976464,0.4218281785388314,0.4110210752655458,
                                  0.9076110025565549,0.7140326444668564,0.6079052140907037,
                                  0.3093727178627821},
                                 {0.8237903842212947,0.9550605114849889,0.8211982825502265,
                                  0.0015651051914729,0.636401370183407 ,0.0511353814421007,
                                  0.2576072203337219},
                                 {0.0595247003477717,0.6037954529493877,0.6865899134926068,
                                  0.1144876380659051,0.3838388830449795,0.456236508955542 ,
                                  0.3690530217609166},
                                 {0.121025000485193 ,0.4189477901874754,0.7511780478084181,
                                  0.0710076645332555,0.0801800140819035,0.3547712918359842,
                                  0.941725711618925 },
                                 {0.668572646722914 ,0.6786699566725648,0.361919990837865 ,
                                  0.5936607912085384,0.0101311840277434,0.6360960464095422,
                                  0.9132869475762367}},
                              
                                {{0.6125734683448192,0.873698593466345 ,0.7239730554660938,
                                  0.1205584164036902,0.9024533072858417,0.0664438443557749,
                                  0.5339674725477638},
                                 {0.1421371237934577,0.0117066107309041,0.422034260127639 ,
                                  0.2950407076541562,0.486000493797603 ,0.5771998867345508,
                                  0.0437390723089474},
                                 {0.1230032951237456,0.5586423964504253,0.3431673531726582,
                                  0.729173487549448 ,0.6522916485919397,0.8456043629899068,
                                  0.6924928266692253},
                                 {0.4299310773901481,0.6729666223944324,0.2753811742554185,
                                  0.3063134987782127,0.7889851599207933,0.4464195164015822,
                                  0.7983803405183525},
                                 {0.8224223736682784,0.8575623376612838,0.9166352358412073,
                                  0.4309852205132962,0.3188668183211388,0.5821987465941139,
                                  0.3711724234376276},
                                 {0.601074166800233 ,0.7055858626438246,0.6884050184921369,
                                  0.3745528870201214,0.1668602016987528,0.4305293384622075,
                                  0.1425941440819087}},
                              
                                {{0.8900970351838083,0.3458774607378574,0.1544597472233238,
                                  0.0254477599057509,0.6458223862864843,0.6369023174547432,
                                  0.3406041665812917},
                                 {0.071711888847651 ,0.409628894895232 ,0.3112174950633936,
                                  0.6771206397350094,0.6057792494526903,0.36459372819754  ,
                                  0.2178925542549901},
                                 {0.9880359661555919,0.4540016211961149,0.688274235780237 ,
                                  0.1405529675783608,0.4855896996593978,0.0275367198694655,
                                  0.50545511791045  },
                                 {0.9640174071177023,0.3842067129858002,0.0389960125749116,
                                  0.0309557909193435,0.3879867599468695,0.160024936463194 ,
                                  0.0233520963281859},
                                 {0.7562136139501768,0.4585230088818907,0.2892447719726545,
                                  0.9000831830164807,0.1161389602005256,0.9559188289553192,
                                  0.3139667173773528},
                                 {0.8884036723086812,0.6029428764794615,0.8267073597414875,
                                  0.9840131632104643,0.2884054378187457,0.961135485407593 ,
                                  0.3894818949024204}},
                              
                                {{0.3855119047369223,0.3403871042545057,0.5414076073475852,
                                  0.1541450431784871,0.5537114853580112,0.541704812410004 ,
                                  0.76190181260487  },
                                 {0.8336499619375856,0.440441350312424 ,0.3023332493514937,
                                  0.2593348973170945,0.1946238367048719,0.0576475821990249,
                                  0.3424479788347133},
                                 {0.2702476921307833,0.9664173858664074,0.5577088262500439,
                                  0.3472770390305076,0.580467600993696 ,0.1391967013456955,
                                  0.4440437492022029},
                                 {0.6262343214210638,0.4888978317924514,0.4017788006543243,
                                  0.9938483888996982,0.880314762148927 ,0.6234059581594401,
                                  0.5693773611822828},
                                 {0.620660479824584 ,0.2013121078224333,0.3951377432643186,
                                  0.0394595328908993,0.4755460374868267,0.5430245839576632,
                                  0.2277497361032096},
                                 {0.9640288471655137,0.909454610937242 ,0.7221432354559059,
                                  0.5333866287653626,0.869969159647685 ,0.1306523438234629,
                                  0.790509794274885 }},
                              
                                {{0.1248208139668483,0.7942106625494054,0.275767568251357 ,
                                  0.8770905810181889,0.9440420938253978,0.1487959694051156,
                                  0.4626756050653935},
                                 {0.9809873781626851,0.4834074359731466,0.8635475225145451,
                                  0.5887315221313199,0.3753299079196019,0.2857835555258702,
                                  0.2032230910860873},
                                 {0.7617981913776972,0.3865405068230279,0.5112754622007512,
                                  0.4923254800011819,0.5772790292331312,0.8655771456245522,
                                  0.9807393435834294},
                                 {0.4075842085738169,0.8275189221063031,0.7645277946547177,
                                  0.5735289514555733,0.9560471436119331,0.2004745155284028,
                                  0.1092642103096214},
                                 {0.8539619669554221,0.4391498166337483,0.8469688542070605,
                                  0.8930902051954149,0.0624575936190714,0.8834647629860607,
                                  0.4483190553694222},
                                 {0.5104324890042322,0.6265925110932102,0.9263826067300868,
                                  0.019116094429764 ,0.4768434683178171,0.6877220424305607,
                                  0.7227069576982285}}}};

        // py_vr_all
        double py_vr_all = 0.08257307738009954;
        // py_st_all
        double py_st_all = 0.2873553155591515;
        // py_vr
        xarray<double> py_vr = {{0.0852068309126688,0.0847144350000448,0.0665795115515863,
                                 0.1044879757008207,0.0994283910451301,0.0627262352216086,
                                 0.0985537283997266},
                                {0.0638931201038309,0.0764149454368496,0.0713175183081865,
                                 0.066508578076682 ,0.0720721452105284,0.0824901481262121,
                                 0.0641277495703859},
                                {0.1077493245420096,0.0699334071648802,0.1007397644226236,
                                 0.1116186015428858,0.0541115754698437,0.0901322000940208,
                                 0.0664647998988194},
                                {0.087982633168223 ,0.0778706200930632,0.0610545739371242,
                                 0.1024460567243694,0.0536455858542131,0.0504355367736437,
                                 0.0773686386350671},
                                {0.072189495809687 ,0.0720540674148998,0.0922003312160627,
                                 0.0762240153024691,0.0845913536241975,0.0819087439628765,
                                 0.0894375580083343}};
        // py_st
        xarray<double> py_st = {{0.2919020913125989,0.2910574427841431,0.2580300593953858,
                                 0.3232459987390728,0.3153226776575547,0.2504520617236133,
                                 0.3139326813183467},
                                {0.2527708846046769,0.2764325332460881,0.267053399731564 ,
                                 0.2578925708055235,0.2684625583028821,0.287210981903917 ,
                                 0.2532345742002578},
                                {0.3282519223736695,0.2644492525322773,0.3173952810339555,
                                 0.3340937017408228,0.2326189490773349,0.3002202526379938,
                                 0.2578076800617456},
                                {0.2966186662504957,0.2790530775552622,0.2470922377111919,
                                 0.3200719555418272,0.231615167582378 ,0.2245785759453552,
                                 0.2781521861051376},
                                {0.2686810298656885,0.2684288870723489,0.303645074414295 ,
                                 0.2760869705409314,0.2908459276390122,0.2861970369568428,
                                 0.2990611275447452}};

        auto st_all = xt::stddev(py_a);
        auto vr_all = xt::variance(py_a);

        auto st = xt::stddev(py_a, {0, 2});
        auto vr = xt::variance(py_a, {0, 2});

        EXPECT_TRUE(xt::allclose(st_all, py_st_all));
        EXPECT_TRUE(xt::allclose(vr_all, py_vr_all));
        EXPECT_TRUE(xt::allclose(st, py_st));
        EXPECT_TRUE(xt::allclose(vr, py_vr));
    }

}

