// Copyright (c) 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package frontend

import (
	"log"
	"os"
	"testing"
	"time"

	"github.com/uber/cadence/.gen/go/shared"
	"github.com/uber/cadence/common"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"
	"github.com/uber/cadence/common/log/loggerimpl"
	"github.com/uber/cadence/common/mocks"
	dc "github.com/uber/cadence/common/service/dynamicconfig"
)

type (
	domainHandlerSuite struct {
		suite.Suite
		config               *Config
		logger               log.Logger
		mockMetadataMgr      *mocks.MetadataManager
		mockClusterMetadata  *mocks.ClusterMetadata
		mockBlobstoreClient  *mocks.BlobstoreClient
		mockProducer         *mocks.KafkaProducer
		mockDomainReplicator DomainReplicator

		handler *domainHandlerImpl
	}
)

func TestDomainHandlerSuite(t *testing.T) {
	s := new(domainHandlerSuite)
	suite.Run(t, s)
}

func (s *domainHandlerSuite) SetupSuite() {
	if testing.Verbose() {
		log.SetOutput(os.Stdout)
	}
}

func (s *domainHandlerSuite) TearDownSuite() {

}

func (s *domainHandlerSuite) SetupTest() {
	logger := loggerimpl.NewNopLogger()
	s.config = NewConfig(dc.NewCollection(dc.NewNopClient(), logger), numHistoryShards, false, false)
	s.mockMetadataMgr = &mocks.MetadataManager{}
	s.mockClusterMetadata = &mocks.ClusterMetadata{}
	s.mockBlobstoreClient = &mocks.BlobstoreClient{}
	s.mockProducer = &mocks.KafkaProducer{}
	s.mockDomainReplicator = NewDomainReplicator(s.mockProducer, logger)
}

func (s *domainHandlerSuite) TearDownTest() {
	s.mockMetadataMgr.AssertExpectations(s.T())
	s.mockProducer.AssertExpectations(s.T())
	s.mockBlobstoreClient.AssertExpectations(s.T())
	s.mockProducer.AssertExpectations(s.T())
}

func (s *domainHandlerSuite) TestMergeDomainData_Overriding() {
	out := s.handler.mergeDomainData(
		map[string]string{
			"k0": "v0",
		},
		map[string]string{
			"k0": "v2",
		},
	)

	assert.Equal(s.T(), map[string]string{
		"k0": "v2",
	}, out)
}

func (s *domainHandlerSuite) TestMergeDomainData_Adding() {
	out := s.handler.mergeDomainData(
		map[string]string{
			"k0": "v0",
		},
		map[string]string{
			"k1": "v2",
		},
	)

	assert.Equal(s.T(), map[string]string{
		"k0": "v0",
		"k1": "v2",
	}, out)
}

func (s *domainHandlerSuite) TestMergeDomainData_Merging() {
	out := s.handler.mergeDomainData(
		map[string]string{
			"k0": "v0",
		},
		map[string]string{
			"k0": "v1",
			"k1": "v2",
		},
	)

	assert.Equal(s.T(), map[string]string{
		"k0": "v1",
		"k1": "v2",
	}, out)
}

func (s *domainHandlerSuite) TestMergeDomainData_Nil() {
	out := s.handler.mergeDomainData(
		nil,
		map[string]string{
			"k0": "v1",
			"k1": "v2",
		},
	)

	assert.Equal(s.T(), map[string]string{
		"k0": "v1",
		"k1": "v2",
	}, out)
}

// test merging bad binaries

var nowInt64 = time.Now().UnixNano()

func (s *domainHandlerSuite) TestMergeBadBinaries_Overriding() {
	out := s.handler.mergeBadBinaries(
		map[string]*shared.BadBinaryInfo{
			"k0": {Reason: common.StringPtr("reason0")},
		},
		map[string]*shared.BadBinaryInfo{
			"k0": {Reason: common.StringPtr("reason2")},
		}, nowInt64,
	)

	assert.True(s.T(), out.Equals(&shared.BadBinaries{
		Binaries: map[string]*shared.BadBinaryInfo{
			"k0": {Reason: common.StringPtr("reason2"), CreatedTimeNano: common.Int64Ptr(nowInt64)},
		},
	}))
}

func (s *domainHandlerSuite) TestMergeBadBinaries_Adding() {
	out := s.handler.mergeBadBinaries(
		map[string]*shared.BadBinaryInfo{
			"k0": {Reason: common.StringPtr("reason0")},
		},
		map[string]*shared.BadBinaryInfo{
			"k1": {Reason: common.StringPtr("reason2")},
		}, nowInt64,
	)

	expected := &shared.BadBinaries{
		Binaries: map[string]*shared.BadBinaryInfo{
			"k0": {Reason: common.StringPtr("reason0")},
			"k1": {Reason: common.StringPtr("reason2"), CreatedTimeNano: common.Int64Ptr(nowInt64)},
		},
	}
	assert.Equal(s.T(), out.String(), expected.String())
}

func (s *domainHandlerSuite) TestMergeBadBinaries_Merging() {
	out := s.handler.mergeBadBinaries(
		map[string]*shared.BadBinaryInfo{
			"k0": {Reason: common.StringPtr("reason0")},
		},
		map[string]*shared.BadBinaryInfo{
			"k0": {Reason: common.StringPtr("reason1")},
			"k1": {Reason: common.StringPtr("reason2")},
		}, nowInt64,
	)

	assert.True(s.T(), out.Equals(&shared.BadBinaries{
		Binaries: map[string]*shared.BadBinaryInfo{
			"k0": {Reason: common.StringPtr("reason1"), CreatedTimeNano: common.Int64Ptr(nowInt64)},
			"k1": {Reason: common.StringPtr("reason2"), CreatedTimeNano: common.Int64Ptr(nowInt64)},
		},
	}))
}

func (s *domainHandlerSuite) TestMergeBadBinaries_Nil() {
	out := s.handler.mergeBadBinaries(
		nil,
		map[string]*shared.BadBinaryInfo{
			"k0": {Reason: common.StringPtr("reason1")},
			"k1": {Reason: common.StringPtr("reason2")},
		}, nowInt64,
	)

	assert.True(s.T(), out.Equals(&shared.BadBinaries{
		Binaries: map[string]*shared.BadBinaryInfo{
			"k0": {Reason: common.StringPtr("reason1"), CreatedTimeNano: common.Int64Ptr(nowInt64)},
			"k1": {Reason: common.StringPtr("reason2"), CreatedTimeNano: common.Int64Ptr(nowInt64)},
		},
	}))
}
