﻿// Copyright (c) Microsoft.  All Rights Reserved.  Licensed under the Apache License, Version 2.0.  See License.txt in the project root for license information.

using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Diagnostics;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.CodeAnalysis.Collections;
using Microsoft.CodeAnalysis.PooledObjects;
using Microsoft.CodeAnalysis.Text;
using Microsoft.CodeAnalysis.Utilities;
using Roslyn.Utilities;

namespace Microsoft.CodeAnalysis.FindSymbols
{
    internal partial class SymbolTreeInfo : IChecksummedObject
    {
        public Checksum Checksum { get; }

        /// <summary>
        /// To prevent lots of allocations, we concatenate all the names in all our
        /// Nodes into one long string.  Each Node then just points at the span in
        /// this string with the portion they care about.
        /// </summary>
        private readonly string _concatenatedNames;

        /// <summary>
        /// The list of nodes that represent symbols. The primary key into the sorting of this 
        /// list is the name. They are sorted case-insensitively with the <see cref="s_totalComparer" />.
        /// Finding case-sensitive matches can be found by binary searching for something that 
        /// matches insensitively, and then searching around that equivalence class for one that 
        /// matches.
        /// </summary>
        private readonly ImmutableArray<Node> _nodes;

        /// <summary>
        /// Inheritance information for the types in this assembly.  The mapping is between
        /// a type's simple name (like 'IDictionary') and the simple metadata names of types 
        /// that implement it or derive from it (like 'Dictionary').
        /// 
        /// Note: to save space, all names in this map are stored with simple ints.  These
        /// ints are the indices into _nodes that contain the nodes with the appropriate name.
        /// 
        /// This mapping is only produced for metadata assemblies.
        /// </summary>
        private readonly OrderPreservingMultiDictionary<int, int> _inheritanceMap;

        /// <summary>
        /// The task that produces the spell checker we use for fuzzy match queries.
        /// We use a task so that we can generate the <see cref="SymbolTreeInfo"/> 
        /// without having to wait for the spell checker construction to finish.
        /// 
        /// Features that don't need fuzzy matching don't want to incur the cost of 
        /// the creation of this value.  And the only feature which does want fuzzy
        /// matching (add-using) doesn't want to block waiting for the value to be
        /// created.
        /// </summary>
        private readonly Task<SpellChecker> _spellCheckerTask;

        private static readonly StringSliceComparer s_caseInsensitiveComparer =
            StringSliceComparer.OrdinalIgnoreCase;

        // We first sort in a case insensitive manner.  But, within items that match insensitively, 
        // we then sort in a case sensitive manner.  This helps for searching as we'll walk all 
        // the items of a specific casing at once.  This way features can cache values for that
        // casing and reuse them.  i.e. if we didn't do this we might get "Prop, prop, Prop, prop"
        // which might cause other features to continually recalculate if that string matches what
        // they're searching for.  However, with this sort of comparison we now get 
        // "prop, prop, Prop, Prop".  Features can take advantage of that by caching their previous
        // result and reusing it when they see they're getting the same string again.
        private static readonly Comparison<string> s_totalComparer = (s1, s2) =>
        {
            var diff = CaseInsensitiveComparison.Comparer.Compare(s1, s2);
            return diff != 0
                ? diff
                : StringComparer.Ordinal.Compare(s1, s2);
        };

        private SymbolTreeInfo(
            Checksum checksum,
            string concatenatedNames,
            ImmutableArray<Node> sortedNodes,
            Task<SpellChecker> spellCheckerTask,
            OrderPreservingMultiDictionary<string, string> inheritanceMap)
            : this(checksum, concatenatedNames, sortedNodes, spellCheckerTask,
                   CreateIndexBasedInheritanceMap(concatenatedNames, sortedNodes, inheritanceMap))
        {
        }

        private SymbolTreeInfo(
            Checksum checksum,
            string concatenatedNames,
            ImmutableArray<Node> sortedNodes,
            Task<SpellChecker> spellCheckerTask,
            OrderPreservingMultiDictionary<int, int> inheritanceMap)
        {
            Checksum = checksum;
            _concatenatedNames = concatenatedNames;
            _nodes = sortedNodes;
            _spellCheckerTask = spellCheckerTask;
            _inheritanceMap = inheritanceMap;
        }

        public static SymbolTreeInfo CreateEmpty(Checksum checksum)
        {
            var unsortedNodes = ImmutableArray.Create(BuilderNode.RootNode);
            SortNodes(unsortedNodes, out var concatenatedNames, out var sortedNodes);

            return new SymbolTreeInfo(checksum, concatenatedNames, sortedNodes,
                CreateSpellCheckerAsync(checksum, concatenatedNames, sortedNodes),
                new OrderPreservingMultiDictionary<string, string>());
        }

        public SymbolTreeInfo WithChecksum(Checksum checksum)
        {
            return new SymbolTreeInfo(
                checksum, _concatenatedNames, _nodes, _spellCheckerTask, _inheritanceMap);
        }

        public Task<ImmutableArray<SymbolAndProjectId>> FindAsync(
            SearchQuery query, IAssemblySymbol assembly, ProjectId assemblyProjectId, SymbolFilter filter, CancellationToken cancellationToken)
        {
            // All entrypoints to this function are Find functions that are only searching
            // for specific strings (i.e. they never do a custom search).
            Contract.ThrowIfTrue(query.Kind == SearchKind.Custom, "Custom queries are not supported in this API");

            return this.FindAsync(
                query, new AsyncLazy<IAssemblySymbol>(assembly),
                assemblyProjectId, filter, cancellationToken);
        }

        public async Task<ImmutableArray<SymbolAndProjectId>> FindAsync(
            SearchQuery query, AsyncLazy<IAssemblySymbol> lazyAssembly, ProjectId assemblyProjectId,
            SymbolFilter filter, CancellationToken cancellationToken)
        {
            // All entrypoints to this function are Find functions that are only searching
            // for specific strings (i.e. they never do a custom search).
            Contract.ThrowIfTrue(query.Kind == SearchKind.Custom, "Custom queries are not supported in this API");

            var symbols = await FindAsyncWorker(query, lazyAssembly, cancellationToken).ConfigureAwait(false);

            return DeclarationFinder.FilterByCriteria(
                symbols.SelectAsArray(s => new SymbolAndProjectId(s, assemblyProjectId)),
                filter);
        }

        private Task<ImmutableArray<ISymbol>> FindAsyncWorker(
            SearchQuery query, AsyncLazy<IAssemblySymbol> lazyAssembly, CancellationToken cancellationToken)
        {
            // All entrypoints to this function are Find functions that are only searching
            // for specific strings (i.e. they never do a custom search).
            Contract.ThrowIfTrue(query.Kind == SearchKind.Custom, "Custom queries are not supported in this API");

            // If the query has a specific string provided, then call into the SymbolTreeInfo
            // helpers optimized for lookup based on an exact name.
            switch (query.Kind)
            {
                case SearchKind.Exact:
                    return this.FindAsync(lazyAssembly, query.Name, ignoreCase: false, cancellationToken: cancellationToken);
                case SearchKind.ExactIgnoreCase:
                    return this.FindAsync(lazyAssembly, query.Name, ignoreCase: true, cancellationToken: cancellationToken);
                case SearchKind.Fuzzy:
                    return this.FuzzyFindAsync(lazyAssembly, query.Name, cancellationToken);
            }

            throw new InvalidOperationException();
        }

        /// <summary>
        /// Finds symbols in this assembly that match the provided name in a fuzzy manner.
        /// </summary>
        private async Task<ImmutableArray<ISymbol>> FuzzyFindAsync(
            AsyncLazy<IAssemblySymbol> lazyAssembly, string name, CancellationToken cancellationToken)
        {
            if (_spellCheckerTask.Status != TaskStatus.RanToCompletion)
            {
                // Spell checker isn't ready.  Just return immediately.
                return ImmutableArray<ISymbol>.Empty;
            }

            var spellChecker = await _spellCheckerTask.ConfigureAwait(false);
            var similarNames = spellChecker.FindSimilarWords(name, substringsAreSimilar: false);
            var result = ArrayBuilder<ISymbol>.GetInstance();

            foreach (var similarName in similarNames)
            {
                var symbols = await FindAsync(lazyAssembly, similarName, ignoreCase: true, cancellationToken: cancellationToken).ConfigureAwait(false);
                result.AddRange(symbols);
            }

            return result.ToImmutableAndFree();
        }

        /// <summary>
        /// Get all symbols that have a name matching the specified name.
        /// </summary>
        private async Task<ImmutableArray<ISymbol>> FindAsync(
            AsyncLazy<IAssemblySymbol> lazyAssembly,
            string name,
            bool ignoreCase,
            CancellationToken cancellationToken)
        {
            var comparer = GetComparer(ignoreCase);
            var results = ArrayBuilder<ISymbol>.GetInstance();
            IAssemblySymbol assemblySymbol = null;

            foreach (var node in FindNodeIndices(name, comparer))
            {
                cancellationToken.ThrowIfCancellationRequested();
                assemblySymbol = assemblySymbol ?? await lazyAssembly.GetValueAsync(cancellationToken).ConfigureAwait(false);

                Bind(node, assemblySymbol.GlobalNamespace, results, cancellationToken);
            }

            return results.ToImmutableAndFree();
        }

        private static StringSliceComparer GetComparer(bool ignoreCase)
        {
            return ignoreCase
                ? StringSliceComparer.OrdinalIgnoreCase
                : StringSliceComparer.Ordinal;
        }

        private IEnumerable<int> FindNodeIndices(string name, StringSliceComparer comparer)
            => FindNodeIndices(_concatenatedNames, _nodes, name, comparer);

        /// <summary>
        /// Gets all the node indices with matching names per the <paramref name="comparer" />.
        /// </summary>
        private static IEnumerable<int> FindNodeIndices(
            string concatenatedNames, ImmutableArray<Node> nodes,
            string name, StringSliceComparer comparer)
        {
            // find any node that matches case-insensitively
            var startingPosition = BinarySearch(concatenatedNames, nodes, name);
            var nameSlice = new StringSlice(name);

            if (startingPosition != -1)
            {
                // yield if this matches by the actual given comparer
                if (comparer.Equals(nameSlice, GetNameSlice(concatenatedNames, nodes, startingPosition)))
                {
                    yield return startingPosition;
                }

                int position = startingPosition;
                while (position > 0 && s_caseInsensitiveComparer.Equals(GetNameSlice(concatenatedNames, nodes, position - 1), nameSlice))
                {
                    position--;
                    if (comparer.Equals(GetNameSlice(concatenatedNames, nodes, position), nameSlice))
                    {
                        yield return position;
                    }
                }

                position = startingPosition;
                while (position + 1 < nodes.Length && s_caseInsensitiveComparer.Equals(GetNameSlice(concatenatedNames, nodes, position + 1), nameSlice))
                {
                    position++;
                    if (comparer.Equals(GetNameSlice(concatenatedNames, nodes, position), nameSlice))
                    {
                        yield return position;
                    }
                }
            }
        }

        private StringSlice GetNameSlice(int nodeIndex)
            => GetNameSlice(_concatenatedNames, _nodes, nodeIndex);

        private static StringSlice GetNameSlice(
            string concatenatedNames, ImmutableArray<Node> nodes, int nodeIndex)
        {
            return new StringSlice(concatenatedNames, nodes[nodeIndex].NameSpan);
        }

        private int BinarySearch(string name)
            => BinarySearch(_concatenatedNames, _nodes, name);

        /// <summary>
        /// Searches for a name in the ordered list that matches per the <see cref="s_caseInsensitiveComparer" />.
        /// </summary>
        private static int BinarySearch(string concatenatedNames, ImmutableArray<Node> nodes, string name)
        {
            var nameSlice = new StringSlice(name);
            int max = nodes.Length - 1;
            int min = 0;

            while (max >= min)
            {
                int mid = min + ((max - min) >> 1);

                var comparison = s_caseInsensitiveComparer.Compare(
                    GetNameSlice(concatenatedNames, nodes, mid), nameSlice);
                if (comparison < 0)
                {
                    min = mid + 1;
                }
                else if (comparison > 0)
                {
                    max = mid - 1;
                }
                else
                {
                    return mid;
                }
            }

            return -1;
        }

        #region Construction

        // Cache the symbol tree infos for assembly symbols that share the same underlying metadata.
        // Generating symbol trees for metadata can be expensive (in large metadata cases).  And it's
        // common for us to have many threads to want to search the same metadata simultaneously.
        // As such, we want to only allow one thread to produce the tree for some piece of metadata
        // at a time.  
        //
        // AsyncLazy would normally be an ok choice here.  However, in the case where all clients
        // cancel their request, we don't want ot keep the AsyncLazy around.  It may capture a lot
        // of immutable state (like a Solution) that we don't want kept around indefinitely.  So we
        // only cache results (the symbol tree infos) if they successfully compute to completion.
        private static readonly ConditionalWeakTable<MetadataId, SemaphoreSlim> s_metadataIdToGate = new ConditionalWeakTable<MetadataId, SemaphoreSlim>();
        private static readonly ConditionalWeakTable<MetadataId, Task<SymbolTreeInfo>> s_metadataIdToInfo =
            new ConditionalWeakTable<MetadataId, Task<SymbolTreeInfo>>();

        private static readonly ConditionalWeakTable<MetadataId, SemaphoreSlim>.CreateValueCallback s_metadataIdToGateCallback =
            _ => new SemaphoreSlim(1);

        private static Task<SpellChecker> GetSpellCheckerTask(
            Solution solution, Checksum checksum, string filePath,
            string concatenatedNames, ImmutableArray<Node> sortedNodes)
        {
            // Create a new task to attempt to load or create the spell checker for this 
            // SymbolTreeInfo.  This way the SymbolTreeInfo will be ready immediately
            // for non-fuzzy searches, and soon afterwards it will be able to perform
            // fuzzy searches as well.
            return Task.Run(() => LoadOrCreateSpellCheckerAsync(
                solution, checksum, filePath, concatenatedNames, sortedNodes));
        }

        private static Task<SpellChecker> CreateSpellCheckerAsync(
            Checksum checksum, string concatenatedNames, ImmutableArray<Node> sortedNodes)
        {
            return Task.FromResult(new SpellChecker(
                checksum, sortedNodes.Select(n => new StringSlice(concatenatedNames, n.NameSpan))));
        }

        private static void SortNodes(
            ImmutableArray<BuilderNode> unsortedNodes,
            out string concatenatedNames,
            out ImmutableArray<Node> sortedNodes)
        {
            // Generate index numbers from 0 to Count-1
            var tmp = new int[unsortedNodes.Length];
            for (int i = 0; i < tmp.Length; i++)
            {
                tmp[i] = i;
            }

            // Sort the index according to node elements
            Array.Sort<int>(tmp, (a, b) => CompareNodes(unsortedNodes[a], unsortedNodes[b], unsortedNodes));

            // Use the sort order to build the ranking table which will
            // be used as the map from original (unsorted) location to the
            // sorted location.
            var ranking = new int[unsortedNodes.Length];
            for (int i = 0; i < tmp.Length; i++)
            {
                ranking[tmp[i]] = i;
            }

            // No longer need the tmp array
            tmp = null;

            var result = ArrayBuilder<Node>.GetInstance(unsortedNodes.Length);
            result.Count = unsortedNodes.Length;

            var concatenatedNamesBuilder = new StringBuilder();
            string lastName = null;

            // Copy nodes into the result array in the appropriate order and fixing
            // up parent indexes as we go.
            for (int i = 0; i < unsortedNodes.Length; i++)
            {
                var n = unsortedNodes[i];
                var currentName = n.Name;

                // Don't bother adding the exact same name the concatenated sequence
                // over and over again.  This can trivially happen because we'll run
                // into the same names with different parents all through metadata 
                // and source symbols.
                if (currentName != lastName)
                {
                    concatenatedNamesBuilder.Append(currentName);
                }

                result[ranking[i]] = new Node(
                    new TextSpan(concatenatedNamesBuilder.Length - currentName.Length, currentName.Length),
                    n.IsRoot ? n.ParentIndex : ranking[n.ParentIndex]);

                lastName = currentName;
            }

            sortedNodes = result.ToImmutableAndFree();
            concatenatedNames = concatenatedNamesBuilder.ToString();
        }

        private static int CompareNodes(
            BuilderNode x, BuilderNode y, ImmutableArray<BuilderNode> nodeList)
        {
            var comp = s_totalComparer(x.Name, y.Name);
            if (comp == 0)
            {
                if (x.ParentIndex != y.ParentIndex)
                {
                    if (x.IsRoot)
                    {
                        return -1;
                    }
                    else if (y.IsRoot)
                    {
                        return 1;
                    }
                    else
                    {
                        return CompareNodes(nodeList[x.ParentIndex], nodeList[y.ParentIndex], nodeList);
                    }
                }
            }

            return comp;
        }

        #endregion

        #region Binding 

        // returns all the symbols in the container corresponding to the node
        private void Bind(
            int index, INamespaceOrTypeSymbol rootContainer, ArrayBuilder<ISymbol> results, CancellationToken cancellationToken)
        {
            cancellationToken.ThrowIfCancellationRequested();

            var node = _nodes[index];
            if (node.IsRoot)
            {
                return;
            }

            if (_nodes[node.ParentIndex].IsRoot)
            {
                results.AddRange(rootContainer.GetMembers(GetName(node)));
            }
            else
            {
                var containerSymbols = ArrayBuilder<ISymbol>.GetInstance();
                try
                {
                    Bind(node.ParentIndex, rootContainer, containerSymbols, cancellationToken);

                    foreach (var containerSymbol in containerSymbols)
                    {
                        cancellationToken.ThrowIfCancellationRequested();

                        if (containerSymbol is INamespaceOrTypeSymbol nsOrType)
                        {
                            results.AddRange(nsOrType.GetMembers(GetName(node)));
                        }
                    }
                }
                finally
                {
                    containerSymbols.Free();
                }
            }
        }

        private string GetName(Node node)
        {
            // TODO(cyrusn): We could consider caching the strings we create in the
            // Nodes themselves.  i.e. we could have a field in the node where the
            // string could be stored once created.  The reason i'm not doing that now
            // is because, in general, we shouldn't actually be allocating that many
            // strings here.  This data structure is not in a hot path, and does not
            // have a usage pattern where may strings are accessed in it.  Rather, 
            // some features generally use it to just see if they can find a symbol
            // corresponding to a single name.  As such, caching doesn't seem valuable.
            return _concatenatedNames.Substring(node.NameSpan.Start, node.NameSpan.Length);
        }

        #endregion

        internal void AssertEquivalentTo(SymbolTreeInfo other)
        {
            Debug.Assert(Checksum.Equals(other.Checksum));
            Debug.Assert(_concatenatedNames == other._concatenatedNames);
            Debug.Assert(_nodes.Length == other._nodes.Length);

            for (int i = 0, n = _nodes.Length; i < n; i++)
            {
                _nodes[i].AssertEquivalentTo(other._nodes[i]);
            }

            Debug.Assert(_inheritanceMap.Keys.Count == other._inheritanceMap.Keys.Count);
            var orderedKeys1 = this._inheritanceMap.Keys.Order().ToList();
            var orderedKeys2 = other._inheritanceMap.Keys.Order().ToList();

            for (int i = 0; i < orderedKeys1.Count; i++)
            {
                var values1 = this._inheritanceMap[i];
                var values2 = other._inheritanceMap[i];

                Debug.Assert(values1.Length == values2.Length);
                for (int j = 0; j < values1.Length; j++)
                {
                    Debug.Assert(values1[j] == values2[j]);
                }
            }
        }

        private static SymbolTreeInfo CreateSymbolTreeInfo(
            Solution solution, Checksum checksum,
            string filePath, ImmutableArray<BuilderNode> unsortedNodes,
            OrderPreservingMultiDictionary<string, string> inheritanceMap)
        {
            SortNodes(unsortedNodes, out var concatenatedNames, out var sortedNodes);
            var createSpellCheckerTask = GetSpellCheckerTask(
                solution, checksum, filePath, concatenatedNames, sortedNodes);

            return new SymbolTreeInfo(
                checksum, concatenatedNames,
                sortedNodes, createSpellCheckerTask, inheritanceMap);
        }

        private static OrderPreservingMultiDictionary<int, int> CreateIndexBasedInheritanceMap(
            string concatenatedNames, ImmutableArray<Node> nodes,
            OrderPreservingMultiDictionary<string, string> inheritanceMap)
        {
            // All names in metadata will be case sensitive.  
            var comparer = GetComparer(ignoreCase: false);
            var result = new OrderPreservingMultiDictionary<int, int>();

            foreach (var kvp in inheritanceMap)
            {
                var baseName = kvp.Key;
                var baseNameIndex = BinarySearch(concatenatedNames, nodes, baseName);
                Debug.Assert(baseNameIndex >= 0);

                foreach (var derivedName in kvp.Value)
                {
                    foreach (var derivedNameIndex in FindNodeIndices(concatenatedNames, nodes, derivedName, comparer))
                    {
                        result.Add(baseNameIndex, derivedNameIndex);
                    }
                }
            }

            return result;
        }

        public ImmutableArray<INamedTypeSymbol> GetDerivedMetadataTypes(
            string baseTypeName, Compilation compilation, CancellationToken cancellationToken)
        {
            var baseTypeNameIndex = BinarySearch(baseTypeName);
            var derivedTypeIndices = _inheritanceMap[baseTypeNameIndex];

            var builder = ArrayBuilder<INamedTypeSymbol>.GetInstance();

            foreach (var derivedTypeIndex in derivedTypeIndices)
            {
                var tempBuilder = ArrayBuilder<ISymbol>.GetInstance();
                try
                {
                    Bind(derivedTypeIndex, compilation.GlobalNamespace, tempBuilder, cancellationToken);
                    foreach (var symbol in tempBuilder)
                    {
                        if (symbol is INamedTypeSymbol namedType)
                        {
                            builder.Add(namedType);
                        }
                    }
                }
                finally
                {
                    tempBuilder.Free();
                }
            }

            return builder.ToImmutableAndFree();
        }
    }
}
