/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at
 * https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2007 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.faces.mirror;

import java.util.regex.Pattern;

/**
 * A base class that defines the basic metadata available for any component
 * feature, whether a property or event.
 *
 * @author gjmurphy
 */
public abstract class FeatureInfo {
    
    /**
     * Pattern used in property info analysis to determine if a name is a Java
     * keyword or reserved word.
     */
    static Pattern JAVA_KEYWORD_PATTERN = Pattern.compile(
            "(" +
            "assert|boolean|break|byte|case|catch|char|class|const|continue|default|do|double|else|" +
            "enum|extends|false|final|finally|float|for|goto|if|implements|import|instanceof|int|" +
            "interface|long|native|new|null|package|private|protected|public|return|short|static|" +
            "super|switch|synchronized|this|throw|throws|transient|true|try|void|volatile|while" +
            ")"
    );
    
    /**
     * Returns the name of this feature, which is guaranteed to be unique within
     * the scope of the containing component.
     */
    public abstract String getName();
    
    /**
     * Returns the display name of this property. If no display name was assigned
     * to this feature, should return the feature name.
     */
    public abstract String getDisplayName();
    
    /**
     * Returns a short description of this property. If no short description was
     * assigned to this feature, should return the feature's display name. 
     */
    public abstract String getShortDescription();
    
    /**
     * Returns true if this feature should be hidden at design-time.
     */
    public abstract boolean isHidden();

    private ClassInfo declaringClassInfo;
    
    /**
     * Returns the class info of the class that declared this feature. This is 
     * useful when a property or event is retrieved from a class that reports its 
     * inherited properties and events. Returns null if the super class is 
     * {@code java.lang.Object}.
     */
    public ClassInfo getDeclaringClassInfo() {
        return this.declaringClassInfo;
    }
    
    void setDeclaringClassInfo(ClassInfo declaringClassInfo) {
        this.declaringClassInfo = declaringClassInfo;
    }
    
    /**
     * Returns a unique, generated key for the property specified, suitable for
     * use as a key in a properties resource bundle file, if the property corresponds
     * to a localizable annotation element. For example, the property {@link PropertyInfo#getDisplayName})
     * corresponds to the localizable annotation element {@link com.sun.faces.Property#displayName}
     * so a unique key for it may be generated by calling
     * <pre>
     *    String key = propertyInfo.getKey("displayName");
     * </pre>
     * If the specified property does not correspond to a localizable annotation, 
     * returns null. If the specified property does not exist, throws {@link
     * java.lang.NoSuchMethodException}. Only properties of type {@link java.lang.String}
     * are supported.
     */
    public final String getKey(String propertyName) throws NoSuchMethodException {
        String methodName = "get" + propertyName.substring(0, 1).toUpperCase() + propertyName.substring(1);
        this.getClass().getMethod(methodName);
        String baseName = this.getDeclaringClassInfo().getClassName();
        return baseName + "_" + this.getName() + "_" + propertyName;
    }

}
